﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using Reactive.Bindings.Notifiers;
using System;
using System.Linq;
using System.Reactive.Linq;
using System.Threading.Tasks;
using System.Collections.Generic;
using System.Windows.Controls;

namespace SpatialAudioEvaluationTool
{
    public class MainWindowViewModel : IDisposable
    {
        public ReactiveProperty<RecordingState> State { get; } = new ReactiveProperty<RecordingState>(RecordingState.NoConnection);

        public ReactiveProperty<double[]> FlPcmSamples { get; }
        public ReactiveProperty<double[]> FrPcmSamples { get; }
        public ReactiveProperty<double[]> FcPcmSamples { get; }
        public ReactiveProperty<double[]> LfePcmSamples { get; }
        public ReactiveProperty<double[]> RlPcmSamples { get; }
        public ReactiveProperty<double[]> RrPcmSamples { get; }
        public ReactiveProperty<MenuItem[]> LinkMenuItems { get; } = new ReactiveProperty<MenuItem[]>();

        public ReactiveProperty<string> TargetName { get; } = new ReactiveProperty<string>();

        public ReactiveCommand ConnectCommand { get; }
        public ReactiveCommand DisconnectCommand { get; }
        public ReactiveCommand LinkClickCommand { get; }

        private BusyNotifier Busy { get; } = new BusyNotifier();

        public Action<object, Exception> ConnectionError { get; set; }
        public Action<object, Exception> MonitoringError { get; set; }

        private readonly int WaveformSampleCount = 4096;
        private WaveformIndicator _waveformIndicator;

        private WaveReceiver _receiver;
        private WavePlayback _playback;

        public MainWindowViewModel()
        {
#if (DEBUG_CULTURE_EN)
            Resources.ExceptionMessages.Culture = System.Globalization.CultureInfo.GetCultureInfo("en");
            Resources.MainWindowResources.Culture = System.Globalization.CultureInfo.GetCultureInfo("en");
            Resources.RecordingStateMessages.Culture = System.Globalization.CultureInfo.GetCultureInfo("en");
#endif

            ConnectCommand = new[]
            {
                State.Select(x => x == RecordingState.NoConnection),
                Busy.Inverse().ToReactiveProperty()
            }.CombineLatestValuesAreAllTrue().ToReactiveCommand();
            ConnectCommand.Subscribe(async _ => await Connect());

            DisconnectCommand = new[]
            {
                State.Select(x => x == RecordingState.Connecting || x == RecordingState.Connected),
                Busy.Inverse().ToReactiveProperty()
            }.CombineLatestValuesAreAllTrue().ToReactiveCommand();
            DisconnectCommand.Subscribe(async _ =>
            {
                using (Busy.ProcessStart())
                {
                    await Disconnect();
                }
            });

            LinkClickCommand = new ReactiveCommand();
            LinkClickCommand.Subscribe(_ => ShowTargetListMenu());

            FlPcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.FrontLeft)).ToReactiveProperty();
            FrPcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.FrontRight)).ToReactiveProperty();
            FcPcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.FrontCenter)).ToReactiveProperty();
            LfePcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.LowFrequency)).ToReactiveProperty();
            RlPcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.RearLeft)).ToReactiveProperty();
            RrPcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.RearRight)).ToReactiveProperty();
        }

        double[] GetWaveform(int samples, WaveformIndicator.ChannelIndex_5_1 channelIndex)
        {
            if(_waveformIndicator == null)
            {
                return new double[samples];
            }
            return _waveformIndicator.GetWaveform(channelIndex);
        }

        private void ShowTargetListMenu()
        {
            var names = new List<MenuItem>();
            var defaultItem = new MenuItem();
            defaultItem.Header = Resources.MainWindowResources.DefaultTargetString;
            defaultItem.Click += TargetListMenuClicked;
            names.Add(defaultItem);
            try
            {
                var targets = Nintendo.Tm.TargetManager.GetTargets();
                foreach (var target in targets)
                {
                    var item = new MenuItem();
                    item.Header = target.GetName();
                    item.Click += TargetListMenuClicked;
                    names.Add(item);
                }
            }
            catch
            {
            }
            LinkMenuItems.Value = names.ToArray();
        }

        private async void TargetListMenuClicked(object sender, EventArgs e)
        {
            var name = ((MenuItem)sender).Header.ToString();
            if (name == Resources.MainWindowResources.DefaultTargetString)
            {
                await Connect();
            }
            else
            {
                await Connect(name);
            }
        }

        private void Shutdown()
        {
            if (_receiver != null)
            {
                Task.Run(async () => await Disconnect()).Wait();
            }
        }

        private void OnCaptureStopped(object sender, Exception e)
        {
            if(e != null)
            {
                MonitoringError?.Invoke(sender, e);
                Shutdown();
            }
        }

        private async Task<bool> Connect(string name = null)
        {
            // TODO: 接続時のエラーメッセージを受け取る
            State.Value = RecordingState.Connecting;

            _receiver = new WaveReceiver(name);

            try
            {
                Nintendo.Tm.Target connectTarget = null;
                if (name == null)
                {
                    var result = await Task.Run(() => Nintendo.Tm.TargetManager.TryGetDefaultTarget(out connectTarget));
                    if (!result)
                    {
                        State.Value = RecordingState.NoConnection;
                        throw new Exception(Resources.ExceptionMessages.DefaultTargetNotFoundException);
                    }
                }
                else
                {
                    var targets = await Task.Run(() => Nintendo.Tm.TargetManager.GetTargets());
                    connectTarget = targets.Where(x => x.GetName() == name).FirstOrDefault();
                    if (connectTarget == null)
                    {
                        State.Value = RecordingState.NoConnection;
                        throw new Exception(Resources.ExceptionMessages.DefaultTargetNotFoundException);
                    }
                }
                await Task.Run(() => { connectTarget.Connect(); });

                TargetName.Value = connectTarget.GetName();  // TODO: 状態遷移に関する仕様変更に弱いので、要リファクタリング
            }
            catch(Exception e)
            {
                Console.WriteLine("{0}", e.Message);
                State.Value = RecordingState.NoConnection;
                ConnectionError?.Invoke(this, e);
                return false;
            }

            try
            {
                _waveformIndicator = new WaveformIndicator(_receiver.Format, WaveformSampleCount);
            }
            catch (Exception e)
            {
                Console.WriteLine("{0}", e.Message);
                Shutdown();
                ConnectionError?.Invoke(this, e);
                return false;
            }
            _receiver.AddWaveProcessor(_waveformIndicator);

            try
            {
                _playback = new WavePlayback(_receiver.Format);
            }
            catch (Exception e)
            {
                Console.WriteLine("{0}", e.Message);
                Shutdown();
                ConnectionError?.Invoke(this, e);
                return false;
            }
            _receiver.AddWaveProcessor(_playback);

            _receiver.CaptureStopped += OnCaptureStopped;

            try
            {
                await _receiver.Start(name);
                State.Value = RecordingState.Connected;
            }
            catch (Exception e)
            {
                Console.WriteLine("{0}", e.Message);
                Shutdown();
                ConnectionError?.Invoke(this, e);
                return false;
            }

            return true;
        }

        private async Task<bool> Disconnect()
        {
            var result = await Task.Run(() =>
            {
                _receiver?.RemoveWaveProcessor(_waveformIndicator);
                _waveformIndicator = null;

                _receiver?.RemoveWaveProcessor(_playback);
                _playback?.Dispose();
                _playback = null;

                _receiver?.Stop();
                _receiver = null;

                State.Value = RecordingState.NoConnection;
                return true;
            });

            TargetName.Value = string.Empty;  // TODO: 状態遷移に関する仕様変更に弱いので、要リファクタリング

            return result;
        }

        public void Dispose()
        {
            Shutdown();
        }
    }
}
