﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Runtime.ExceptionServices;
using System.Threading.Tasks;

namespace Nintendo.AudioTool
{
    /// <summary>
    ///  ターゲットのシステムミキサー出力相当のオーディオデータをキャプチャするクラスです。
    /// </summary>
    public class SystemAudioCapture : IDisposable
    {
        /// <summary>
        ///  キャプチャ用のサービスポートが見つからなかったことを示す例外です。
        /// </summary>
        public class ServicePortNotFoundException : Exception
        {
            /// <summary>
            ///  コンストラクタです。
            /// </summary>
            public ServicePortNotFoundException()
                : base(AudioTool.Resources.ExceptionMessages.ServicePortNotFoundExceptionMessage)
            {
            }

            /// <summary>
            ///  コンストラクタです。
            /// </summary>
            /// <param name="message">例外のメッセージです。</param>
            public ServicePortNotFoundException(string message)
                : base(message)
            {
            }

            /// <summary>
            ///  コンストラクタです。
            /// </summary>
            /// <param name="message">例外のメッセージです。</param>
            /// <param name="inner">InnerException として扱う例外です。</param>
            public ServicePortNotFoundException(string message, Exception inner)
                : base(message, inner)
            {
            }
        }

        /// <summary>
        ///  指定したターゲットが見つからなかったことを示す例外です。
        /// </summary>
        public class TargetNotFoundException : Exception
        {
            /// <summary>
            ///  コンストラクタです。
            /// </summary>
            public TargetNotFoundException()
                : base(AudioTool.Resources.ExceptionMessages.TargetNotFoundExceptionMessage)
            {
            }

            /// <summary>
            ///  コンストラクタです。
            /// </summary>
            /// <param name="message">例外のメッセージです。</param>
            public TargetNotFoundException(string message)
                : base(message)
            {
            }

            /// <summary>
            ///  コンストラクタです。
            /// </summary>
            /// <param name="message">例外のメッセージです。</param>
            /// <param name="inner">InnerException として扱う例外です。</param>
            public TargetNotFoundException(string message, Exception inner)
                : base(message, inner)
            {
            }
        }

        /// <summary>
        ///  波形のフォーマットです。
        /// </summary>
        public class WaveFormat
        {
            /// <summary>
            ///  1 サンプル当たりのビット数です。
            /// </summary>
            public int BitDepth { get; set; }
            /// <summary>
            ///  サンプルレートです。
            /// </summary>
            public int SampleRate { get; set; }
            /// <summary>
            ///  チャンネル数です。
            /// </summary>
            public int ChannelCount { get; set; }
        }

        /// <summary>
        ///  キャプチャする波形のフォーマットです。
        /// </summary>
        public WaveFormat Format { get; } = new WaveFormat();

#if DEBUG
        public int AdspLoad
        {
            get { return _impl.AdspLoad; }
        }
#endif  // DEBUG

        /// <summary>
        ///  キャプチャが行われた際に呼ばれるイベントハンドラです。
        /// </summary>
        public event EventHandler<SystemAudioCaptureEventArgs> DataAvailable;

        /// <summary>
        ///  キャプチャが終了した際に呼ばれるイベントハンドラです。
        /// </summary>
        public event EventHandler<SystemAudioCaptureStoppedEventArgs> CaptureStopped;

        private string _targetName;
        private Detail.SystemAudioCaptureImpl _impl = new Detail.SystemAudioCaptureImpl();
        private Detail.TargetConnector _connector = new Detail.TargetConnector();

        /// <summary>
        ///  コンストラクタです。デフォルトのターゲットに対してキャプチャを行います。
        /// </summary>
        public SystemAudioCapture()
        {
            Initialize();
            _targetName = string.Empty;
        }

        /// <summary>
        ///  コンストラクタです。指定したターゲットに対してキャプチャを行います。
        /// </summary>
        /// <param name="targetName"> キャプチャするターゲットです。 </param>
        public SystemAudioCapture(string targetName)
        {
            Initialize();
            _targetName = targetName;
        }

        private void Initialize()
        {
#if (DEBUG_CULTURE_EN)
            Resources.ExceptionMessages.Culture = System.Globalization.CultureInfo.GetCultureInfo("en");
#endif
            Format.BitDepth = 16;
            Format.ChannelCount = 6;
            Format.SampleRate = 48000;
        }

        /// <summary>
        ///  キャプチャを開始します。
        /// </summary>
        /// <exception cref="Nintendo.Tm.TmException">TargetManager との接続時にエラーが発生しました。</exception>
        /// <exception cref="ServicePortNotFoundException">キャプチャ用のサービスポートが見つかりませんでした。</exception>
        /// <exception cref="TargetNotFoundException">指定したターゲットが見つかりませんでした。</exception>
        /// <seealso cref="System.Net.Sockets.Socket.Connect(System.Net.EndPoint)"/>
        public void Start()
        {
            var startTask = StartAsync();
            try
            {
                startTask.Wait();
            }
            catch(Exception e)
            {
                ExceptionDispatchInfo.Capture(e.InnerException).Throw();
            }
        }

        /// <summary>
        ///  キャプチャを非同期に開始します。
        /// </summary>
        /// <returns>キャプチャ処理のタスクです。</returns>
        /// <exception cref="Nintendo.Tm.TmException">TargetManager との接続時にエラーが発生しました。</exception>
        /// <exception cref="ServicePortNotFoundException">キャプチャ用のサービスポートが見つかりませんでした。</exception>
        /// <exception cref="TargetNotFoundException">指定したターゲットが見つかりませんでした。</exception>
        /// <seealso cref="System.Net.Sockets.Socket.Connect(System.Net.EndPoint)"/>
        public async Task StartAsync()
        {
            await _connector.ConnectAsync(_targetName);
            _impl.Start(Format, _connector.Stream, DataAvailable, CaptureStopped + OnCaptureStopped);
        }

        private void OnCaptureStopped(object sender, SystemAudioCaptureStoppedEventArgs e)
        {
            if (e.Exception != null)
            {
                _connector.Disconnect();
            }
        }

        /// <summary>
        ///  キャプチャを終了します。
        /// </summary>
        public void Stop()
        {
            _impl.Stop();
            _connector.Disconnect();
        }

        /// <summary>
        ///  クラスを破棄します。
        /// </summary>
        public void Dispose()
        {
            Stop();
        }
    }
}
