﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <windows.h>
#include <stdio.h>
#include "Types.h"
#include "soundfile.h"
#include "endian.h"
#include "aifffile.h"
#include "Wavfile.h"


#define SOUND_FILE_WAVE 0
#define SOUND_FILE_AIFF 1


/*---------------------------------------------------------------------------*
  Name:         getAiffInfo

  Description:  get AIFF file information.

  Arguments:    path          path of an AIFF file.
                soundinfo     pointer of a SOUNDINFO structure.
                buffer        pointer of a samples buffer.

  Returns:      SoundFileResult_Success       OK.
                SoundFileResult_OperationFailed   can't open an AIFF file.
 *---------------------------------------------------------------------------*/

int getAiffInfo(char *path, SOUNDINFO *soundinfo, void *buffer)
{
    FILE        *file;
    AIFFINFO    aiffinfo;
    errno_t     err;

    if (( err = fopen_s( &file, path, "rb" ) ) != 0)
    {
        return SoundFileResult_OperationFailed;
    }

    aiffReadHeader(&aiffinfo, file);

    soundinfo->channels      = aiffGetChannels(&aiffinfo);
    soundinfo->bitsPerSample = aiffGetBitsPerSample(&aiffinfo);
    soundinfo->sampleRate    = aiffGetSampleRate(&aiffinfo);
    soundinfo->samples       = aiffGetSamples(&aiffinfo);
    soundinfo->loopStart     = aiffGetLoopStart(&aiffinfo);
    soundinfo->loopEnd       = aiffGetLoopEnd(&aiffinfo);
    soundinfo->bufferLength  = 0;

    switch (soundinfo->bitsPerSample)
    {
        case 8:

            soundinfo->bufferLength = soundinfo->samples * soundinfo->channels;

            break;

        case 16:

            soundinfo->bufferLength = soundinfo->samples * soundinfo->channels * 2;

            break;
    }

    if (buffer)
    {
        fread(buffer, soundinfo->bufferLength, 1, file);

        if (soundinfo->bitsPerSample == 16)
        {
            uint16_t* p = (uint16_t*)(buffer);
            int i;
            for (i = 0; i < soundinfo->bufferLength / 2; ++i)
            {
                p[i] = reverse_endian_16(p[i]);
            }
        }
    }

    fclose(file);

    return SoundFileResult_Success;
}


/*---------------------------------------------------------------------------*
  Name:         getWaveInfo

  Description:  get Wave file information.

  Arguments:    path          path of a Wave file.
                soundinfo     pointer of a SOUNDINFO structure.
                buffer        pointer of a samples buffer.

  Returns:      SoundFileResult_Success       OK.
                SoundFileResult_OperationFailed   can't open a Wave file.
 *---------------------------------------------------------------------------*/

int getWaveInfo(char *path, SOUNDINFO *soundinfo, void *buffer)
{
    FILE        *file;
    WAVECHUNK   wavechunk;
    LOOPINFO    loopinfo;
    errno_t     err;

    if (( err = fopen_s( &file, path, "rb" ) ) != 0)
    {
        return SoundFileResult_OperationFailed;
    }

    if (wavReadHeader(&wavechunk, &loopinfo, file))
    {
        soundinfo->channels      = wavGetChannels(&wavechunk);
        soundinfo->bitsPerSample = wavGetBitsPerSample(&wavechunk);
        soundinfo->sampleRate    = wavGetSampleRate(&wavechunk);
        soundinfo->samples       = wavGetSamples(&wavechunk);
        soundinfo->loopStart     = loopinfo.start;
        soundinfo->loopEnd       = loopinfo.end;
        soundinfo->bufferLength  = 0;
    }
    else
    {
        return SoundFileResult_FileCorrupted;
    }


    switch (soundinfo->bitsPerSample)
    {
        case 8:

            soundinfo->bufferLength = soundinfo->samples * soundinfo->channels;

            break;

        case 16:

            soundinfo->bufferLength = soundinfo->samples * soundinfo->channels * 2;

            break;
    }

    if (buffer)
    {
        fread(buffer, soundinfo->bufferLength, 1, file);

        if (soundinfo->bitsPerSample == 8)
        {
            int i;
            char *p;

            p = buffer;

            for (i = 0; i < soundinfo->bufferLength; i++)
            {
                *p = *p + 0x80;
                p++;
            }
        }
    }

    fclose(file);

    return SoundFileResult_Success;
}


/*---------------------------------------------------------------------------*
  Name:         getFileType

  Description:  check file type (AIFF or Wave) of an input file.

  Arguments:    path          path of an input file.
                soundinfo     pointer of a SOUNDINFO structure (not used).

  Returns:      SoundFileResult_InvalidFormat  neither AIFF nor Wave.
                SOUND_FILE_AIFF          AIFF.
                SOUND_FILE_WAVE          Wave.
                SoundFileResult_OperationFailed   can't open an input file.
 *---------------------------------------------------------------------------*/

int getFileType(char *path, SOUNDINFO *soundinfo)
{
    FILE        *file;
    u32         data, result;
    errno_t     err;

    if (( err = fopen_s( &file, path, "rb" ) ) != 0)
    {
        return SoundFileResult_OperationFailed;
    }
    else
    {
        result = SoundFileResult_InvalidFormat;
    }

    fread(&data, 1, sizeof(u32), file);

    switch (data)
    {
        case CHUNK_FORM:

            fread(&data, 1, sizeof(u32), file);
            fread(&data, 1, sizeof(u32), file);

            if (data == CHUNK_AIFF)
            {
                result = SOUND_FILE_AIFF;
            }

            break;

        case CHUNK_RIFF:

            fread(&data, 1, sizeof(u32), file);
            fread(&data, 1, sizeof(u32), file);

            if (data == CHUNK_WAVE)
            {
                result = SOUND_FILE_WAVE;
            }

            break;
    }

    fclose(file);

    return result;
}


/*---------------------------------------------------------------------------*
  Name:         getSoundInfo

  Description:  get information of an input sound file (AIFF or Wave).

  Arguments:    path          path of an input file.
                soundinfo     pointer of a SOUNDINFO structure.

  Returns:      SoundFileResult_Success       OK.
                SoundFileResult_OperationFailed   can't open an AIFF file.
                SoundFileResult_InvalidFormat  neither AIFF nor Wave.
 *---------------------------------------------------------------------------*/

int getSoundInfo(char *path, SOUNDINFO *soundinfo)
{
    u32 result;

    result = getFileType(path, soundinfo);

    switch (result)
    {
        case SOUND_FILE_AIFF:

            result = getAiffInfo(path, soundinfo, NULL);

            break;

        case SOUND_FILE_WAVE:

            result = getWaveInfo(path, soundinfo, NULL);

            break;
    }

    return result;
}


/*---------------------------------------------------------------------------*
  Name:         getSoundSamples

  Description:  get information and samples of an input sound file (AIFF or Wave).

  Arguments:    path          path of an input file.
                soundinfo     pointer of a SOUNDINFO structure.
                dest          pointer of a samples buffer.

  Returns:      SoundFileResult_Success       OK.
                SoundFileResult_OperationFailed   can't open an AIFF file.
                SoundFileResult_InvalidFormat  neither AIFF nor Wave.
 *---------------------------------------------------------------------------*/

int getSoundSamples(char *path, SOUNDINFO *soundinfo, void *dest)
{
    u32 result = getFileType(path, soundinfo);

    switch (result)
    {
        case SOUND_FILE_AIFF:

            result = getAiffInfo(path, soundinfo, dest);

            break;

        case SOUND_FILE_WAVE:

            result = getWaveInfo(path, soundinfo, dest);

            break;
    }

    return result;
}


/*---------------------------------------------------------------------------*
  Name:         writeWaveFile

  Description:  create a Wave file.

  Arguments:    path          path of an output file.
                soundinfo     pointer of a SOUNDINFO structure.
                samples       pointer of a samples buffer.

  Returns:      SoundFileResult_Success       OK.
                SoundFileResult_OperationFailed   can't open an output file.
 *---------------------------------------------------------------------------*/

int writeWaveFile(char *path, SOUNDINFO *info, void *samples)
{
    WAVECHUNK   wavechunk;
    SMPLCHUNK   smpl;
    SMPLLOOP    loop;
    FILE        *file;
    errno_t     err;

    if (( err = fopen_s( &file, path, "wb" ) ) != 0)
    {
        return SoundFileResult_OperationFailed;
    }

    wavCreateHeader(
        &wavechunk,
        info->channels,
        info->samples,
        info->bitsPerSample,
        info->sampleRate,
        info->loopEnd
        );

    wavWriteHeader(
        &wavechunk,
        file
        );

    fwrite(samples, info->bufferLength, sizeof(char), file);

    if (info->loopEnd)
    {
        wavCreateSmplChunk(&wavechunk, &smpl, &loop, info->loopStart, info->loopEnd);
        wavWriteSmplChunk(&smpl, &loop, file);
    }

    fclose(file);

    return SoundFileResult_Success;
}


/*---------------------------------------------------------------------------*
  Name:         writeAiffFile

  Description:  create an AIFF file.

  Arguments:    path          path of an output file.
                soundinfo     pointer of a SOUNDINFO structure.
                samples       pointer of a samples buffer.

  Returns:      SoundFileResult_Success       OK.
                SoundFileResult_OperationFailed   can't open an output file.
 *---------------------------------------------------------------------------*/

int writeAiffFile(char *path, SOUNDINFO *info, void *samples)
{
    AIFFINFO    aiffinfo;
    FILE        *file;
    errno_t     err;

    if (( err = fopen_s( &file, path, "wb" ) ) != 0)
    {
        return SoundFileResult_OperationFailed;
    }

    aiffCreateHeader(
        &aiffinfo,
        info->channels,
        info->samples,
        info->bitsPerSample,
        info->sampleRate
        );

    aiffWriteHeader(
        &aiffinfo,
        file,
        info->channels,
        info->samples,
        info->bitsPerSample,
        info->loopEnd
        );

    if (info->bitsPerSample == 16)
    {
        uint16_t* p = (uint16_t*)(samples);
        int i;
        for (i = 0; i < info->bufferLength / 2; ++i)
        {
            p[i] = reverse_endian_16(p[i]);
        }
    }

    fwrite(samples, info->bufferLength, sizeof(char), file);

    if (info->loopEnd)
    {
        aiffCreateMark(&aiffinfo, info->loopStart, info->loopEnd);
        aiffWriteMark(&aiffinfo, file);
    }

    fclose(file);

    return SoundFileResult_Success;
}
