﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SnapShotDumper.h"
#include "SnapShotDumper_Output.h"
#include "SnapShotDumper_System.h"

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs.h>
#include <nn/svc/svc_Types.h>
#include <nn/os/os_ThreadCommon.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/util/util_FormatString.h>
#include <nn/htc.h>
#include <nn/htc/htc_Result.h>

//==============================================================================
//  Redirect memory operations to our custom allocators/deallocators.
//==============================================================================

void* operator new ( size_t Size )
{
    SNAP_SHOT_DUMPER_TRACE( "new", "new %lld", Size );
    void* pResult = SnapShotDumper::System::Allocate( Size );
    return( pResult );
}

void* operator new [] ( size_t Size )
{
    SNAP_SHOT_DUMPER_TRACE( "new [] ", "new [] %lld", Size );
    void* pResult = SnapShotDumper::System::Allocate( Size );
    return ( pResult );
}

void operator delete ( void* pMemory ) NN_NOEXCEPT
{
    SNAP_SHOT_DUMPER_TRACE( "delete", "delete" );
    SnapShotDumper::System::Deallocate( pMemory, 0 );
}

void operator delete [] ( void* pMemory ) NN_NOEXCEPT
{
    SNAP_SHOT_DUMPER_TRACE( "delete", "delete" );
    SnapShotDumper::System::Deallocate( pMemory, 0 );
}

void* malloc(size_t Size)
{
    SNAP_SHOT_DUMPER_TRACE( "malloc", "malloc %lld", Size );
    return SnapShotDumper::System::Allocate( Size );
}

void free( void* pMemory )
{
    SNAP_SHOT_DUMPER_TRACE( "free", "free" );
    SnapShotDumper::System::Deallocate( pMemory, 0 );
}

void* calloc(size_t num, size_t size)
{
    SNAP_SHOT_DUMPER_TRACE( "calloc", "calloc %lld", num * size );
    size_t Size = num * size;
    void* p = SnapShotDumper::System::Allocate( Size );

    if ( p != NULL )
    {
        memset( p, 0, Size );
    }
    return p;
}

void* realloc(void* p, size_t newSize)
{
    SNAP_SHOT_DUMPER_TRACE( "realloc", "realloc %lld", newSize );
    return NULL;
}

void *aligned_alloc(size_t alignment, size_t size )
{
    SNAP_SHOT_DUMPER_TRACE( "aligned_alloc", "aligned_alloc %lld", size );
    return SnapShotDumper::System::Allocate( size );
}

size_t malloc_usable_size( const void* p )
{
    SNAP_SHOT_DUMPER_TRACE( "malloc_usable_size", "malloc_usable_size" );
    return 0;
}

//============================================================================================

namespace SnapShotDumper
{

namespace System
{

enum
{
    HEAP_SIZE = 0x8000,
};

NN_ALIGNAS(4096) char         g_HeapBuffer[HEAP_SIZE];
nn::lmem::HeapHandle        g_HeapHandle;
const char SdRootFsName[] = "sdmc";
const char PcRootFsName[] = "host";
bool s_MountHost = false;
bool s_MountSdmc = false;

//==============================================================================

void* Allocate( size_t size )
{
    SNAP_SHOT_DUMPER_TRACE( "Allocate", "Allocating %lld", size );

    void* p = nn::lmem::AllocateFromExpHeap( g_HeapHandle, size );
    if( p == nullptr )
    {
        SNAP_SHOT_DUMPER_LOG( "Alloc() returned NULL when trying to allocate %d bytes (%d bytes available)\n",  size, nn::lmem::GetExpHeapTotalFreeSize( g_HeapHandle ) );
    }
    return p;
}

//==============================================================================

void Deallocate( void* Ptr, size_t size )
{
    SNAP_SHOT_DUMPER_TRACE( "Deallocate", "Deallocating %lld", size );
    (void)size;
    nn::lmem::FreeToExpHeap( g_HeapHandle, Ptr );
}

//============================================================================================

nn::Result Init()
{
    g_HeapHandle = nn::lmem::CreateExpHeap( &g_HeapBuffer, sizeof( g_HeapBuffer ), nn::lmem::CreationOption_NoOption );

    nn::fs::SetAllocator( Allocate, Deallocate );

    return nn::dbg::InitializeForSnapShotDumper();
}

void ExtractDirectoryName(char* buffer, size_t bufferSize, const char* path)
{
    std::strncpy(buffer, path, bufferSize);
    *(buffer + bufferSize - 1) = '\0';

    char* ptr = buffer + bufferSize - 1;
    while (ptr > buffer && *ptr != '\\' && *ptr != '/')
    {
        ptr--;
    }
    *ptr = '\0';
}

void ExtractFileName(char* buffer, size_t bufferSize, const char* path)
{
    const char *fileNamePtr = path;
    while (*path != '\0')
    {
        if (*path == '\\' || *path == '/')
        {
            fileNamePtr = path + 1;
        }
        path++;
    }
    std::strncpy(buffer, fileNamePtr, bufferSize);
    *(buffer + bufferSize - 1) = '\0';
}

bool IsHostAvailable()
{
    nn::htc::Initialize();
    const char EnvVarName[] = "NINTENDO_SDK_ROOT";
    size_t length;
    nn::Result result = nn::htc::GetEnvironmentVariableLength(&length, EnvVarName);
    nn::htc::Finalize();

    return !(nn::htc::ResultConnectionFailure().Includes(result));
}

void EnsureDirectory(char *path)
{
    // If path points to network drive, do nothing and return
    // This is not because of technical issue but troublesomeness
    if (*path == '\\' || *path == '/')
    {
        return;
    }

    nn::fs::MountHostRoot();
    char *ptr = path;
    while (*ptr != '\0')
    {
        while (*ptr != '\0' && *ptr != '\\' && *ptr != '/')
        {
            ptr++;
        }
        char org = *ptr;
        *ptr = '\0';
        if (! (ptr > path && *(ptr - 1) == ':'))
        {
            nn::fs::CreateDirectory(path);
        }
        *ptr = org;
        ptr++;
    }
    nn::fs::UnmountHostRoot();
}

nn::Result CreateSuffixedFile(char *filePathBuffer, size_t bufferSize, const char* driveName, const char* fileName)
{
    nn::Result result;
    const int SuffixEnd = 100;

    for (int i = 0; i < SuffixEnd; i++)
    {
        nn::util::SNPrintf(filePathBuffer, bufferSize, "%s:/%s_%02d%s", driveName, fileName, i, SNAP_SHOT_DUMPER_EXTENSION);

        result = nn::fs::CreateFile(filePathBuffer, 0);
        if (result.IsSuccess())
        {
            return nn::ResultSuccess();
        }
    }
    return result;
}

nn::Result FsInit(char *filePathBuffer, size_t bufferSize)
{
    if (IsHostAvailable())
    {
        char fileName[bufferSize];
        ExtractFileName(fileName, bufferSize, filePathBuffer);
        char dirBuffer[bufferSize];
        ExtractDirectoryName(dirBuffer, bufferSize, filePathBuffer);
        EnsureDirectory(dirBuffer);

        nn::Result result = nn::fs::MountHost(PcRootFsName, dirBuffer);
        if ( result.IsSuccess() )
        {
            nn::util::SNPrintf(filePathBuffer, bufferSize, "%s:/%s%s", PcRootFsName, fileName, SNAP_SHOT_DUMPER_EXTENSION);
            nn::Result fileResult = nn::fs::CreateFile( filePathBuffer, 0 );

            if (nn::fs::ResultPathAlreadyExists::Includes(fileResult))
            {
                fileResult = CreateSuffixedFile(filePathBuffer, bufferSize, PcRootFsName, fileName);
            }
            if (fileResult.IsSuccess())
            {
                NN_LOG("[SnapShotDumper] Start dumping to %s...\n", dirBuffer);
                s_MountHost = true;
                return nn::ResultSuccess();
            }
            nn::fs::Unmount(PcRootFsName);
        }

        NN_LOG("[SnapShotDumper] Failed to dump to %s.\nTrying SD card.\n", filePathBuffer);
    }

    const char SdRootDirName[] = "NXDMP";
    nn::Result result = nn::fs::MountSdCardForDebug(SdRootFsName);
    if ( result.IsSuccess() )
    {
        {
            char rootDir[bufferSize];
            nn::util::SNPrintf(rootDir, bufferSize, "%s:/%s", SdRootFsName, SdRootDirName);
            nn::Result dirResult = nn::fs::CreateDirectory(rootDir);
            if ( dirResult.IsFailure() && !nn::fs::ResultPathAlreadyExists().Includes(dirResult) )
            {
                nn::fs::Unmount(SdRootFsName);
                return dirResult;
            }
        }
        char fileName[bufferSize];
        ExtractFileName(fileName, bufferSize, filePathBuffer);
        char path[bufferSize];
        nn::util::SNPrintf(path, bufferSize, "%s/%s", SdRootDirName, fileName);

        nn::util::SNPrintf(filePathBuffer, bufferSize, "%s:/%s%s", SdRootFsName, path, SNAP_SHOT_DUMPER_EXTENSION);

        result = nn::fs::CreateFile( filePathBuffer, 0 );
        if (nn::fs::ResultPathAlreadyExists::Includes(result))
        {
            result = CreateSuffixedFile(filePathBuffer, bufferSize, SdRootFsName, path);
        }

        if (result.IsSuccess())
        {
            NN_LOG("[SnapShotDumper] Start dumping to SD card...\n");
            s_MountSdmc = true;
        }
    }
    return result;
}

//============================================================================================

nn::Result Close()
{
    if (s_MountHost)
    {
        nn::fs::Unmount(PcRootFsName);
    }
    if (s_MountSdmc)
    {
        nn::fs::Unmount(SdRootFsName);
    }
    return nn::dbg::FinalizeForSnapShotDumper();
}

//============================================================================================

nn::Result CompileProcessDetails( Output* pOutput )
{
    nn::svc::Handle ProcessHandle = pOutput->GetHandle();

    if( ProcessHandle.IsValid() == true )
    {
        //====================================================
        // Loop until we've consumed all the debug events.
        //====================================================
        bool FoundException = false;
        for(;;)
        {
            nn::svc::DebugEventInfo eventInfo;
            nn::Result DebugResult = nn::dbg::GetDebugEvent(&eventInfo, ProcessHandle);
            if( DebugResult.IsFailure() )
            {
                break;
            }

            switch ( eventInfo.event )
            {
            case nn::svc::DebugEvent_CreateProcess:
                SNAP_SHOT_DUMPER_LOG( "CompileProcessDetails:  eventInfo.event = DebugEvent_CreateProcess\n" );
                pOutput->SetProcessType( &eventInfo );
                break;

            case nn::svc::DebugEvent_Exception:
                SNAP_SHOT_DUMPER_LOG( "CompileProcessDetails:  DebugEvent_Exception %d\n", eventInfo.info.exception.exceptionCode );
                //NN_SDK_LOG( "CompileProcessDetails:  DebugEvent_Exception %d\n", eventInfo.info.exception.exceptionCode );
                if ( FoundException == false )
                {
                    pOutput->SetException( &eventInfo );
                    FoundException = true;
                }
                break;

            case nn::svc::DebugEvent_CreateThread:
                SNAP_SHOT_DUMPER_LOG( "CompileProcessDetails:  eventInfo.event = DebugEvent_CreateThread\n" );
                pOutput->CreateThread( &eventInfo );
                break;

            case nn::svc::DebugEvent_ExitThread:
                SNAP_SHOT_DUMPER_LOG( "CompileProcessDetails:  eventInfo.event = DebugEvent_ExitThread\n" );
                pOutput->ExitThread( &eventInfo );
                break;

            default:
                SNAP_SHOT_DUMPER_LOG( "CompileProcessDetails:  eventInfo.event = %d\n", eventInfo.event );
                break;
            }
        }

        return nn::ResultSuccess();
    }

    // ProcessHandle.IsValid() != true
    //FIXME
    return nn::ResultSuccess();
}

//============================================================================================

enum
{
    HIGHEST_APP_PRIORITY    = 28,
    APP_PRIORITY_RANGE      = 32,
    LOWEST_APP_PRIORITY     = (HIGHEST_APP_PRIORITY + APP_PRIORITY_RANGE),
};

//============================================================================================

s16 GetThreadPriority(  nn::svc::Handle debugHandle, nn::Bit64 ThreadId )
{
    s16 Priority = nn::os::DefaultThreadPriority;

    // Try to read the thread priority from the system.
    nn::Bit64 r1 = 0;
    nn::Bit32 r2 = 0;
    nn::Result result = nn::dbg::GetDebugThreadParam( &r1, &r2, debugHandle, ThreadId, nn::svc::DebugThreadParam_Priority );
    if( result.IsSuccess() )
    {
        //Cleanse this for public consumption.
        Priority = r2 - HIGHEST_APP_PRIORITY;
    }

    return Priority;
}

//==============================================================================

s16 GetThreadStatus( nn::svc::Handle debugHandle, nn::Bit64 ThreadId )
{
    //Default to runnable suspended
    s16 Ret = 'S';

    nn::Bit64 r1 = 0;
    nn::Bit32 r2 = 0;
    nn::Result result = nn::dbg::GetDebugThreadParam( &r1, &r2, debugHandle, ThreadId, nn::svc::DebugThreadParam_State );
    if( result.IsSuccess() )
    {
        switch( r2 )
        {
            default:
            case nn::svc::ThreadState::ThreadState_Runnable:
                Ret = ((r1 & nn::svc::ThreadSuspend_Debug)? 'S': 'R');
                break;

            case nn::svc::ThreadState::ThreadState_Wait:
                Ret = 'W';
                break;

            case nn::svc::ThreadState::ThreadState_Terminated:
                Ret = 'M';
                break;
        }
    }

    return Ret;
}

//==============================================================================

s16 GetThreadCore(  nn::svc::Handle debugHandle, nn::Bit64 ThreadId )
{
    s16 Core      = 0;

    nn::Bit64 r1 = 0;
    nn::Bit32 r2 = 0;
    nn::Result result = nn::dbg::GetDebugThreadParam( &r1, &r2, debugHandle, ThreadId, nn::svc::DebugThreadParam_CurrentCore );
    if( result.IsSuccess() )
    {
        Core = r2;

        //Get the affinity mask.
        result = nn::dbg::GetDebugThreadParam( &r1, &r2, debugHandle, ThreadId, nn::svc::DebugThreadParam_AffinityMask );
        if( result.IsSuccess() )
        {
            nn::Bit32 affinityMask = (nn::Bit32)r1;

            // This goes in the upper 8 bits of this field.
            coredump::s16 temp = ( affinityMask & 0xFF );
            Core |= (temp << 8);
        }
    }

    return Core;
}


//============================================================================================
} // namespace System
//============================================================================================
}// namespace SnapShotDumper
//============================================================================================

