﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/svc/svc_Types.h>
#include <nn/os.h>
#include <nn/osdbg.h>
#include <nn/arp/arp_Api.h>
#include <cstdlib>
#include <array>
#include "..\..\..\..\Chris\Sources\Libraries\os\detail\os_DefaultUserExceptionHandlerImpl.h"
#include <nn/rocrt/rocrt.h>
#include <nn/ro/detail/ro_RoExceptionInfo.h>
#include <nn/diag/text/diag_SdkTextOs.h>

#include "SnapShotDumper.h"
#include "SnapShotDumper_System.h"
#include "SnapShotDumper_Output.h"

#include "coredump\coredump_Format.h"
#include "..\..\Libraries\tmagent\dbg\dbg_RegisterDefs.h"

//============================================================================================

namespace SnapShotDumper
{

//============================================================================================

template <typename T> T GetValueAtAddress( uintptr_t Address, nn::svc::Handle ProcessHandle )
{
    T Value = 0;
    nn::dbg::ReadDebugProcessMemory( (uintptr_t)&Value, ProcessHandle, Address, sizeof( T ) );
    return Value;
}

//============================================================================================

void GetMemoryAtAddress( uintptr_t Address, nn::svc::Handle ProcessHandle, void* pMemBuffer, size_t Size )
{
    nn::dbg::ReadDebugProcessMemory( (uintptr_t)pMemBuffer, ProcessHandle, Address, Size );
}

//============================================================================================
// Copy-pasted from [SDK]\Programs\Chris\Sources\Libraries\os\detail\os_DefaultUserExceptionHandlerImpl-os.horizon.cpp, line 511,
// then altered to read memory using the debug handle.

void Output::LogStackDump( uintptr_t StackTop, uintptr_t StackBottom )
{
    uintptr_t dumpTop = std::max( static_cast<uintptr_t>(nn::util::align_down( StackTop, 16 )), StackTop );
    uintptr_t dumpBottom = std::min( dumpTop + 16 * 16, StackBottom );

    NN_LOG( "Stack dump:\n" );

    for( uintptr_t ptr = dumpTop; ptr < dumpBottom; ++ptr )
    {
        if( ptr % 16 == 0 )
        {
            NN_LOG( "0x%P: ", ptr );
        }

        if( ptr < StackTop )
        {
            NN_LOG( "   " );
        }
        else
        {
            char Writechar = GetValueAtAddress<char>( ptr, m_ProcessHandle );
            NN_LOG( "%02X ", Writechar );
            NN_UNUSED( Writechar );
        }

        if( ptr % 4 == 3 )
        {
            NN_LOG( " " );
        }

        if( ptr % 16 == 15 )
        {
            char membuffer[32];
            nn::Result Result = nn::dbg::ReadDebugProcessMemory( (uintptr_t)membuffer, m_ProcessHandle, ptr - 15, sizeof( membuffer ) );
            if( Result.IsSuccess() )
            {
                for( int i = 0; i < 16; i++ )
                {
                    uint8_t c = membuffer[i];
                    NN_LOG( "%c", (c >= 0x20 && c <= 0x7e) ? c : '.' );
                    NN_UNUSED( c );
                }
                NN_LOG( "\n" );
            }
            else
            {
                NN_LOG( "\n" );
                break;
            }
        }
    }

    NN_LOG( "\n" );
}

//===========================================================================================

class SymbolTable
{
public:
    //===========================================================================================

    SymbolTable()
    {
        m_StartAddress = 0;
        m_EndAddress = 0;
        m_Count = 0;
        m_Index = -1;
        m_pSymTab = 0;
        m_pStrTab = 0;
        m_Initialized = false;
        m_IsSystemModule = false;
    }

    //===========================================================================================

    void Initialize( uintptr_t Address, uint64_t Size, nn::svc::Handle ProcessHandle ) NN_NOEXCEPT
    {
        m_StartAddress = Address;
        m_EndAddress = m_StartAddress + Size;
        m_ProcessHandle = ProcessHandle;
        m_Count = 0;
        m_Index = -1;
        m_pSymTab = 0;
        m_pStrTab = 0;
        m_IsSystemModule = false;

        nn::rocrt::ModuleHeaderLocation Location;
        GetMemoryAtAddress( m_StartAddress, ProcessHandle, &Location, sizeof( Location ) );

        nn::rocrt::ModuleHeader Header;
        GetMemoryAtAddress( m_StartAddress + Location.headerOffset, ProcessHandle, &Header, sizeof( Header ) );

        if( Header.signature != nn::rocrt::ModuleHeaderVersion )
        {
            return;
        }

        nn::ro::detail::Elf::Dyn dyn;
        uintptr_t pDynAddr = m_StartAddress + Header.dynamicOffset + Location.headerOffset;
        GetMemoryAtAddress( pDynAddr, ProcessHandle, &dyn, sizeof( dyn ) );

        for( ; dyn.GetTag() != nn::ro::detail::Elf::DT_NULL; )
        {
            if( dyn.GetTag() == nn::ro::detail::Elf::DT_SYMTAB )
            {
                m_pSymTab = m_StartAddress + dyn.GetPtr();
            }
            else if( dyn.GetTag() == nn::ro::detail::Elf::DT_STRTAB )
            {
                m_pStrTab = m_StartAddress + dyn.GetPtr();
            }
            else if( dyn.GetTag() == nn::ro::detail::Elf::DT_HASH )
            {
                uintptr_t HashAddress = m_StartAddress + dyn.GetPtr() + sizeof( uint32_t );
                m_Count = GetValueAtAddress<int>( HashAddress, ProcessHandle );
            }
            else
            {
                //NN_LOG( "SymbolTableParser.Initialize found type %d.\n", dyn.GetTag() );
            }
            pDynAddr += sizeof( dyn );
            GetMemoryAtAddress( pDynAddr, ProcessHandle, &dyn, sizeof( dyn ) );
        }

        if( !(m_pSymTab != 0 && m_pStrTab != 0 && m_Count > 0) )
        {
            return;
        }

        m_Initialized = true;
    }

    //=========================================================================

    bool MoveNext() NN_NOEXCEPT
    {
        if( !m_Initialized )
        {
            return false;
        }
        if( !(m_Index + 1 < m_Count) )
        {
            return false;
        }
        MoveTo( m_Index + 1 );
        return true;
    }

    //=========================================================================

    void MoveTo( int index )
    {
        if( m_Initialized )
        {
            m_Index = index;
            uintptr_t SymbolTableAddress = m_pSymTab + (m_Index * sizeof( nn::ro::detail::Elf::Sym ));
            GetMemoryAtAddress( SymbolTableAddress, m_ProcessHandle, &m_CurrentSymbol, sizeof( nn::ro::detail::Elf::Sym ) );
        }
    }

    //=========================================================================

    int GetIndex()
    {
        return m_Index;
    }

    //=========================================================================

    void GetSymbolName( char* pBuffer, size_t SizeOfBuffer ) const NN_NOEXCEPT
    {
        if( m_Initialized )
        {
            uintptr_t NameAddress = m_pStrTab + m_CurrentSymbol.GetName();
            GetMemoryAtAddress( NameAddress, m_ProcessHandle, pBuffer, SizeOfBuffer );
        }
    }

    //=========================================================================

    size_t GetSymbolSize() const NN_NOEXCEPT
    {
        if( m_Initialized )
        {
            return m_CurrentSymbol.GetSize();
        }
        return 0;
    }

    //=========================================================================

    uintptr_t GetSymbolAddress() const NN_NOEXCEPT
    {
        if( m_Initialized )
        {
            return m_StartAddress + m_CurrentSymbol.GetValue();
        }
        return 0;
    }

    //=========================================================================

    unsigned char GetType() const NN_NOEXCEPT
    {
        if( m_Initialized )
        {
            return m_CurrentSymbol.GetType();
        }
        return 0;
    }

    //=========================================================================

    bool Contains( uintptr_t AtAddress )
    {
        if( m_Initialized )
        {
            return ((m_StartAddress <= AtAddress) && (m_EndAddress > AtAddress));
        }
        return false;
    }

    //=========================================================================

    uintptr_t FindNearest( uintptr_t AtAddress, char* pSymbolName, size_t BufferSize, size_t* pSymbolSize )
    {
        if( m_Initialized == false )
        {
            return 0;
        }

        MoveTo( 0 );
        int Index = -1;
        uintptr_t AddressMax = 0u;
        do
        {
            const auto symType = GetType();
            if( !(symType == nn::ro::detail::Elf::STT_FUNC || symType == nn::ro::detail::Elf::STT_OBJECT) )
            {
                continue;
            }

            const auto symAddress = GetSymbolAddress();
            if( AddressMax < symAddress && symAddress <= AtAddress )
            {
                Index = GetIndex();
                AddressMax = symAddress;
            }
        }
        while( MoveNext() );

        if( Index >= 0 )
        {
            MoveTo( Index );
            GetSymbolName( pSymbolName, BufferSize );
            *pSymbolSize = GetSymbolSize();
        }

        return AddressMax;
    }

    void SetIsSystemModule( bool IsSystemModule )
    {
        m_IsSystemModule = IsSystemModule;
    }

    bool GetIsSystemModule(){ return m_IsSystemModule;}

    //=========================================================================

private:
    uintptr_t m_pSymTab;
    uintptr_t m_pStrTab;
    uintptr_t m_StartAddress;
    uintptr_t m_EndAddress;
    nn::svc::Handle m_ProcessHandle;
    int m_Count;
    int m_Index;
    bool m_Initialized;
    nn::ro::detail::Elf::Sym m_CurrentSymbol;
    bool m_IsSystemModule;
};

//===========================================================================================

static s32 s_NumberOfSymbolTables = 0;
static SymbolTable* s_pSymbolTables = NULL;

static void InitializeSymbolTables( s32 NumberOfModules, nn::dbg::ModuleInfo* pModules, nn::svc::Handle ProcessHandle )
{
    s_NumberOfSymbolTables = NumberOfModules;
    if( s_NumberOfSymbolTables > 0 )
    {
        s_pSymbolTables = new SymbolTable[s_NumberOfSymbolTables];

        for( int Index = 0; Index < s_NumberOfSymbolTables; Index += 1 )
        {
            SymbolTable* pTable = &s_pSymbolTables[Index];
            pTable->Initialize( pModules[Index].address, pModules[Index].size, ProcessHandle );
        }
    }
}

//===========================================================================================

static SymbolTable* GetSymbolTable( uintptr_t address )
{
    for( int Index = 0; Index < s_NumberOfSymbolTables; Index += 1 )
    {
        SymbolTable* pTable = &s_pSymbolTables[Index];
        if( pTable->Contains( address ) )
        {
            return pTable;
        }
    }
    return NULL;
}

//===========================================================================================

static void FinalizeSymbolTables()
{
    if( s_pSymbolTables != NULL )
    {
        delete [] s_pSymbolTables;
        s_pSymbolTables = NULL;
    }
}

//============================================================================================

static bool IsSystemSymbol( uintptr_t Address )
{
    SymbolTable* pTable = GetSymbolTable( Address );
    if( pTable != NULL )
    {
        return pTable->GetIsSystemModule();
    }
    return false;
}

//============================================================================================
// Logs the symbol at the passed address.
// Notice the ShowSymbolName variable defaults to TRUE:  The DefaultUserExceptionHandler shows
// the symbol conditional upon whether or not the app is a system app.  Not sure how to do
// that here in SSD, or even if it's relevant.  However, we'll leave the condition there in
// case the design calls for us to filter out system symbols.
//============================================================================================
void Output::LogBacktraceItem( uintptr_t Address, bool ShowSymbolName = true )
{
    if( ShowSymbolName )
    {
        SymbolTable* pTable = GetSymbolTable( Address );
        if( pTable )
        {
            char SymbolName[1024];
            size_t SymbolSize;
            uintptr_t SymbolAddress = pTable->FindNearest( Address - 1, SymbolName, sizeof( SymbolName ), &SymbolSize );
            if( SymbolAddress != 0 )
            {
                auto isOutOfFunction = Address > SymbolAddress + SymbolSize;
                uintptr_t SymbolOffset = Address - SymbolAddress;
                NN_UNUSED( isOutOfFunction );
                NN_UNUSED( SymbolOffset );

#if defined(NN_BUILD_CONFIG_ADDRESS_64)
                NN_LOG( "  0x%016llx %s+0x%llx %s\n", Address, SymbolName, SymbolOffset, isOutOfFunction ? " (too far)" : "");
#else
                NN_LOG( "  0x%08x %s+0x%x %s\n", Address, SymbolName, SymbolOffset, isOutOfFunction ? " (too far)" : "" );
#endif
                return;
            }
        }
    }
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
    NN_LOG( "  0x%016llx (unknown)\n", Address );
#else
    NN_LOG( "  0x%08x (unknown)\n", Address );
#endif
}

//============================================================================================

void Output::LogStackTrace( nn::svc::ThreadContext* pContext )
{
    NN_LOG( "Stack trace:\n" );
    const int traceAddressCountMax = 32;

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    uintptr_t linkAddress = static_cast<uintptr_t>(pContext->r[14]);
    uintptr_t frameAddress = static_cast<uintptr_t>(pContext->r[11]);
#else
    uintptr_t linkAddress = static_cast<uintptr_t>(pContext->r[30]);
    uintptr_t frameAddress = static_cast<uintptr_t>(pContext->r[29]);
#endif
    bool isUserSymbol = !IsSystemSymbol( linkAddress );
    for( int traceAddressCount = 0; traceAddressCount < traceAddressCountMax - 1; traceAddressCount++ )
    {
        if( !frameAddress )
        {
            break;
        }
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
        if( frameAddress & 0x3 )
        {
            break;
        }
        struct
        {
            uint32_t frame;
            uint32_t lr;
        } frame;
        {
            uint32_t v = *reinterpret_cast<uint32_t*>(frameAddress);
            if( pContext->r[13] <= v && v < pContext->r[13] /* + MemoryPageSize */ )
            {
                // clang
                frame.frame = GetValueAtAddress<uint32_t>( frameAddress, m_ProcessHandle );
                frame.lr = GetValueAtAddress<uint32_t>( frameAddress + 4, m_ProcessHandle );
            }
            else
            {
                // gcc
                frame.lr = GetValueAtAddress<uint32_t>( frameAddress, m_ProcessHandle );
                frame.frame = GetValueAtAddress<uint32_t>( frameAddress - 4, m_ProcessHandle );
            }
            if( frame.frame == 0 || frame.lr == 0 )
            {
                // 不正なアドレスをたどってしまった場合は中断する
                break;
            }
        }

#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)

        if( frameAddress & 0xF )
        {
            break;
        }
        struct
        {
            uint64_t frame;
            uint64_t lr;
        } frame;
        {
            frame.frame = GetValueAtAddress<uint64_t>( frameAddress, m_ProcessHandle );
            frame.lr = GetValueAtAddress<uint64_t>( frameAddress + sizeof( frame.frame ), m_ProcessHandle );

            if( frame.frame == 0 || frame.lr == 0 )
            {
                // 不正なアドレスをたどってしまった場合は中断する
                break;
            }
        }
#endif
        bool isCalledByUser = !IsSystemSymbol( frame.lr );
        LogBacktraceItem( linkAddress, isUserSymbol || isCalledByUser );
        linkAddress = frame.lr;
        frameAddress = frame.frame;
        isUserSymbol = isCalledByUser; // ユーザが呼び出しているなら、次はユーザーシンボル
    }

    LogBacktraceItem( linkAddress, isUserSymbol );

    NN_LOG( "\n" );
}

//============================================================================================

//============================================================================================

enum
{
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    NUMBER_OF_CONTEXT_GP_REGISTERS = 29
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    NUMBER_OF_CONTEXT_GP_REGISTERS = 13
#endif
};

//============================================================================================

void Output::AddThreadToLog( coredump::coredump_compiled_data_thread* pThread )
{
    nn::svc::ThreadContext* pContext = GetContext( pThread->m_Id );
    if( pContext == NULL )
    {
        NN_LOG( "Output::AddThread nn::dbg::GetDebugThreadContext failed.\n" );
        return;
    }

    //======================================================================================
    // Convert to the UserExceptionInfo type so we can call the nn::os::detail logging API.
    //======================================================================================
    nn::os::UserExceptionInfo info;

    //Save our registers
    int RegIndex = 0;
    for( RegIndex = 0; RegIndex < NUMBER_OF_CONTEXT_GP_REGISTERS; RegIndex += 1 )
    {
        info.detail.r[RegIndex] = pContext->r[RegIndex];
    }
    info.detail.r[RegIndex + 0] = pContext->fp;
    info.detail.r[RegIndex + 1] = pContext->lr;

    info.detail.fp = pContext->fp;
    info.detail.lr = pContext->lr;
    info.detail.sp = pContext->sp;
    info.detail.pc = pContext->pc;

    for( RegIndex = 0; RegIndex < 32; RegIndex += 1 )
    {
#if defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
        info.detail.V[RegIndex] = pContext->v[RegIndex];
#else
        info.detail.Q[RegIndex] = pContext->v[RegIndex];
#endif
    }

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    info.detail.pstate = pContext->pstate;
    info.detail.afsr0 = pContext->fpcr;
    info.detail.afsr1 = pContext->fpsr;
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    info.detail.pstate = pContext->cpsr;
    info.detail.afsr0 = pContext->fpscr;
    info.detail.afsr1 = pContext->fpexc;
#endif
    info.detail.far = pContext->tpidr;

#if defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    uintptr_t stackTop = reinterpret_cast<uintptr_t>(pContext->sp);
    uintptr_t stackBottom = reinterpret_cast<uintptr_t>(pContext->sp) + pThread->m_StackSize;
#else
    uintptr_t stackTop = reinterpret_cast<uintptr_t>((nn::Bit32)pContext->sp);
    uintptr_t stackBottom = reinterpret_cast<uintptr_t>((nn::Bit32)pContext->sp) + (nn::Bit32)pThread->m_StackSize;
#endif

    NN_LOG( "Thread         : %02lld (%s)\n", pThread->m_Id, pThread->m_Name );
    NN_LOG( "Stack Top      : 0x%P\n", stackTop );
    NN_LOG( "Stack Bottom   : 0x%P  (size=0x%zx)\n", stackBottom, pThread->m_StackSize );
    NN_LOG( "\n" );

#if defined(NN_DETAIL_ENABLE_SDK_LOG)
    nn::os::detail::PrintGeneralPurposeRegisters( &info );
    nn::os::detail::PrintSpecialRegisters( &info );
    nn::os::detail::PrintFloatingPointRegisters( &info );
#endif
    // Calling these from here causes us to crash, so we'll have to bake our own.
    // nn::os::detail::PrintStackDump( &info, stackTop, stackBottom );
    // nn::os::detail::PrintStackTrace( &info );
    LogStackDump( stackTop, stackBottom );
    LogStackTrace( pContext );
}

//============================================================================================

void Output::AddModulesToLog()
{
//Modules:
//    base               size               path
//        0x0000003744400000 0x000000000000F000 nnrtld
//        0x000000374440F000 0x000000000000A000 OsUserExceptionHandler
//        0x0000003744419000 0x00000000001AA000 nvn
//        0x00000037445C3000 0x0000000000033000 nnDisplay
//        0x00000037445F6000 0x000000000076A000 nnSdk

    s32 NumberOfModules = 0;
    nn::dbg::ModuleInfo* pModuleBuffer = GetModules( &NumberOfModules );
    if( pModuleBuffer != NULL )
    {
        NN_LOG( "Modules:\n" );
        NN_LOG( "  base       size       name/path\n" );

        for( int ModuleIndex = 0; ModuleIndex < NumberOfModules; ModuleIndex += 1 )
        {
            nn::dbg::ModuleInfo* pDefined = &pModuleBuffer[ModuleIndex];

            char ModuleName[1024];
            memset( ModuleName, 0, sizeof( ModuleName ) );
            if( FindModuleName( pDefined->address, ModuleName ) == false )
            {
                //=================================================================================================
                // Until we get a proper file name and path from the OS, we need to tell whoever's looking at
                // this file that all we have is a 20-byte identifier.  We'll do that by using this "ID="
                // string, then putting the identifier after that.
                //=================================================================================================
                strcpy( ModuleName, "ID=" );
                char* pWriteId = ModuleName + 3;
                memcpy( pWriteId, pDefined->moduleId, sizeof( pDefined->moduleId ) );
            }

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
            NN_LOG( "  0x%08llx 0x%08llx %s\n", pDefined->address, pDefined->size, ModuleName );
#else
            NN_LOG( "  0x%08x 0x%08x %s\n", pDefined->address, pDefined->size, ModuleName );
#endif
        }
    }
}

//============================================================================================

static void LogSummary( int ExceptionId )
{
    NN_LOG( "\n==============================================================================\n" );
    const char* pExceptionTypeString = NN_TEXT_OS( "不正な ExceptionType です" );

    switch( ExceptionId )
    {
    case nn::svc::DebugException::DebugException_UndefinedInstruction:
        pExceptionTypeString = NN_TEXT_OS( "未定義命令の実行" );
        break;

    case nn::svc::DebugException::DebugException_AccessViolationInstruction:
        pExceptionTypeString = NN_TEXT_OS( "不正なメモリ領域への命令アクセス" );
        break;

    case nn::svc::DebugException::DebugException_AccessViolationData:
        pExceptionTypeString = NN_TEXT_OS( "不正なメモリ領域へのデータアクセス" );
        break;
    case nn::svc::DebugException::DebugException_DataTypeMissaligned:
        pExceptionTypeString = NN_TEXT_OS( "不正なアライメントでの命令アクセス" );
        break;

    case nn::svc::DebugException::DebugException_AttachBreak:
    case nn::svc::DebugException::DebugException_BreakPoint:
    case nn::svc::DebugException::DebugException_UserBreak:
    case nn::svc::DebugException::DebugException_DebuggerBreak:
    default:
        pExceptionTypeString = NN_TEXT_OS( "例外命令の実行" );
        break;

    // Removed as per SIGLO-76675, comment ID: 2810953
    //case nn::svc::DebugException::DebugException_UndefinedSystemCall:
    //    pExceptionTypeString = NN_TEXT_OS( "無効なシステムコールの呼び出し" );
    //    break;

    case nn::svc::DebugException::DebugException_MemorySystemError:
        pExceptionTypeString = NN_TEXT_OS( "メモリシステムでのエラー" );
        break;
    }

    NN_LOG( "  %s\n", pExceptionTypeString );
    NN_LOG( "==============================================================================\n" );
    NN_LOG( "\n" );
}

//============================================================================================

static const char* ppSystemModuleNames[] =
{
    "glslc",
    "llgd-target",
    "multimedia",
    "nnPcm",
    "nvperf_host",
    "nvperf_target",
    "opengl",
    "vulkan"
};
#define NUMBER_OF_SYSTEM_MODULES ( ( sizeof(ppSystemModuleNames) ) / ( sizeof(char*) ) )

static bool IsSystemModule( char* pModuleName )
{
    for( int Index = 0; Index < NUMBER_OF_SYSTEM_MODULES; Index += 1 )
    {
        if( ::StringsEqual( pModuleName, ppSystemModuleNames[Index] )  )
        {
            return true;
        }
    }

    // Because there are different versions of nnSdk (nnSdkEn, etc.), just check for the base name.
    return ( strncmp( pModuleName, "nnSdk", 5 ) == 0 );
}

//============================================================================================
// The format to output is the same as implemented in Programs\Chris\Sources\Libraries\os\detail\os_DefaultUserExceptionHandlerImpl-os.horizon.cpp.
// Output example is like Samples\Sources\Applications\OsUserExceptionHandler\OsUserExceptionHandler_OutputExample.txt.
nn::Result Output::Log()
{
    LogSummary( m_Data.m_ExceptionId );

    //======================================================
    // We'll need our symbol tables for stack unwinding.
    //======================================================
    s32 NumberOfModules = 0;
    nn::dbg::ModuleInfo* pModuleBuffer = GetModules( &NumberOfModules );
    if( pModuleBuffer != NULL )
    {
        InitializeSymbolTables( NumberOfModules, pModuleBuffer, m_ProcessHandle );

        //========================================================
        // We will filter out system modules based on their name.
        //========================================================
        for( int Index = 0; Index < NumberOfModules; Index += 1 )
        {
            char ModuleName[sizeof( coredump::coredump_module_info::m_ModuleName )];
            if( FindModuleName( pModuleBuffer[Index].address, ModuleName ) )
            {
                if( IsSystemModule( ModuleName ) )
                {
                    SymbolTable* pTable = &s_pSymbolTables[Index];
                    pTable->SetIsSystemModule( true );
                }
            }
        }
    }

    for( int ThreadIndex = 0; ThreadIndex < m_Data.m_NumberOfThreads; ThreadIndex += 1 )
    {
        AddThreadToLog( &m_Data.m_pThreads[ThreadIndex] );
    }

    AddModulesToLog();

    FinalizeSymbolTables();

    return nn::ResultSuccess();
}

//============================================================================================
}// namespace SnapShotDumper
//============================================================================================

