﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <string>
#include <utility>
#include <vector>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Web.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Web.h"
#include "SettingsManager_Utility.h"

namespace nn { namespace settings { namespace fwdbg {

void SetWebInspectorFlag(bool value) NN_NOEXCEPT;

void SetAllowedSslHosts(
    const ::nn::settings::system::AllowedSslHost values[],
    size_t count) NN_NOEXCEPT;

void SetHostFsMountPoint(
    const ::nn::settings::system::HostFsMountPoint& value) NN_NOEXCEPT;

}}} // namespace nn::settings::fwdbg

namespace {

//!< Cruiser 用デバッグ設定のキー
const char* const SettingNameWebSettings = "web_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< Web インスペクタを使用するか否かを表す値のキー
const char* const KeyUseWebInspector = "use_web_inspector";

//!< SSL チェック無効化ホストのキー
const char* const KeyAllowedSslHosts = "allowed_ssl_hosts";

//!< SSL チェック無効化ホストのホスト名のキー
const char* const KeyHostName = "host_name";

//!< ホスト FS のマウントポイントのキー
const char* const KeyHostRootPath = "host_root_path";

//!< Web インスペクタを使用するか否かを表す値をエクスポートします。
bool ExportWebInspectorFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyUseWebInspector);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::UsesWebInspector()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyUseWebInspector, ::std::move(node)));

    return true;
}

//!< Web インスペクタを使用するか否かを表す値をインポートします。
bool ImportWebInspectorFlag(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyUseWebInspector;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto webInspectorFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&webInspectorFlag));

    ::nn::settings::fwdbg::SetWebInspectorFlag(webInspectorFlag);

    return true;
}

//!< SSL チェック無効化ホストをエクスポートします。
bool ExportAllowedSslHosts(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    const size_t EntryCountMax = ::nn::settings::system::AllowedSslHostCountMax;

    ::std::unique_ptr<::nn::settings::system::AllowedSslHost[]> entries(
        new ::nn::settings::system::AllowedSslHost[EntryCountMax]());

    const size_t count = ::nn::settings::system::GetAllowedSslHosts(
        entries.get(), EntryCountMax);

    auto arrayNode = Node::CreateArrayNode();

    for (size_t i = 0; i < count; ++i)
    {
        NameScope nameScope(KeyAllowedSslHosts);

        const ::nn::settings::system::AllowedSslHost& settings = entries[i];

        auto dictNode = Node::CreateObjectNode();

        auto valueNode = Node::CreateStringNode();

        {
            NameScope subNameScope(KeyHostName);

            COMMAND_DO(valueNode.SetValue(settings.name));
        }

        COMMAND_DO(dictNode.AppendMember(KeyHostName, ::std::move(valueNode)));

        COMMAND_DO(arrayNode.AppendElement(::std::move(dictNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyAllowedSslHosts, ::std::move(arrayNode)));

    return true;
}

//!< SSL チェック無効化ホストをインポートします。
bool ImportAllowedSslHosts(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyAllowedSslHosts;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    const size_t EntryCountMax = ::nn::settings::system::AllowedSslHostCountMax;

    ::std::unique_ptr<::nn::settings::system::AllowedSslHost[]> entries(
        new ::nn::settings::system::AllowedSslHost[EntryCountMax]());

    size_t count = 0;

    auto elementCount = size_t();

    COMMAND_DO(pNode->GetElementCount(&elementCount));

    for (size_t i = 0; i < elementCount; ++i)
    {
        if (count >= EntryCountMax)
        {
            break;
        }

        ::nn::settings::system::AllowedSslHost& settings = entries[count];

        ::std::unique_ptr<const Node> pDictNode;
        COMMAND_DO(pNode->GetElement(&pDictNode, i));

        ::std::unique_ptr<const Node> pValueNode;
        COMMAND_DO(pDictNode->GetMember(&pValueNode, KeyHostName));

        NameScope subNameScope(KeyHostName);

        ::std::string value;
        COMMAND_DO(pValueNode->GetValue(&value));

        if (value.size() >= NN_ARRAY_SIZE(settings.name))
        {
            PrintErrorCode(
                ErrorCode::NodeValueTooLong, subNameScope.Get(), value);

            return false;
        }

        ::std::copy(value.begin(), value.end(), settings.name);

        ++count;
    }

    ::nn::settings::fwdbg::SetAllowedSslHosts(entries.get(), count);

    return true;
}

//!< ホスト FS のマウントポイントをエクスポートします。
bool ExportHostFsMountPoint(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyHostRootPath);

        auto currentValueNode = Node::CreateStringNode();

        ::nn::settings::system::HostFsMountPoint hostFsMountPoint = {};

        ::nn::settings::system::GetHostFsMountPoint(&hostFsMountPoint);

        COMMAND_DO(
            currentValueNode.SetValue(hostFsMountPoint.path));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(""));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyHostRootPath, ::std::move(node)));

    return true;
}

//!< ホスト FS のマウントポイントをインポートします。
bool ImportHostFsMountPoint(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyHostRootPath;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string value;
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    ::nn::settings::system::HostFsMountPoint hostFsMountPoint = {};

    if (value.size() >= NN_ARRAY_SIZE(hostFsMountPoint.path))
    {
        PrintErrorCode(ErrorCode::NodeValueTooLong, nameScope.Get(), value);

        return false;
    }

    ::std::copy(value.begin(), value.end(), hostFsMountPoint.path);

    ::nn::settings::fwdbg::SetHostFsMountPoint(hostFsMountPoint);

    return true;
}

} // namespace

bool IsSettingNameWebSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameWebSettings);
}

bool ExportWebSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameWebSettings);

        COMMAND_DO(ExportWebInspectorFlag(&node));

        COMMAND_DO(ExportAllowedSslHosts(&node));

        COMMAND_DO(ExportHostFsMountPoint(&node));
    }

    COMMAND_DO(pNode->AppendMember(SettingNameWebSettings, ::std::move(node)));

    return true;
}

bool ImportWebSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameWebSettings);

    ::std::vector<::std::string> keys;

    COMMAND_DO(node.GetKeys(&keys));

    COMMAND_DO(ImportWebInspectorFlag(node, keys));

    COMMAND_DO(ImportAllowedSslHosts(node, keys));

    COMMAND_DO(ImportHostFsMountPoint(node, keys));

    return true;
}
