﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Tv.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Tv.h"
#include "SettingsManager_Utility.h"

namespace {

//!< 設定情報ヘッダのキー
const char* const SettingNameTvSettings = "tv_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 選択肢のキー
const char* const KeyChoices = "choices";

//!< TV 設定のゲームに 4K の利用を許可するか否かを表す値のキー
const char* const KeyTvSettingsAllows4k = "allows_4k";

//!< TV 設定のゲームに 3D の利用を許可するか否かを表す値のキー
const char* const KeyTvSettingsAllows3d = "allows_3d";

//!< TV 設定の CEC を利用するか否かを表す値のキー
const char* const KeyTvSettingsAllowsCec = "allows_cec";

//!< TV 設定の画面焼け軽減を行うか否かを表す値のキー
const char* const KeyTvSettingsPreventsScreenBurnIn = "prevents_screen_burn_in";

//!< TV 設定の TV の解像度設定のキー
const char* const KeyTvResolution = "tv_resolution";

//!< TV 設定の HDMI のコンテンツタイプ設定のキー
const char* const KeyHdmiContentType = "hdmi_content_type";

//!< TV 設定の RGB レンジ設定のキー
const char* const KeyRgbRange = "rgb_range";

//!< TV 設定の CMU モード設定のキー
const char* const KeyCmuMode = "cmu_mode";

//!< TV 設定の TV のアンダースキャン設定のキー
const char* const KeyTvUnderscan = "tv_underscan";

//!< TV 設定の TV のガンマ設定のキー
const char* const KeyTvGamma = "tv_gamma";

//!< TV 設定の CMU モード用のコントラスト比設定のキー
const char* const KeyContrastRatio = "contrast_ratio";

//!< 自動を表す TV の解像度設定
const char* const TvResolutionAuto = "Auto";

//!< 1080p を表す TV の解像度設定
const char* const TvResolution1080p = "1080p";

//!< 720p を表す TV の解像度設定
const char* const TvResolution720p = "720p";

//!< 480p を表す TV の解像度設定
const char* const TvResolution480p = "480p";

//!< 無しを表す HDMI のコンテンツタイプ設定
const char* const HdmiContentTypeNone = "None";

//!< グラフィックスを表す HDMI のコンテンツタイプ設定
const char* const HdmiContentTypeGraphics = "Graphics";

//!< シネマを表す HDMI のコンテンツタイプ設定
const char* const HdmiContentTypeCinema = "Cinema";

//!< フォトを表す HDMI のコンテンツタイプ設定
const char* const HdmiContentTypePhoto = "Photo";

//!< ゲームを表す HDMI のコンテンツタイプ設定
const char* const HdmiContentTypeGame = "Game";

//!< 自動を表す RGB レンジ設定
const char* const RgbRangeAuto = "Auto";

//!< フルを表す RGB レンジ設定
const char* const RgbRangeFull = "Full";

//!< リミテッドを表す RGB レンジ設定
const char* const RgbRangeLimited = "Limited";

//!< 無しを表す CMU モード設定
const char* const CmuModeNone = "None";

//!< 色反転を表す CMU モード設定
const char* const CmuModeColorInvert = "ColorInvert";

//!< ハイコントラストを表す CMU モード設定
const char* const CmuModeHighContrast = "HighContrast";

//!< グレースケールを表す CMU モード設定
const char* const CmuModeGrayScale = "GrayScale";

//!< TV 設定のゲームに 4K の利用を許可するか否かを表す値をエクスポートします。
bool ExportTvSettingsAllows4k(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyTvSettingsAllows4k);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::TvFlag::Allows4k>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyTvSettingsAllows4k, ::std::move(node)));

    return true;
}

//!< TV 設定のゲームに 4K の利用を許可するか否かを表す値をインポートします。
bool ImportTvSettingsAllows4k(
    ::nn::settings::system::TvSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyTvSettingsAllows4k));

    NameScope nameScope(KeyTvSettingsAllows4k);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto allows4k = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&allows4k));

    pSettings->flags.Set<::nn::settings::system::TvFlag::Allows4k>(allows4k);

    return true;
}

//!< TV 設定のゲームに 3D の利用を許可するか否かを表す値をエクスポートします。
bool ExportTvSettingsAllows3d(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyTvSettingsAllows3d);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::TvFlag::Allows3d>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyTvSettingsAllows3d, ::std::move(node)));

    return true;
}

//!< TV 設定のゲームに 3D の利用を許可するか否かを表す値をインポートします。
bool ImportTvSettingsAllows3d(
    ::nn::settings::system::TvSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyTvSettingsAllows3d));

    NameScope nameScope(KeyTvSettingsAllows3d);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto allows3d = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&allows3d));

    pSettings->flags.Set<::nn::settings::system::TvFlag::Allows3d>(allows3d);

    return true;
}

//!< TV 設定の CEC を利用するか否かを表す値をエクスポートします。
bool ExportTvSettingsAllowsCec(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyTvSettingsAllowsCec);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::TvFlag::AllowsCec>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyTvSettingsAllowsCec, ::std::move(node)));

    return true;
}

//!< TV 設定の CEC を利用するか否かを表す値をインポートします。
bool ImportTvSettingsAllowsCec(
    ::nn::settings::system::TvSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyTvSettingsAllowsCec));

    NameScope nameScope(KeyTvSettingsAllowsCec);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto allowsCec = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&allowsCec));

    pSettings->flags.Set<::nn::settings::system::TvFlag::AllowsCec>(allowsCec);

    return true;
}

//!< TV 設定の画面焼け軽減を行うか否かを表す値をエクスポートします。
bool ExportTvSettingsPreventsScreenBurnIn(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyTvSettingsPreventsScreenBurnIn);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::TvFlag::PreventsScreenBurnIn>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyTvSettingsPreventsScreenBurnIn, ::std::move(node)));

    return true;
}

//!< TV 設定の画面焼け軽減を行うか否かを表す値をインポートします。
bool ImportTvSettingsPreventsScreenBurnIn(
    ::nn::settings::system::TvSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyTvSettingsPreventsScreenBurnIn));

    NameScope nameScope(KeyTvSettingsPreventsScreenBurnIn);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto allowsCec = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&allowsCec));

    pSettings->flags.Set<
        ::nn::settings::system::TvFlag::PreventsScreenBurnIn>(allowsCec);

    return true;
}

//!< TV 設定の TV の解像度設定をエクスポートします。
bool ExportTvSettingsTvResolution(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyTvResolution);

        auto currentValueNode = Node::CreateStringNode();

        const auto tvResolution =
            static_cast<::nn::settings::system::TvResolution>(
                settings.tvResolution);

        switch (tvResolution)
        {
        case ::nn::settings::system::TvResolution_Auto:
            COMMAND_DO(currentValueNode.SetValue(TvResolutionAuto));
            break;

        case ::nn::settings::system::TvResolution_1080p:
            COMMAND_DO(currentValueNode.SetValue(TvResolution1080p));
            break;

        case ::nn::settings::system::TvResolution_720p:
            COMMAND_DO(currentValueNode.SetValue(TvResolution720p));
            break;

        case ::nn::settings::system::TvResolution_480p:
            COMMAND_DO(currentValueNode.SetValue(TvResolution480p));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(TvResolutionAuto));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        const char* const choices[] =
        {
            TvResolutionAuto,
            TvResolution1080p,
            TvResolution720p,
            TvResolution480p,
        };

        for (const char* const choice : choices)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(choiceValueNode.SetValue(choice));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyTvResolution, ::std::move(node)));

    return true;
}

//!< TV 設定の TV の解像度設定をインポートします。
bool ImportTvSettingsTvResolution(::nn::settings::system::TvSettings* pSettings,
                                  const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyTvResolution));

    NameScope nameScope(KeyTvResolution);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string tvResolution;
    COMMAND_DO(pCurrentValueNode->GetValue(&tvResolution));

    if (tvResolution == TvResolutionAuto)
    {
        pSettings->tvResolution =
            static_cast<int32_t>(::nn::settings::system::TvResolution_Auto);

        return true;
    }

    if (tvResolution == TvResolution1080p)
    {
        pSettings->tvResolution =
            static_cast<int32_t>(::nn::settings::system::TvResolution_1080p);

        return true;
    }

    if (tvResolution == TvResolution720p)
    {
        pSettings->tvResolution =
            static_cast<int32_t>(::nn::settings::system::TvResolution_720p);

        return true;
    }

    if (tvResolution == TvResolution480p)
    {
        pSettings->tvResolution =
            static_cast<int32_t>(::nn::settings::system::TvResolution_480p);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), tvResolution);

    return false;
}

//!< TV 設定の HDMI のコンテンツタイプ設定をエクスポートします。
bool ExportTvSettingsHdmiContentType(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyHdmiContentType);

        auto currentValueNode = Node::CreateStringNode();

        const auto hdmiContentType =
            static_cast<::nn::settings::system::HdmiContentType>(
                settings.hdmiContentType);

        switch (hdmiContentType)
        {
        case ::nn::settings::system::HdmiContentType_None:
            COMMAND_DO(currentValueNode.SetValue(HdmiContentTypeNone));
            break;

        case ::nn::settings::system::HdmiContentType_Graphics:
            COMMAND_DO(currentValueNode.SetValue(HdmiContentTypeGraphics));
            break;

        case ::nn::settings::system::HdmiContentType_Cinema:
            COMMAND_DO(currentValueNode.SetValue(HdmiContentTypeCinema));
            break;

        case ::nn::settings::system::HdmiContentType_Photo:
            COMMAND_DO(currentValueNode.SetValue(HdmiContentTypePhoto));
            break;

        case ::nn::settings::system::HdmiContentType_Game:
            COMMAND_DO(currentValueNode.SetValue(HdmiContentTypeGame));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(HdmiContentTypeGame));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        const char* const choices[] =
        {
            HdmiContentTypeNone,
            HdmiContentTypeGraphics,
            HdmiContentTypeCinema,
            HdmiContentTypePhoto,
            HdmiContentTypeGame,
        };

        for (const char* const choice : choices)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(choiceValueNode.SetValue(choice));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyHdmiContentType, ::std::move(node)));

    return true;
}

//!< TV 設定の HDMI のコンテンツタイプ設定をインポートします。
bool ImportTvSettingsHdmiContentType(
    ::nn::settings::system::TvSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyHdmiContentType));

    NameScope nameScope(KeyHdmiContentType);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string hdmiContentType;
    COMMAND_DO(pCurrentValueNode->GetValue(&hdmiContentType));

    if (hdmiContentType == HdmiContentTypeNone)
    {
        pSettings->hdmiContentType =
            static_cast<int32_t>(::nn::settings::system::HdmiContentType_None);

        return true;
    }

    if (hdmiContentType == HdmiContentTypeGraphics)
    {
        pSettings->hdmiContentType =
            static_cast<int32_t>(
                ::nn::settings::system::HdmiContentType_Graphics);

        return true;
    }

    if (hdmiContentType == HdmiContentTypeCinema)
    {
        pSettings->hdmiContentType =
            static_cast<int32_t>(
                ::nn::settings::system::HdmiContentType_Cinema);

        return true;
    }

    if (hdmiContentType == HdmiContentTypePhoto)
    {
        pSettings->hdmiContentType =
            static_cast<int32_t>(::nn::settings::system::HdmiContentType_Photo);

        return true;
    }

    if (hdmiContentType == HdmiContentTypeGame)
    {
        pSettings->hdmiContentType =
            static_cast<int32_t>(::nn::settings::system::HdmiContentType_Game);

        return true;
    }

    PrintErrorCode(
        ErrorCode::NodeValueInvalid, nameScope.Get(), hdmiContentType);

    return false;
}

//!< TV 設定の RGB レンジ設定をエクスポートします。
bool ExportTvSettingsRgbRange(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyRgbRange);

        auto currentValueNode = Node::CreateStringNode();

        const auto rgbRange =
            static_cast<::nn::settings::system::RgbRange>(settings.rgbRange);

        switch (rgbRange)
        {
        case ::nn::settings::system::RgbRange_Auto:
            COMMAND_DO(currentValueNode.SetValue(RgbRangeAuto));
            break;

        case ::nn::settings::system::RgbRange_Full:
            COMMAND_DO(currentValueNode.SetValue(RgbRangeFull));
            break;

        case ::nn::settings::system::RgbRange_Limited:
            COMMAND_DO(currentValueNode.SetValue(RgbRangeLimited));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(RgbRangeAuto));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        const char* const choices[] =
        {
            RgbRangeAuto,
            RgbRangeFull,
            RgbRangeLimited,
        };

        for (const char* const choice : choices)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(choiceValueNode.SetValue(choice));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyRgbRange, ::std::move(node)));

    return true;
}

//!< TV 設定の RGB レンジ設定をインポートします。
bool ImportTvSettingsRgbRange(
    ::nn::settings::system::TvSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyRgbRange));

    NameScope nameScope(KeyRgbRange);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string rgbRange;
    COMMAND_DO(pCurrentValueNode->GetValue(&rgbRange));

    if (rgbRange == RgbRangeAuto)
    {
        pSettings->rgbRange =
            static_cast<int32_t>(::nn::settings::system::RgbRange_Auto);

        return true;
    }

    if (rgbRange == RgbRangeFull)
    {
        pSettings->rgbRange =
            static_cast<int32_t>(::nn::settings::system::RgbRange_Full);

        return true;
    }

    if (rgbRange == RgbRangeLimited)
    {
        pSettings->rgbRange =
            static_cast<int32_t>(::nn::settings::system::RgbRange_Limited);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), rgbRange);

    return false;
}

//!< TV 設定の CMU モード設定をエクスポートします。
bool ExportTvSettingsCmuMode(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyCmuMode);

        auto currentValueNode = Node::CreateStringNode();

        switch (static_cast<::nn::settings::system::CmuMode>(settings.cmuMode))
        {
        case ::nn::settings::system::CmuMode_None:
            COMMAND_DO(currentValueNode.SetValue(CmuModeNone));
            break;

        case ::nn::settings::system::CmuMode_ColorInvert:
            COMMAND_DO(currentValueNode.SetValue(CmuModeColorInvert));
            break;

        case ::nn::settings::system::CmuMode_HighContrast:
            COMMAND_DO(currentValueNode.SetValue(CmuModeHighContrast));
            break;

        case ::nn::settings::system::CmuMode_GrayScale:
            COMMAND_DO(currentValueNode.SetValue(CmuModeGrayScale));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(CmuModeNone));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        const char* const choices[] =
        {
            CmuModeNone,
            CmuModeColorInvert,
            CmuModeHighContrast,
            CmuModeGrayScale,
        };

        for (const char* const choice : choices)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(choiceValueNode.SetValue(choice));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyCmuMode, ::std::move(node)));

    return true;
}

//!< TV 設定の CMU モード設定をインポートします。
bool ImportTvSettingsCmuMode(
    ::nn::settings::system::TvSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyCmuMode));

    NameScope nameScope(KeyCmuMode);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string cmuMode;
    COMMAND_DO(pCurrentValueNode->GetValue(&cmuMode));

    if (cmuMode == CmuModeNone)
    {
        pSettings->cmuMode =
            static_cast<int32_t>(::nn::settings::system::CmuMode_None);

        return true;
    }

    if (cmuMode == CmuModeColorInvert)
    {
        pSettings->cmuMode =
            static_cast<int32_t>(::nn::settings::system::CmuMode_ColorInvert);

        return true;
    }

    if (cmuMode == CmuModeHighContrast)
    {
        pSettings->cmuMode =
            static_cast<int32_t>(::nn::settings::system::CmuMode_HighContrast);

        return true;
    }

    if (cmuMode == CmuModeGrayScale)
    {
        pSettings->cmuMode =
            static_cast<int32_t>(::nn::settings::system::CmuMode_GrayScale);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), cmuMode);

    return false;
}

//!< TV 設定の TV のアンダースキャン設定をエクスポートします。
bool ExportTvSettingsTvUnderscan(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyTvUnderscan);

        auto currentValueNode = Node::CreateIntegerNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                static_cast<int32_t>(settings.tvUnderscan)));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateIntegerNode();

        COMMAND_DO(defaultValueNode.SetValue(0));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyTvUnderscan, ::std::move(node)));

    return true;
}

//!< TV 設定の TV のアンダースキャン設定をインポートします。
bool ImportTvSettingsTvUnderscan(
    ::nn::settings::system::TvSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyTvUnderscan));

    NameScope nameScope(KeyTvUnderscan);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto tvUnderscan = int32_t();
    COMMAND_DO(pCurrentValueNode->GetValue(&tvUnderscan));

    pSettings->tvUnderscan = static_cast<uint32_t>(tvUnderscan);

    return true;
}

//!< TV 設定の TV のガンマ設定をエクスポートします。
bool ExportTvSettingsTvGamma(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyTvGamma);

        auto currentValueNode = Node::CreateDoubleNode();

        COMMAND_DO(
            currentValueNode.SetValue(static_cast<double>(settings.tvGamma)));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateDoubleNode();

        COMMAND_DO(defaultValueNode.SetValue(1.0));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyTvGamma, ::std::move(node)));

    return true;
}

//!< TV 設定の TV のガンマ設定をインポートします。
bool ImportTvSettingsTvGamma(
    ::nn::settings::system::TvSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyTvGamma));

    NameScope nameScope(KeyTvGamma);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto tvGamma = double();
    COMMAND_DO(pCurrentValueNode->GetValue(&tvGamma));

    pSettings->tvGamma = static_cast<float>(tvGamma);

    return true;
}

//!< TV 設定の CMU モード用のコントラスト比設定をエクスポートします。
bool ExportTvSettingsContrastRatio(
    Node* pDictNode,
    const ::nn::settings::system::TvSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyContrastRatio);

        auto currentValueNode = Node::CreateDoubleNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                static_cast<double>(settings.contrastRatio)));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateDoubleNode();

        COMMAND_DO(defaultValueNode.SetValue(0.5));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyContrastRatio, ::std::move(node)));

    return true;
}

//!< TV 設定の CMU モード用のコントラスト比設定をインポートします。
bool ImportTvSettingsContrastRatio(
    ::nn::settings::system::TvSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyContrastRatio));

    NameScope nameScope(KeyContrastRatio);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto contrastRatio = double();
    COMMAND_DO(pCurrentValueNode->GetValue(&contrastRatio));

    pSettings->contrastRatio = static_cast<float>(contrastRatio);

    return true;
}

} // namespace

bool IsSettingNameTvSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameTvSettings);
}

bool ExportTvSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    ::nn::settings::system::TvSettings settings = {};
    ::nn::settings::system::GetTvSettings(&settings);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameTvSettings);

        COMMAND_DO(ExportTvSettingsAllows4k(&node, settings));
        COMMAND_DO(ExportTvSettingsAllows3d(&node, settings));
        COMMAND_DO(ExportTvSettingsAllowsCec(&node, settings));
        COMMAND_DO(ExportTvSettingsPreventsScreenBurnIn(&node, settings));
        COMMAND_DO(ExportTvSettingsTvResolution(&node, settings));
        COMMAND_DO(ExportTvSettingsHdmiContentType(&node, settings));
        COMMAND_DO(ExportTvSettingsRgbRange(&node, settings));
        COMMAND_DO(ExportTvSettingsCmuMode(&node, settings));
        COMMAND_DO(ExportTvSettingsTvUnderscan(&node, settings));
        COMMAND_DO(ExportTvSettingsTvGamma(&node, settings));
        COMMAND_DO(ExportTvSettingsContrastRatio(&node, settings));
    }

    COMMAND_DO(pNode->AppendMember(SettingNameTvSettings, ::std::move(node)));

    return true;
}

bool ImportTvSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameTvSettings);

    ::nn::settings::system::TvSettings settings = {};
    COMMAND_DO(ImportTvSettingsAllows4k(&settings, node));
    COMMAND_DO(ImportTvSettingsAllows3d(&settings, node));
    COMMAND_DO(ImportTvSettingsAllowsCec(&settings, node));
    COMMAND_DO(ImportTvSettingsPreventsScreenBurnIn(&settings, node));
    COMMAND_DO(ImportTvSettingsTvResolution(&settings, node));
    COMMAND_DO(ImportTvSettingsHdmiContentType(&settings, node));
    COMMAND_DO(ImportTvSettingsRgbRange(&settings, node));
    COMMAND_DO(ImportTvSettingsCmuMode(&settings, node));
    COMMAND_DO(ImportTvSettingsTvUnderscan(&settings, node));
    COMMAND_DO(ImportTvSettingsTvGamma(&settings, node));
    COMMAND_DO(ImportTvSettingsContrastRatio(&settings, node));

    ::nn::settings::system::SetTvSettings(settings);

    return true;
}
