﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <string>
#include <utility>
#include <vector>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/time/time_SteadyClockTimePoint.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_Uuid.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_SystemApplication.h"
#include "SettingsManager_Utility.h"

namespace {

//!< システムアプリケーション設定のキー
const char* const SettingNameSystemApplicationSettings =
    "system_application_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 選択肢のキー
const char* const KeyChoices = "choices";

//!< 携帯モード時誤操作防止画面の有無を表すフラグのキー
const char* const KeyLockScreenFlag = "lock_screen_flag";

//!< カラーセット ID のキー
const char* const KeyColorSetId = "color_set_id";

//!< バッテリー残量のパーセンテージ表示の有無を表すフラグのキー
const char* const KeyBatteryPercentageFlag = "battery_percentage_flag";

//!< Quest メニューを起動するか否かを表す値のキー
const char* const KeyQuestFlag = "quest_flag";

//!< リージョンコードのキー
const char* const KeyRegionCode = "region_code";

//!< 初回起動が完了しているか否かを表す値のキー
const char* const KeyInitialLaunchCompletionFlag =
    "initial_launch_completion_flag";

//!< ユーザ追加が可能なシーンまで進んだか否かを表す値のキー
const char* const KeyInitialLaunchUserAdditionFlag =
    "initial_launch_user_addition_flag";

//!< 完了時刻が設定されているか否かを表す値のキー
const char* const KeyInitialLaunchTimeStampFlag =
    "initial_launch_timestamp_flag";

//!< 初回起動完了時のタイムスタンプのオフセット値のキー
const char* const KeyInitialLaunchTimeStampOffset =
    "initial_launch_timestamp_offset";

//!< 初回起動完了時のタイムスタンプのソース ID のキー
const char* const KeyInitialLaunchTimeStampSourceId =
    "initial_launch_timestamp_source_id";

//!< 本体の識別名のキー
const char* const KeyDeviceNickName = "device_nick_name";

//!< 日本リージョンを表すリージョンコード
const char* const RegionCodeJapan = "Japan";

//!< 米国リージョンを表すリージョンコード
const char* const RegionCodeUsa = "Usa";

//!< 欧州リージョンを表すリージョンコード
const char* const RegionCodeEurope = "Europe";

//!< 豪州リージョンを表すリージョンコード
const char* const RegionCodeAustralia = "Australia";

//!< 携帯モード時誤操作防止画面の有無を表すフラグをエクスポートします。
bool ExportLockScreenFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyLockScreenFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::GetLockScreenFlag()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyLockScreenFlag, ::std::move(node)));

    return true;
}

//!< 携帯モード時誤操作防止画面の有無を表すフラグをインポートします。
bool ImportLockScreenFlag(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyLockScreenFlag;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto lockScreenFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&lockScreenFlag));

    ::nn::settings::system::SetLockScreenFlag(lockScreenFlag);

    return true;
}

//!< カラーセット ID をエクスポートします。
bool ExportColorSetId(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyColorSetId);

        auto currentValueNode = Node::CreateIntegerNode();

        COMMAND_DO(
            currentValueNode.SetValue(::nn::settings::system::GetColorSetId()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateIntegerNode();

        COMMAND_DO(defaultValueNode.SetValue(0));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyColorSetId, ::std::move(node)));

    return true;
}

//!< カラーセット ID をインポートします。
bool ImportColorSetId(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyColorSetId;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyColorSetId));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto colorSetId = int32_t();
    COMMAND_DO(pCurrentValueNode->GetValue(&colorSetId));

    ::nn::settings::system::SetColorSetId(colorSetId);

    return true;
}

//!< バッテリー残量のパーセンテージ表示の有無を表すフラグをエクスポートします。
bool ExportBatteryPercentageFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyBatteryPercentageFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::GetBatteryPercentageFlag()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyBatteryPercentageFlag, ::std::move(node)));

    return true;
}

//!< バッテリー残量のパーセンテージ表示の有無を表すフラグをインポートします。
bool ImportBatteryPercentageFlag(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyBatteryPercentageFlag;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto batteryPercentageFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&batteryPercentageFlag));

    ::nn::settings::system::SetBatteryPercentageFlag(batteryPercentageFlag);

    return true;
}

//!< Quest メニューを起動するか否かを表す値をエクスポートします。
bool ExportQuestFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyQuestFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(::nn::settings::system::GetQuestFlag()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyQuestFlag, ::std::move(node)));

    return true;
}

//!< Quest メニューを起動するか否かを表す値をインポートします。
bool ImportQuestFlag(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyQuestFlag;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto questFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&questFlag));

    ::nn::settings::system::SetQuestFlag(questFlag);

    return true;
}

//!< リージョンコードをエクスポートします。
bool ExportRegionCode(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyRegionCode);

        auto currentValueNode = Node::CreateStringNode();

        auto regionCode = ::nn::settings::system::RegionCode();

        ::nn::settings::system::GetRegionCode(&regionCode);

        switch (regionCode)
        {
        case ::nn::settings::system::RegionCode_Japan:
            COMMAND_DO(currentValueNode.SetValue(RegionCodeJapan));
            break;

        case ::nn::settings::system::RegionCode_Usa:
            COMMAND_DO(currentValueNode.SetValue(RegionCodeUsa));
            break;

        case ::nn::settings::system::RegionCode_Europe:
            COMMAND_DO(currentValueNode.SetValue(RegionCodeEurope));
            break;

        case ::nn::settings::system::RegionCode_Australia:
            COMMAND_DO(currentValueNode.SetValue(RegionCodeAustralia));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(RegionCodeUsa));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        const char* const choices[] =
        {
            RegionCodeJapan,
            RegionCodeUsa,
            RegionCodeEurope,
            RegionCodeAustralia,
        };

        for (const char* const choice : choices)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(choiceValueNode.SetValue(choice));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyRegionCode, ::std::move(node)));

    return true;
}

//!< リージョンコードをインポートします。
bool ImportRegionCode(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyRegionCode;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string regionCode;
    COMMAND_DO(pCurrentValueNode->GetValue(&regionCode));

    if (regionCode == RegionCodeJapan)
    {
        ::nn::settings::system::SetRegionCode(
            ::nn::settings::system::RegionCode_Japan);

        return true;
    }

    if (regionCode == RegionCodeUsa)
    {
        ::nn::settings::system::SetRegionCode(
            ::nn::settings::system::RegionCode_Usa);

        return true;
    }

    if (regionCode == RegionCodeEurope)
    {
        ::nn::settings::system::SetRegionCode(
            ::nn::settings::system::RegionCode_Europe);

        return true;
    }

    if (regionCode == RegionCodeAustralia)
    {
        ::nn::settings::system::SetRegionCode(
            ::nn::settings::system::RegionCode_Australia);

        return true;
    }

    PrintErrorCode(
        ErrorCode::NodeValueInvalid, nameScope.Get(), regionCode);

    return false;
}

//!< 初回起動設定の制御フラグをエクスポートします。
bool ExportInitialLaunchSettingsFlags(
    Node* pDictNode,
    const ::nn::settings::system::InitialLaunchFlagSet& flags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    {
        auto node = Node::CreateObjectNode();

        {
            NameScope nameScope(KeyInitialLaunchCompletionFlag);

            auto currentValueNode = Node::CreateBooleanNode();

            COMMAND_DO(
                currentValueNode.SetValue(
                    flags.Test<
                        ::nn::settings::system::InitialLaunchFlag::
                            IsCompleted>()));

            COMMAND_DO(
                node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

            auto defaultValueNode = Node::CreateBooleanNode();

            COMMAND_DO(defaultValueNode.SetValue(false));

            COMMAND_DO(
                node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
        }

        COMMAND_DO(
            pDictNode->AppendMember(
                KeyInitialLaunchCompletionFlag, ::std::move(node)));
    }

    {
        auto node = Node::CreateObjectNode();

        {
            NameScope nameScope(KeyInitialLaunchUserAdditionFlag);

            auto currentValueNode = Node::CreateBooleanNode();

            COMMAND_DO(
                currentValueNode.SetValue(
                    flags.Test<
                        ::nn::settings::system::InitialLaunchFlag::
                            IsUserAddable>()));

            COMMAND_DO(
                node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

            auto defaultValueNode = Node::CreateBooleanNode();

            COMMAND_DO(defaultValueNode.SetValue(false));

            COMMAND_DO(
                node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
        }

        COMMAND_DO(
            pDictNode->AppendMember(
                KeyInitialLaunchUserAdditionFlag, ::std::move(node)));
    }

    {
        auto node = Node::CreateObjectNode();

        {
            NameScope nameScope(KeyInitialLaunchTimeStampFlag);

            auto currentValueNode = Node::CreateBooleanNode();

            COMMAND_DO(
                currentValueNode.SetValue(
                    flags.Test<
                        ::nn::settings::system::InitialLaunchFlag::
                            HasTimeStamp>()));

            COMMAND_DO(
                node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

            auto defaultValueNode = Node::CreateBooleanNode();

            COMMAND_DO(defaultValueNode.SetValue(false));

            COMMAND_DO(
                node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
        }

        COMMAND_DO(
            pDictNode->AppendMember(
                KeyInitialLaunchTimeStampFlag, ::std::move(node)));
    }

    return true;
}

//!< 初回起動設定の完了時刻をエクスポートします。
bool ExportInitialLaunchSettingsTimeStamp(
    Node* pDictNode,
    const ::nn::time::SteadyClockTimePoint& timeStamp) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    {
        auto node = Node::CreateObjectNode();

        {
            NameScope nameScope(KeyInitialLaunchTimeStampOffset);

            auto currentValueNode = Node::CreateStringNode();

            ::std::string value;
            EncodeInteger(&value, timeStamp.value);

            COMMAND_DO(currentValueNode.SetValue(value));

            COMMAND_DO(
                node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

            auto defaultValueNode = Node::CreateStringNode();

            COMMAND_DO(defaultValueNode.SetValue("0"));

            COMMAND_DO(
                node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
        }

        COMMAND_DO(
            pDictNode->AppendMember(
                KeyInitialLaunchTimeStampOffset, ::std::move(node)));
    }

    {
        auto node = Node::CreateObjectNode();

        {
            NameScope nameScope(KeyInitialLaunchTimeStampSourceId);

            auto currentValueNode = Node::CreateStringNode();

            ::std::string value;
            EncodeUuid(&value, timeStamp.sourceId);

            COMMAND_DO(currentValueNode.SetValue(value));

            COMMAND_DO(
                node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

            auto defaultValueNode = Node::CreateStringNode();

            EncodeUuid(&value, ::nn::util::InvalidUuid);

            COMMAND_DO(defaultValueNode.SetValue(value));

            COMMAND_DO(
                node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
        }

        COMMAND_DO(
            pDictNode->AppendMember(
                KeyInitialLaunchTimeStampSourceId, ::std::move(node)));
    }

    return true;
}

//!< 初回起動設定をエクスポートします。
bool ExportInitialLaunchSettings(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    ::nn::settings::system::InitialLaunchSettings settings = {};

    ::nn::settings::system::GetInitialLaunchSettings(&settings);

    COMMAND_DO(
        ExportInitialLaunchSettingsFlags(pDictNode, settings.flags));

    COMMAND_DO(
        ExportInitialLaunchSettingsTimeStamp(pDictNode, settings.timeStamp));

    return true;
}

//!< 初回起動設定の制御フラグをインポートします。
bool ImportInitialLaunchSettingsFlags(
    ::nn::settings::system::InitialLaunchFlagSet* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, KeyInitialLaunchCompletionFlag));

        NameScope nameScope(KeyInitialLaunchCompletionFlag);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        auto completionFlag = bool();
        COMMAND_DO(pCurrentValueNode->GetValue(&completionFlag));

        pOutValue->Set<
            ::nn::settings::system::InitialLaunchFlag::IsCompleted>(
                completionFlag);
    }

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(
            dictNode.GetMember(&pNode, KeyInitialLaunchUserAdditionFlag));

        NameScope nameScope(KeyInitialLaunchUserAdditionFlag);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        auto userAdditionFlag = bool();
        COMMAND_DO(pCurrentValueNode->GetValue(&userAdditionFlag));

        pOutValue->Set<
            ::nn::settings::system::InitialLaunchFlag::IsUserAddable>(
                userAdditionFlag);
    }

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, KeyInitialLaunchTimeStampFlag));

        NameScope nameScope(KeyInitialLaunchTimeStampFlag);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        auto timeStampFlag = bool();
        COMMAND_DO(pCurrentValueNode->GetValue(&timeStampFlag));

        pOutValue->Set<
            ::nn::settings::system::InitialLaunchFlag::HasTimeStamp>(
                timeStampFlag);
    }

    return true;
}

//!< 初回起動設定の完了時刻をインポートします。
bool ImportInitialLaunchSettingsTimeStamp(
    ::nn::time::SteadyClockTimePoint* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, KeyInitialLaunchTimeStampOffset));

        NameScope nameScope(KeyInitialLaunchTimeStampOffset);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        ::std::string timeStampOffset;
        COMMAND_DO(pCurrentValueNode->GetValue(&timeStampOffset));

        if (!DecodeInteger(&pOutValue->value, timeStampOffset))
        {
            PrintErrorCode(
                ErrorCode::NodeValueInvalid, nameScope.Get(), timeStampOffset);

            return false;
        }
    }

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(
            dictNode.GetMember(&pNode, KeyInitialLaunchTimeStampSourceId));

        NameScope nameScope(KeyInitialLaunchTimeStampSourceId);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        ::std::string timeStampSourceId;
        COMMAND_DO(pCurrentValueNode->GetValue(&timeStampSourceId));

        if (!DecodeUuid(&pOutValue->sourceId, timeStampSourceId))
        {
            PrintErrorCode(
                ErrorCode::NodeValueInvalid, nameScope.Get(),
                timeStampSourceId);

            return false;
        }
    }

    return true;
}

//!< 初回起動設定をインポートします。
bool ImportInitialLaunchSettings(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const needles[] =
    {
        KeyInitialLaunchCompletionFlag,
        KeyInitialLaunchUserAdditionFlag,
        KeyInitialLaunchTimeStampFlag,
        KeyInitialLaunchTimeStampOffset,
        KeyInitialLaunchTimeStampSourceId,
    };

    auto skips = true;

    for (const char* needle : needles)
    {
        if (::std::find(keys.begin(), keys.end(), needle) != keys.end())
        {
            skips = false;

            break;
        }
    }

    if (skips)
    {
        return true;
    }

    ::nn::settings::system::InitialLaunchSettings settings = {};

    COMMAND_DO(
        ImportInitialLaunchSettingsFlags(&settings.flags, dictNode));

    COMMAND_DO(
        ImportInitialLaunchSettingsTimeStamp(&settings.timeStamp, dictNode));

    ::nn::settings::system::SetInitialLaunchSettings(settings);

    return true;
}

//!< 本体の識別名をエクスポートします。
bool ExportDeviceNickName(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    ::nn::settings::system::DeviceNickName name = {};

    ::nn::settings::system::GetDeviceNickName(&name);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyDeviceNickName);

        auto currentValueNode = Node::CreateStringNode();

        COMMAND_DO(currentValueNode.SetValue(name.string));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(""));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyDeviceNickName, ::std::move(node)));

    return true;
}

//!< 本体の識別名をインポートします。
bool ImportDeviceNickName(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyDeviceNickName;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::nn::settings::system::DeviceNickName name = {};

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string value;
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    auto length = int();

    ::nn::util::CharacterEncodingResult result =
        ::nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(
            &length, value.c_str(), static_cast<int>(value.length()));

    NN_SDK_ASSERT_EQUAL(result, ::nn::util::CharacterEncodingResult_Success);

    NN_UNUSED(result);

    if (length >= 32)
    {
        PrintErrorCode(ErrorCode::NodeValueTooLong, nameScope.Get(), value);

        return false;
    }

    ::std::copy(value.begin(), value.end(), name.string);

    ::nn::settings::system::SetDeviceNickName(name);

    return true;
}

} // namespace

bool IsSettingNameSystemApplicationSettings(
    const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameSystemApplicationSettings);
}

bool ExportSystemApplicationSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameSystemApplicationSettings);

        COMMAND_DO(ExportColorSetId(&node));

        COMMAND_DO(ExportDeviceNickName(&node));

        COMMAND_DO(ExportInitialLaunchSettings(&node));

        COMMAND_DO(ExportLockScreenFlag(&node));

        COMMAND_DO(ExportBatteryPercentageFlag(&node));

        COMMAND_DO(ExportQuestFlag(&node));

        COMMAND_DO(ExportRegionCode(&node));
    }

    COMMAND_DO(
        pNode->AppendMember(
            SettingNameSystemApplicationSettings, ::std::move(node)));

    return true;
}

bool ImportSystemApplicationSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameSystemApplicationSettings);

    ::std::vector<::std::string> keys;

    COMMAND_DO(node.GetKeys(&keys));

    COMMAND_DO(ImportColorSetId(node, keys));

    COMMAND_DO(ImportInitialLaunchSettings(node, keys));

    COMMAND_DO(ImportDeviceNickName(node, keys));

    COMMAND_DO(ImportLockScreenFlag(node, keys));

    COMMAND_DO(ImportBatteryPercentageFlag(node, keys));

    COMMAND_DO(ImportQuestFlag(node, keys));

    COMMAND_DO(ImportRegionCode(node, keys));

    return true;
}
