﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Sleep.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Sleep.h"
#include "SettingsManager_Utility.h"

namespace {

//!< 設定情報ヘッダのキー
const char* const SettingNameSleepSettings = "sleep_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 選択肢のキー
const char* const KeyChoices = "choices";

//!< スリープ設定のメディア再生時に自動スリープするか否かを表す値のキー
const char* const KeySleepsWhilePlayingMedia = "sleeps_while_playing_media";

//!< スリープ設定の携帯モード時の自動スリープまでの時間のキー
const char* const KeyHandheldSleepPlan = "handheld_sleep_plan";

//!< スリープ設定の据置モード時の自動スリープまでの時間のキー
const char* const KeyConsoleSleepPlan = "console_sleep_plan";

//!< 1 分を表すスリープ設定の携帯モード時の自動スリープまでの時間
const char* const HandheldSleepPlan1Min = "1Min";

//!< 3 分を表すスリープ設定の携帯モード時の自動スリープまでの時間
const char* const HandheldSleepPlan3Min = "3Min";

//!< 5 分を表すスリープ設定の携帯モード時の自動スリープまでの時間
const char* const HandheldSleepPlan5Min = "5Min";

//!< 10 分を表すスリープ設定の携帯モード時の自動スリープまでの時間
const char* const HandheldSleepPlan10Min = "10Min";

//!< 30 分を表すスリープ設定の携帯モード時の自動スリープまでの時間
const char* const HandheldSleepPlan30Min = "30Min";

//!< しないを表すスリープ設定の携帯モード時の自動スリープまでの時間
const char* const HandheldSleepPlanNever = "Never";

//!< 1 時間を表す据置モード時の自動スリープまでの時間
const char* const ConsoleSleepPlan1Hour = "1Hour";

//!< 2 時間を表す据置モード時の自動スリープまでの時間
const char* const ConsoleSleepPlan2Hour = "2Hour";

//!< 3 時間を表す据置モード時の自動スリープまでの時間
const char* const ConsoleSleepPlan3Hour = "3Hour";

//!< 6 時間を表す据置モード時の自動スリープまでの時間
const char* const ConsoleSleepPlan6Hour = "6Hour";

//!< 12 時間を表す据置モード時の自動スリープまでの時間
const char* const ConsoleSleepPlan12Hour = "12Hour";

//!< しないを表す据置モード時の自動スリープまでの時間
const char* const ConsoleSleepPlanNever = "Never";

//!< スリープ設定のメディア再生時に自動スリープするか否かを表す値をエクスポートします。
bool ExportSleepSettingsSleepsWhilePlayingMedia(
    Node* pDictNode,
    const ::nn::settings::system::SleepSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeySleepsWhilePlayingMedia);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::SleepFlag::
                        SleepsWhilePlayingMedia>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeySleepsWhilePlayingMedia, ::std::move(node)));

    return true;
}

//!< スリープ設定のメディア再生時に自動スリープするか否かを表す値をインポートします。
bool ImportSleepSettingsSleepsWhilePlayingMedia(
    ::nn::settings::system::SleepSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySleepsWhilePlayingMedia));

    NameScope nameScope(KeySleepsWhilePlayingMedia);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto sleepsWhilePlayingMedia = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&sleepsWhilePlayingMedia));

    pSettings->flags.Set<
        ::nn::settings::system::SleepFlag::SleepsWhilePlayingMedia>(
            sleepsWhilePlayingMedia);

    return true;
}

//!< スリープ設定の携帯モード時の自動スリープまでの時間をエクスポートします。
bool ExportSleepSettingsHandheldSleepPlan(
    Node* pDictNode,
    const ::nn::settings::system::SleepSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyHandheldSleepPlan);

        auto currentValueNode = Node::CreateStringNode();

        const auto handheldSleepPlan =
            static_cast<::nn::settings::system::HandheldSleepPlan>(
                settings.handheldSleepPlan);

        switch (handheldSleepPlan)
        {
        case ::nn::settings::system::HandheldSleepPlan_1Min:
            COMMAND_DO(currentValueNode.SetValue(HandheldSleepPlan1Min));
            break;

        case ::nn::settings::system::HandheldSleepPlan_3Min:
            COMMAND_DO(currentValueNode.SetValue(HandheldSleepPlan3Min));
            break;

        case ::nn::settings::system::HandheldSleepPlan_5Min:
            COMMAND_DO(currentValueNode.SetValue(HandheldSleepPlan5Min));
            break;

        case ::nn::settings::system::HandheldSleepPlan_10Min:
            COMMAND_DO(currentValueNode.SetValue(HandheldSleepPlan10Min));
            break;

        case ::nn::settings::system::HandheldSleepPlan_30Min:
            COMMAND_DO(currentValueNode.SetValue(HandheldSleepPlan30Min));
            break;

        case ::nn::settings::system::HandheldSleepPlan_Never:
            COMMAND_DO(currentValueNode.SetValue(HandheldSleepPlanNever));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(HandheldSleepPlan10Min));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        const char* const choices[] =
        {
            HandheldSleepPlan1Min,
            HandheldSleepPlan3Min,
            HandheldSleepPlan5Min,
            HandheldSleepPlan10Min,
            HandheldSleepPlan30Min,
            HandheldSleepPlanNever,
        };

        for (const char* const choice : choices)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(choiceValueNode.SetValue(choice));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyHandheldSleepPlan, ::std::move(node)));

    return true;
}

//!< スリープ設定の携帯モード時の自動スリープまでの時間をインポートします。
bool ImportSleepSettingsHandheldSleepPlan(
    ::nn::settings::system::SleepSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyHandheldSleepPlan));

    NameScope nameScope(KeyHandheldSleepPlan);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string handheldSleepPlan;
    COMMAND_DO(pCurrentValueNode->GetValue(&handheldSleepPlan));

    if (handheldSleepPlan == HandheldSleepPlan1Min)
    {
        pSettings->handheldSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::HandheldSleepPlan_1Min);

        return true;
    }

    if (handheldSleepPlan == HandheldSleepPlan3Min)
    {
        pSettings->handheldSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::HandheldSleepPlan_3Min);

        return true;
    }

    if (handheldSleepPlan == HandheldSleepPlan5Min)
    {
        pSettings->handheldSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::HandheldSleepPlan_5Min);

        return true;
    }

    if (handheldSleepPlan == HandheldSleepPlan10Min)
    {
        pSettings->handheldSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::HandheldSleepPlan_10Min);

        return true;
    }

    if (handheldSleepPlan == HandheldSleepPlan30Min)
    {
        pSettings->handheldSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::HandheldSleepPlan_30Min);

        return true;
    }

    if (handheldSleepPlan == HandheldSleepPlanNever)
    {
        pSettings->handheldSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::HandheldSleepPlan_Never);

        return true;
    }

    PrintErrorCode(
        ErrorCode::NodeValueInvalid, nameScope.Get(), handheldSleepPlan);

    return false;
}

//!< スリープ設定の据置モード時の自動スリープまでの時間をエクスポートします。
bool ExportSleepSettingsConsoleSleepPlan(
    Node* pDictNode,
    const ::nn::settings::system::SleepSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyConsoleSleepPlan);

        auto currentValueNode = Node::CreateStringNode();

        const auto consoleSleepPlan =
            static_cast<::nn::settings::system::ConsoleSleepPlan>(
                settings.consoleSleepPlan);

        switch (consoleSleepPlan)
        {
        case ::nn::settings::system::ConsoleSleepPlan_1Hour:
            COMMAND_DO(currentValueNode.SetValue(ConsoleSleepPlan1Hour));
            break;

        case ::nn::settings::system::ConsoleSleepPlan_2Hour:
            COMMAND_DO(currentValueNode.SetValue(ConsoleSleepPlan2Hour));
            break;

        case ::nn::settings::system::ConsoleSleepPlan_3Hour:
            COMMAND_DO(currentValueNode.SetValue(ConsoleSleepPlan3Hour));
            break;

        case ::nn::settings::system::ConsoleSleepPlan_6Hour:
            COMMAND_DO(currentValueNode.SetValue(ConsoleSleepPlan6Hour));
            break;

        case ::nn::settings::system::ConsoleSleepPlan_12Hour:
            COMMAND_DO(currentValueNode.SetValue(ConsoleSleepPlan12Hour));
            break;

        case ::nn::settings::system::ConsoleSleepPlan_Never:
            COMMAND_DO(currentValueNode.SetValue(ConsoleSleepPlanNever));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(ConsoleSleepPlan1Hour));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        const char* const choices[] =
        {
            ConsoleSleepPlan1Hour,
            ConsoleSleepPlan2Hour,
            ConsoleSleepPlan3Hour,
            ConsoleSleepPlan6Hour,
            ConsoleSleepPlan12Hour,
            ConsoleSleepPlanNever,
        };

        for (const char* const choice : choices)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(choiceValueNode.SetValue(choice));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyConsoleSleepPlan, ::std::move(node)));

    return true;
}

//!< スリープ設定の据置モード時の自動スリープまでの時間をインポートします。
bool ImportSleepSettingsConsoleSleepPlan(
    ::nn::settings::system::SleepSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyConsoleSleepPlan));

    NameScope nameScope(KeyConsoleSleepPlan);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string consoleSleepPlan;
    COMMAND_DO(pCurrentValueNode->GetValue(&consoleSleepPlan));

    if (consoleSleepPlan == ConsoleSleepPlan1Hour)
    {
        pSettings->consoleSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::ConsoleSleepPlan_1Hour);

        return true;
    }

    if (consoleSleepPlan == ConsoleSleepPlan2Hour)
    {
        pSettings->consoleSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::ConsoleSleepPlan_2Hour);

        return true;
    }

    if (consoleSleepPlan == ConsoleSleepPlan3Hour)
    {
        pSettings->consoleSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::ConsoleSleepPlan_3Hour);

        return true;
    }

    if (consoleSleepPlan == ConsoleSleepPlan6Hour)
    {
        pSettings->consoleSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::ConsoleSleepPlan_6Hour);

        return true;
    }

    if (consoleSleepPlan == ConsoleSleepPlan12Hour)
    {
        pSettings->consoleSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::ConsoleSleepPlan_12Hour);

        return true;
    }

    if (consoleSleepPlan == ConsoleSleepPlanNever)
    {
        pSettings->consoleSleepPlan =
            static_cast<int32_t>(
                ::nn::settings::system::ConsoleSleepPlan_Never);

        return true;
    }

    PrintErrorCode(
        ErrorCode::NodeValueInvalid, nameScope.Get(), consoleSleepPlan);

    return false;
}

} // namespace

bool IsSettingNameSleepSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameSleepSettings);
}

bool ExportSleepSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    ::nn::settings::system::SleepSettings settings = {};
    ::nn::settings::system::GetSleepSettings(&settings);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameSleepSettings);

        COMMAND_DO(ExportSleepSettingsSleepsWhilePlayingMedia(&node, settings));
        COMMAND_DO(ExportSleepSettingsHandheldSleepPlan(&node, settings));
        COMMAND_DO(ExportSleepSettingsConsoleSleepPlan(&node, settings));
    }

    COMMAND_DO(
        pNode->AppendMember(SettingNameSleepSettings, ::std::move(node)));

    return true;
}

bool ImportSleepSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameSleepSettings);

    ::nn::settings::system::SleepSettings settings = {};
    COMMAND_DO(ImportSleepSettingsSleepsWhilePlayingMedia(&settings, node));
    COMMAND_DO(ImportSleepSettingsHandheldSleepPlan(&settings, node));
    COMMAND_DO(ImportSleepSettingsConsoleSleepPlan(&settings, node));

    ::nn::settings::system::SetSleepSettings(settings);

    return true;
}
