﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <string>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

//!< JSON ノードを扱うクラスです。
class Node final
{
    NN_DISALLOW_COPY(Node);

private:
    void* m_pImpl;

public:
    Node() NN_NOEXCEPT;

    Node(Node&& other) NN_NOEXCEPT;

    ~Node() NN_NOEXCEPT;

    Node& operator=(Node&& other) NN_NOEXCEPT;

    //!< 形名を返します。
    const ::std::string& GetTypeName() const NN_NOEXCEPT;

    //!< 配列ノードを作成します。
    static Node CreateArrayNode() NN_NOEXCEPT;

    //!< 配列ノードの子ノードの数を取得します。
    bool GetElementCount(size_t* pOutValue) const NN_NOEXCEPT;

    //!< 配列ノードの子ノードを取得します。
    bool GetElement(::std::unique_ptr<const Node>* pOutValue,
                    size_t index) const NN_NOEXCEPT;

    //!< 配列ノードに子ノードを追加します。
    bool AppendElement(Node value) NN_NOEXCEPT;

    //!< 真理値ノードを作成します。
    static Node CreateBooleanNode() NN_NOEXCEPT;

    //!< 真理値ノードか否かを表す値を返します。
    bool IsBooleanNode() const NN_NOEXCEPT;

    //!< 真理値ノードの値を取得します。
    bool GetValue(bool* pOutValue) const NN_NOEXCEPT;

    //!< 真理値ノードに値を設定します。
    bool SetValue(bool value) NN_NOEXCEPT;

    //!< 浮動小数点数ノードを作成します。
    static Node CreateDoubleNode() NN_NOEXCEPT;

    //!< 浮動小数点数ノードか否かを表す値を返します。
    bool IsDoubleNode() const NN_NOEXCEPT;

    //!< 浮動小数点数ノードの値を取得します。
    bool GetValue(double* pOutValue) const NN_NOEXCEPT;

    //!< 浮動小数点数ノードに値を設定します。
    bool SetValue(double value) NN_NOEXCEPT;

    //!< 整数ノードを作成します。
    static Node CreateIntegerNode() NN_NOEXCEPT;

    //!< 整数ノードか否かを表す値を返します。
    bool IsIntegerNode() const NN_NOEXCEPT;

    //!< 整数ノードの値を取得します。
    bool GetValue(int32_t* pOutValue) const NN_NOEXCEPT;

    //!< 整数ノードに値を設定します。
    bool SetValue(int32_t value) NN_NOEXCEPT;

    //!< オブジェクトノードを作成します。
    static Node CreateObjectNode() NN_NOEXCEPT;

    //!< オブジェクトノードのキーを取得します。
    bool GetKeys(::std::vector<::std::string>* pOutValue) const NN_NOEXCEPT;

    //!< オブジェクトノードの子ノードを取得します。
    bool GetMember(
        ::std::unique_ptr<const Node>* pOutValue, const ::std::string& key
        ) const NN_NOEXCEPT;

    //!< オブジェクトノードに子ノードを追加します。
    bool AppendMember(const ::std::string& key, Node value) NN_NOEXCEPT;

    //!< 文字列ノードを作成します。
    static Node CreateStringNode() NN_NOEXCEPT;

    //!< 文字列ノードか否かを表す値を返します。
    bool IsStringNode() const NN_NOEXCEPT;

    //!< 文字列ノードの値を取得します。
    bool GetValue(::std::string* pOutValue) const NN_NOEXCEPT;

    //!< 文字列ノードに値を設定します。
    bool SetValue(const char* value) NN_NOEXCEPT;

    //!< 文字列ノードに値を設定します。
    bool SetValue(const ::std::string& value) NN_NOEXCEPT;
};

//!< JSON ドキュメントを扱うクラスです。
class Document final
{
    NN_DISALLOW_COPY(Document);

private:
    void* m_pImpl;

public:
    Document() NN_NOEXCEPT;

    Document(Document&& other) NN_NOEXCEPT;

    ~Document() NN_NOEXCEPT;

    Document& operator=(Document&& other) NN_NOEXCEPT;

    //!< ドキュメントをロードします。
    bool Load(const ::std::string& value) NN_NOEXCEPT;

    //!< ドキュメントをセーブします。
    bool Save(::std::string* pOutValue) const NN_NOEXCEPT;

    //!< ルートノードを返します。
    Node GetRootNode() NN_NOEXCEPT;
};
