﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <limits>
#include <memory>
#include <sstream>
#include <string>
#include <utility>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/util/util_BytePtr.h>
#include <nn/TargetConfigs/build_Base.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

#define RAPIDJSON_NO_INT64DEFINE
#define RAPIDJSON_ASSERT(x)             NN_SDK_ASSERT(x)
#define RAPIDJSON_HAS_CXX11_TYPETRAITS  1 //NOLINT(preprocessor/const)
#define RAPIDJSON_HAS_STDSTRING         1 //NOLINT(preprocessor/const)
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#endif
#include <rapidjson/document.h>
#include <rapidjson/encodedstream.h>
#include <rapidjson/prettywriter.h>
#include <rapidjson/error/en.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop)
#endif

namespace {

//!< ノードの型の名前の定義です。
struct NodeTypeName final
{
    static const ::std::string Null;    //!< Null ノードの名前
    static const ::std::string Boolean; //!< 真理値ノードの名前
    static const ::std::string Object;  //!< オブジェクトノードの名前
    static const ::std::string Array;   //!< 配列ノードの名前
    static const ::std::string Integer; //!< 整数ノードの名前
    static const ::std::string Double;  //!< 浮動小数点数ノードの名前
    static const ::std::string String;  //!< 文字列ノードの名前
    static const ::std::string Unknown; //!< 未知のノードの名前
};

//!< ノードを表す構造体です。
struct NodeType final
{
    //!< JSON 値の所有権を持つか否かを表す値
    int hasOwnerShip;

    //!< 読み書き可能な JSON 値
    ::rapidjson::Value* pWritable;

    //!< 読み取り専用の JSON 値
    const ::rapidjson::Value* pConstant;
};

//!< ランタイムメモリアロケータを返します。
::rapidjson::CrtAllocator& GetRuntimeMemoryAllocator() NN_NOEXCEPT;

//!< メモリプールアロケータを返します。
::rapidjson::MemoryPoolAllocator<>& GetMemoryPoolAllocator() NN_NOEXCEPT;

//!< 空のノードインスタンスを構築します。
Node* ConstructEmptyNodeInstance() NN_NOEXCEPT;

//!< ノードを構築します。
NodeType* ConstructNode(::rapidjson::Type type) NN_NOEXCEPT;

//!< ノードを読み書き可能なものとして構築します。
NodeType* ConstructWritableNode(::rapidjson::Value& nodeValue) NN_NOEXCEPT;

//!< ノードを読み取り専用なものとして構築します。
NodeType* ConstructConstantNode(
    const ::rapidjson::Value& nodeValue) NN_NOEXCEPT;

//!< ノードを破棄します。
void DestructNode(NodeType* pNode) NN_NOEXCEPT;

//!< 読み書き可能な JSON 値を返します。
::rapidjson::Value& GetWritableValue(NodeType* pNode) NN_NOEXCEPT;

//!< 読み取り専用の JSON 値を返します。
const ::rapidjson::Value& GetConstantValue(const NodeType* pNode) NN_NOEXCEPT;

//!< ノードの型の名前を返します。
const ::std::string& GetNodeTypeName(
    const ::rapidjson::Value& nodeValue) NN_NOEXCEPT;

//!< 配列ノードであることを保証します。
bool EnsureArrayNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT;

//!< 真理値ノードであることを保証します。
bool EnsureBooleanNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT;

//!< 浮動小数点数ノードであることを保証します。
bool EnsureDoubleNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT;

//!< 整数ノードであることを保証します。
bool EnsureIntegerNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT;

//!< オブジェクトノードであることを保証します。
bool EnsureObjectNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT;

//!< 文字列ノードであることを保証します。
bool EnsureStringNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT;

} // namespace

Node::Node() NN_NOEXCEPT
{
    m_pImpl = ConstructNode(::rapidjson::kNullType);
}

Node::Node(Node&& other) NN_NOEXCEPT
{
    m_pImpl = other.m_pImpl;
    other.m_pImpl = nullptr;
}

Node::~Node() NN_NOEXCEPT
{
    DestructNode(reinterpret_cast<NodeType*>(m_pImpl));
}

Node& Node::operator=(Node&& other) NN_NOEXCEPT
{
    Node node(::std::move(other));
    ::std::swap(m_pImpl, node.m_pImpl);
    return *this;
}

const ::std::string& Node::GetTypeName() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    return GetNodeTypeName(
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl)));
}

Node Node::CreateArrayNode() NN_NOEXCEPT
{
    ::std::unique_ptr<Node> pNode(ConstructEmptyNodeInstance());
    pNode->m_pImpl = ConstructNode(::rapidjson::kArrayType);
    return ::std::move(*pNode);
}

bool Node::GetElementCount(size_t* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));
    COMMAND_DO(EnsureArrayNode(nodeValue));
    *pOutValue = static_cast<size_t>(nodeValue.Size());
    return true;
}

bool Node::GetElement(
    ::std::unique_ptr<const Node>* pOutValue, size_t index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);

    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));

    COMMAND_DO(EnsureArrayNode(nodeValue));

    const auto count = static_cast<size_t>(nodeValue.Size());

    if (count <= index)
    {
        PrintErrorCode(
            ErrorCode::NodeTypeShortArray, NameScope().Get(), count, index);

        return false;
    }

    ::std::unique_ptr<Node> pNode(ConstructEmptyNodeInstance());
    pNode->m_pImpl = ConstructConstantNode(
        nodeValue[static_cast<::rapidjson::SizeType>(index)]);

    *pOutValue = ::std::move(pNode);

    return true;
}

bool Node::AppendElement(Node value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);

    ::rapidjson::Value& nodeValue =
          GetWritableValue(reinterpret_cast<NodeType*>(m_pImpl));

    COMMAND_DO(EnsureArrayNode(nodeValue));

    ::rapidjson::Value& element =
          GetWritableValue(reinterpret_cast<NodeType*>(value.m_pImpl));

    nodeValue.PushBack(::std::move(element), GetMemoryPoolAllocator());

    return true;
}

Node Node::CreateBooleanNode() NN_NOEXCEPT
{
    ::std::unique_ptr<Node> pNode(ConstructEmptyNodeInstance());
    pNode->m_pImpl = ConstructNode(::rapidjson::kFalseType);
    return ::std::move(*pNode);
}

bool Node::IsBooleanNode() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));
    return nodeValue.IsBool();
}

bool Node::GetValue(bool* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));
    COMMAND_DO(EnsureBooleanNode(nodeValue));
    *pOutValue = nodeValue.GetBool();
    return true;
}

bool Node::SetValue(bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    ::rapidjson::Value& nodeValue =
        GetWritableValue(reinterpret_cast<NodeType*>(m_pImpl));
    COMMAND_DO(EnsureBooleanNode(nodeValue));
    nodeValue.SetBool(value);
    return true;
}

Node Node::CreateDoubleNode() NN_NOEXCEPT
{
    ::std::unique_ptr<Node> pNode(ConstructEmptyNodeInstance());
    pNode->m_pImpl = ConstructNode(::rapidjson::kNumberType);
    ::rapidjson::Value& nodeValue =
        GetWritableValue(reinterpret_cast<NodeType*>(pNode->m_pImpl));
    nodeValue.SetDouble(0);
    return ::std::move(*pNode);
}

bool Node::IsDoubleNode() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));
    return nodeValue.IsDouble();
}

bool Node::GetValue(double* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));
    COMMAND_DO(EnsureDoubleNode(nodeValue));
    *pOutValue = nodeValue.GetDouble();
    return true;
}

bool Node::SetValue(double value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    ::rapidjson::Value& nodeValue =
        GetWritableValue(reinterpret_cast<NodeType*>(m_pImpl));
    COMMAND_DO(EnsureDoubleNode(nodeValue));
    nodeValue.SetDouble(value);
    return true;
}

Node Node::CreateIntegerNode() NN_NOEXCEPT
{
    ::std::unique_ptr<Node> pNode(ConstructEmptyNodeInstance());
    pNode->m_pImpl = ConstructNode(::rapidjson::kNumberType);
    ::rapidjson::Value& nodeValue =
        GetWritableValue(reinterpret_cast<NodeType*>(pNode->m_pImpl));
    nodeValue.SetInt(0);
    return ::std::move(*pNode);
}

bool Node::IsIntegerNode() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));
    return (nodeValue.IsNumber() && !nodeValue.IsDouble());
}

bool Node::GetValue(int32_t* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);

    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));

    COMMAND_DO(EnsureIntegerNode(nodeValue));

    const double value = nodeValue.GetDouble();

    if (value < static_cast<double>(::std::numeric_limits<int32_t>::min()) ||
        value > static_cast<double>(::std::numeric_limits<int32_t>::max()))
    {
        ::std::ostringstream stream;
        stream.precision(0);
        stream << value;

        PrintErrorCode(
            ErrorCode::NodeValueOverLimitation, NameScope().Get(),
            stream.str());

        return false;
    }

    *pOutValue = static_cast<int32_t>(value);

    return true;
}

bool Node::SetValue(int32_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    ::rapidjson::Value& nodeValue =
        GetWritableValue(reinterpret_cast<NodeType*>(m_pImpl));
    COMMAND_DO(EnsureIntegerNode(nodeValue));
    nodeValue.SetInt(value);
    return true;
}

Node Node::CreateObjectNode() NN_NOEXCEPT
{
    ::std::unique_ptr<Node> pNode(ConstructEmptyNodeInstance());
    pNode->m_pImpl = ConstructNode(::rapidjson::kObjectType);
    return ::std::move(*pNode);
}

bool Node::GetKeys(::std::vector<::std::string>* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);

    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));

    COMMAND_DO(EnsureObjectNode(nodeValue));

    pOutValue->clear();

    ::rapidjson::Value::ConstMemberIterator itr = nodeValue.MemberBegin();

    while (itr != nodeValue.MemberEnd())
    {
        pOutValue->push_back(itr->name.GetString());

        ++itr;
    }

    return true;
}

bool Node::GetMember(
    ::std::unique_ptr<const Node>* pOutValue, const ::std::string& key
    ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);

    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));

    COMMAND_DO(EnsureObjectNode(nodeValue));

    ::rapidjson::Value::ConstMemberIterator itr = nodeValue.FindMember(key);

    if (itr != nodeValue.MemberEnd())
    {
        ::std::unique_ptr<Node> pNode(ConstructEmptyNodeInstance());
        pNode->m_pImpl = ConstructConstantNode(itr->value);

        *pOutValue = ::std::move(pNode);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeKeyMissing, NameScope(key).Get());

    return false;
}

bool Node::AppendMember(const ::std::string& key, Node value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);

    ::rapidjson::Value& nodeValue =
          GetWritableValue(reinterpret_cast<NodeType*>(m_pImpl));

    COMMAND_DO(EnsureObjectNode(nodeValue));

    ::rapidjson::Value& member =
          GetWritableValue(reinterpret_cast<NodeType*>(value.m_pImpl));

    nodeValue.AddMember(
        ::rapidjson::Value(key, GetMemoryPoolAllocator()), ::std::move(member),
        GetMemoryPoolAllocator());

    return true;
}

Node Node::CreateStringNode() NN_NOEXCEPT
{
    ::std::unique_ptr<Node> pNode(ConstructEmptyNodeInstance());
    pNode->m_pImpl = ConstructNode(::rapidjson::kStringType);
    return ::std::move(*pNode);
}

bool Node::IsStringNode() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));
    return nodeValue.IsString();
}

bool Node::GetValue(::std::string* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    const ::rapidjson::Value& nodeValue =
        GetConstantValue(reinterpret_cast<NodeType*>(m_pImpl));
    COMMAND_DO(EnsureStringNode(nodeValue));
    *pOutValue =
        ::std::string(nodeValue.GetString(), nodeValue.GetStringLength());
    return true;
}

bool Node::SetValue(const char* value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(value);
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    ::rapidjson::Value& nodeValue =
        GetWritableValue(reinterpret_cast<NodeType*>(m_pImpl));
    COMMAND_DO(EnsureStringNode(nodeValue));
    nodeValue.SetString(value, GetMemoryPoolAllocator());
    return true;
}

bool Node::SetValue(const ::std::string& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);
    ::rapidjson::Value& nodeValue =
        GetWritableValue(reinterpret_cast<NodeType*>(m_pImpl));
    COMMAND_DO(EnsureStringNode(nodeValue));
    nodeValue.SetString(value, GetMemoryPoolAllocator());
    return true;
}

Document::Document() NN_NOEXCEPT
{
    m_pImpl = new ::rapidjson::Document(
        ::rapidjson::kObjectType, &GetMemoryPoolAllocator());
}

Document::Document(Document&& other) NN_NOEXCEPT
{
    m_pImpl = other.m_pImpl;
    other.m_pImpl = nullptr;
}

Document::~Document() NN_NOEXCEPT
{
    if (m_pImpl != nullptr)
    {
        delete reinterpret_cast<::rapidjson::Document*>(m_pImpl);
    }
}

Document& Document::operator=(Document&& other) NN_NOEXCEPT
{
    Document document(::std::move(other));
    ::std::swap(m_pImpl, document.m_pImpl);
    return *this;
}

bool Document::Load(const ::std::string& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pImpl);

    ::rapidjson::Document& document =
          *reinterpret_cast<::rapidjson::Document*>(m_pImpl);

    ::rapidjson::StringStream stringStream(value.c_str());

    ::rapidjson::EncodedInputStream<
        ::rapidjson::UTF8<>,
        ::rapidjson::StringStream> inputStream(stringStream);

    document.ParseStream<::rapidjson::kParseCommentsFlag>(inputStream);

    if (document.HasParseError())
    {
        PrintErrorCode(
            ErrorCode::JsonParsingFailure,
            document.GetErrorOffset(),
            ::rapidjson::GetParseError_En(document.GetParseError()));

        return false;
    }

    return true;
}

bool Document::Save(::std::string* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::rapidjson::StringBuffer stringBuffer(&GetRuntimeMemoryAllocator());

    ::rapidjson::EncodedOutputStream<
        ::rapidjson::UTF8<>,
        ::rapidjson::StringBuffer> outputStream(stringBuffer);

    ::rapidjson::PrettyWriter<
        ::rapidjson::EncodedOutputStream<
            ::rapidjson::UTF8<>,
            ::rapidjson::StringBuffer>> writer(outputStream);

    const ::rapidjson::Document& document =
          *reinterpret_cast<::rapidjson::Document*>(m_pImpl);

    document.Accept(writer);

    *pOutValue =
        ::std::string(stringBuffer.GetString(), stringBuffer.GetSize());

    return true;
}

Node Document::GetRootNode() NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(Node) == sizeof(NodeType*));
    NN_STATIC_ASSERT(NN_ALIGNOF(Node) == NN_ALIGNOF(NodeType*));
    ::std::unique_ptr<Node> pNode(ConstructEmptyNodeInstance());
    NodeType* pRootNode = ConstructWritableNode(
        *reinterpret_cast<::rapidjson::Document*>(m_pImpl));
    ::std::memcpy(pNode.get(), &pRootNode, sizeof(pRootNode));
    return ::std::move(*pNode);
}

namespace {

const ::std::string NodeTypeName::Null = "null";
const ::std::string NodeTypeName::Boolean = "bool";
const ::std::string NodeTypeName::Object = "object";
const ::std::string NodeTypeName::Array = "array";
const ::std::string NodeTypeName::Integer = "integer";
const ::std::string NodeTypeName::Double = "double";
const ::std::string NodeTypeName::String = "string";
const ::std::string NodeTypeName::Unknown = "unknown";

::rapidjson::CrtAllocator& GetRuntimeMemoryAllocator() NN_NOEXCEPT
{
    static ::rapidjson::CrtAllocator s_MemoryAllocator;
    return s_MemoryAllocator;
}

::rapidjson::MemoryPoolAllocator<>& GetMemoryPoolAllocator() NN_NOEXCEPT
{
    static ::rapidjson::MemoryPoolAllocator<>
           s_MemoryAllocator(64 * 1024, &GetRuntimeMemoryAllocator());
    return s_MemoryAllocator;
}

Node* ConstructEmptyNodeInstance() NN_NOEXCEPT
{
    auto pStorage = new char[sizeof(Node) + NN_ALIGNOF(Node)]();
    ::nn::util::BytePtr bytePtr(pStorage);
    bytePtr.AlignUp(NN_ALIGNOF(Node));
    auto pNode = new Node(::std::move(*bytePtr.Get<Node>()));
    delete[] pStorage;
    return pNode;
}

NodeType* ConstructNode(::rapidjson::Type type) NN_NOEXCEPT
{
    auto pImpl = new NodeType();
    pImpl->hasOwnerShip = static_cast<int>(true);
    auto pValue = new ::rapidjson::Value(type);
    pImpl->pWritable = pValue;
    pImpl->pConstant = pValue;
    return pImpl;
}

NodeType* ConstructWritableNode(::rapidjson::Value& nodeValue) NN_NOEXCEPT
{
    auto pImpl = new NodeType();
    pImpl->hasOwnerShip = static_cast<int>(false);
    pImpl->pWritable = &nodeValue;
    pImpl->pConstant = &nodeValue;
    return pImpl;
}

NodeType* ConstructConstantNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT
{
    auto pImpl = new NodeType();
    pImpl->hasOwnerShip = static_cast<int>(false);
    pImpl->pWritable = nullptr;
    pImpl->pConstant = &nodeValue;
    return pImpl;
}

void DestructNode(NodeType* pNode) NN_NOEXCEPT
{
    if (pNode != nullptr)
    {
        if (pNode->hasOwnerShip && pNode->pWritable != nullptr)
        {
            delete pNode->pWritable;
        }

        delete pNode;
    }
}

::rapidjson::Value& GetWritableValue(NodeType* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);
    NN_SDK_REQUIRES_NOT_NULL(pNode->pWritable);
    return *pNode->pWritable;
}

const ::rapidjson::Value& GetConstantValue(const NodeType* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);
    NN_SDK_REQUIRES_NOT_NULL(pNode->pConstant);
    return *pNode->pConstant;
}

const ::std::string& GetNodeTypeName(
    const ::rapidjson::Value& nodeValue) NN_NOEXCEPT
{
    if (nodeValue.IsNull())
    {
        return NodeTypeName::Null;
    }

    if (nodeValue.IsBool())
    {
        return NodeTypeName::Boolean;
    }

    if (nodeValue.IsObject())
    {
        return NodeTypeName::Object;
    }

    if (nodeValue.IsArray())
    {
        return NodeTypeName::Array;
    }

    if (nodeValue.IsNumber() && !nodeValue.IsDouble())
    {
        return NodeTypeName::Integer;
    }

    if (nodeValue.IsDouble())
    {
        return NodeTypeName::Double;
    }

    if (nodeValue.IsString())
    {
        return NodeTypeName::String;
    }

    return NodeTypeName::Unknown;
}

bool EnsureArrayNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT
{
    if (nodeValue.IsArray())
    {
        return true;
    }

    const char* const name = GetNodeTypeName(nodeValue).c_str();

    PrintErrorCode(
        ErrorCode::NodeTypeMismatch, NameScope().Get(), "array", name);

    return false;
}

bool EnsureBooleanNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT
{
    if (nodeValue.IsBool())
    {
        return true;
    }

    const char* const name = GetNodeTypeName(nodeValue).c_str();

    PrintErrorCode(
        ErrorCode::NodeTypeMismatch, NameScope().Get(), "boolean", name);

    return false;
}

bool EnsureDoubleNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT
{
    if (nodeValue.IsDouble())
    {
        return true;
    }

    const char* const name = GetNodeTypeName(nodeValue).c_str();

    PrintErrorCode(
        ErrorCode::NodeTypeMismatch, NameScope().Get(), "double", name);

    return false;
}

bool EnsureIntegerNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT
{
    if (nodeValue.IsNumber() && !nodeValue.IsDouble())
    {
        return true;
    }

    const char* const name = GetNodeTypeName(nodeValue).c_str();

    PrintErrorCode(
        ErrorCode::NodeTypeMismatch, NameScope().Get(), "integer", name);

    return false;
}

bool EnsureObjectNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT
{
    if (nodeValue.IsObject())
    {
        return true;
    }

    const char* const name = GetNodeTypeName(nodeValue).c_str();

    PrintErrorCode(
        ErrorCode::NodeTypeMismatch, NameScope().Get(), "object", name);

    return false;
}

bool EnsureStringNode(const ::rapidjson::Value& nodeValue) NN_NOEXCEPT
{
    if (nodeValue.IsString())
    {
        return true;
    }

    const char* const name = GetNodeTypeName(nodeValue).c_str();

    PrintErrorCode(
        ErrorCode::NodeTypeMismatch, NameScope().Get(), "string", name);

    return false;
}

} // namespace
