﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_PushNotification.h>

#include "SettingsManager_NameScope.h"
#include "SettingsManager_PushNotification.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< プッシュ通知設定のキー
const char* const SettingNamePushNotificationSettings =
    "push_notification_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< スリープ状態における動作モードのキー
const char* const KeyActivityModeOnSleep = "activity_mode_on_sleep";

//!< スリープ状態における動作モードをエクスポートします。
bool ExportActivityModeOnSleep(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyActivityModeOnSleep);

        auto currentValueNode = Node::CreateIntegerNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::
                    GetPushNotificationActivityModeOnSleep()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateIntegerNode();

        COMMAND_DO(defaultValueNode.SetValue(0));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(
                   KeyActivityModeOnSleep, ::std::move(node)));

    return true;
}

//!< スリープ状態における動作モードインポートします。
bool ImportActivityModeOnSleep(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyActivityModeOnSleep));

    NameScope nameScope(KeyActivityModeOnSleep);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto activityModeOnSleep = int32_t();
    COMMAND_DO(pCurrentValueNode->GetValue(&activityModeOnSleep));

    ::nn::settings::system::SetPushNotificationActivityModeOnSleep(
        activityModeOnSleep);

    return true;
}

} // namespace

bool IsSettingNamePushNotificationSettings(
    const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNamePushNotificationSettings);
}

bool ExportPushNotificationSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNamePushNotificationSettings);

        COMMAND_DO(ExportActivityModeOnSleep(&node));
    }

    COMMAND_DO(pNode->AppendMember(
                   SettingNamePushNotificationSettings, ::std::move(node)));

    return true;
}

bool ImportPushNotificationSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNamePushNotificationSettings);

    COMMAND_DO(ImportActivityModeOnSleep(node));

    return true;
}
