﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/settings/system/settings_OnlineStorage.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_UuidTypes.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_OnlineStorage.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< アカウント毎のオンラインストレージ設定のキー
const char* const SettingNameAccountOnlineStorageSettings =
    "account_online_storage_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< アカウントの識別子のキー
const char* const KeyAccountUid = "account_uid";

//!< 起動時に新規セーブデータを確認するか否かを表す値のキー
const char* const KeySaveDataCheckAtLaunchFlag =
    "savedata_check_at_launch_flag";

//!< アカウントの識別子をエクスポートします。
bool ExportAccountUid(
    Node* pDictNode,
    const ::nn::settings::system::AccountOnlineStorageSettings& settings
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyAccountUid);

        auto currentValueNode = Node::CreateStringNode();

        ::std::string value;
        COMMAND_DO(EncodeUuid(&value, ConvertToUuid(settings.uid)));

        COMMAND_DO(currentValueNode.SetValue(value));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        ::nn::util::Uuid uuid = {};
        COMMAND_DO(EncodeUuid(&value, uuid));

        COMMAND_DO(defaultValueNode.SetValue(value));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyAccountUid, ::std::move(node)));

    return true;
}

//!< アカウントの識別子をインポートします。
bool ImportAccountUid(
    ::nn::settings::system::AccountOnlineStorageSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyAccountUid));

    NameScope nameScope(KeyAccountUid);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string value;
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    ::nn::util::Uuid uuid = {};

    if (!DecodeUuid(&uuid, value))
    {
        PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), value);

        return false;
    }

    pOutValue->uid = ConvertToUid(uuid);

    return true;
}

//!< フレンドのオンライン通知を行うか否かを表す値をエクスポートします。
bool ExportSaveDataCheckAtLaunchFlag(
    Node* pDictNode,
    const ::nn::settings::system::AccountOnlineStorageSettings& settings
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeySaveDataCheckAtLaunchFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::AccountOnlineStorageFlag::
                        SaveDataCheckAtLaunch>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeySaveDataCheckAtLaunchFlag, ::std::move(node)));

    return true;
}

//!< フレンドのオンライン通知を行うか否かを表す値をインポートします。
bool ImportSaveDataCheckAtLaunchFlag(
    ::nn::settings::system::AccountOnlineStorageSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySaveDataCheckAtLaunchFlag));

    NameScope nameScope(KeySaveDataCheckAtLaunchFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->flags.Set<
        ::nn::settings::system::AccountOnlineStorageFlag::
            SaveDataCheckAtLaunch>(value);

    return true;
}

} // namespace

bool IsSettingNameAccountOnlineStorageSettings(
    const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameAccountOnlineStorageSettings);
}

bool ExportAccountOnlineStorageSettings(Node* pNode) NN_NOEXCEPT
{
    const int EntryCountMax =
        ::nn::settings::system::AccountOnlineStorageSettingsCountMax;

    ::std::unique_ptr<::nn::settings::system::AccountOnlineStorageSettings[]
          > entries(
              new ::nn::settings::system::AccountOnlineStorageSettings[
                  EntryCountMax]());

    const int count =
        ::nn::settings::system::GetAccountOnlineStorageSettings(
            entries.get(), EntryCountMax);

    auto arrayNode = Node::CreateArrayNode();

    for (int i = 0; i < count; ++i)
    {
        NameScope nameScope(SettingNameAccountOnlineStorageSettings);

        const ::nn::settings::system::AccountOnlineStorageSettings&
            settings = entries[i];

        auto dictNode = Node::CreateObjectNode();

        COMMAND_DO(ExportAccountUid(&dictNode, settings));

        COMMAND_DO(ExportSaveDataCheckAtLaunchFlag(&dictNode, settings));

        COMMAND_DO(arrayNode.AppendElement(::std::move(dictNode)));
    }

    COMMAND_DO(
        pNode->AppendMember(
            SettingNameAccountOnlineStorageSettings, ::std::move(arrayNode)));

    return true;
}

bool ImportAccountOnlineStorageSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameAccountOnlineStorageSettings);

    const int EntryCountMax =
        ::nn::settings::system::AccountOnlineStorageSettingsCountMax;

    ::std::unique_ptr<::nn::settings::system::AccountOnlineStorageSettings[]
          > entries(
              new ::nn::settings::system::AccountOnlineStorageSettings[
                  EntryCountMax]());

    int count = 0;

    auto elementCount = size_t();

    COMMAND_DO(node.GetElementCount(&elementCount));

    for (size_t i = 0; i < elementCount; ++i)
    {
        if (count >= EntryCountMax)
        {
            break;
        }

        ::nn::settings::system::AccountOnlineStorageSettings&
              settings = entries[count];

        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(node.GetElement(&pNode, i));

        COMMAND_DO(ImportAccountUid(&settings, *pNode));

        COMMAND_DO(ImportSaveDataCheckAtLaunchFlag(&settings, *pNode));

        ++count;
    }

    ::nn::settings::system::SetAccountOnlineStorageSettings(
        entries.get(), count);

    return true;
}
