﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_FileSystem.h"
#include "SettingsManager_Utility.h"

namespace {

//!< ホストファイルシステムのマウント名を返します。
::std::string GetHostFsMountName() NN_NOEXCEPT
{
    return "HostFs";
}

} // namespace

void* AllocateForFileSystem(size_t size) NN_NOEXCEPT
{
    return ::std::malloc(size);
}

void DeallocateForFileSystem(void* addr, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    ::std::free(addr);
}

HostFsFile::HostFsFile() NN_NOEXCEPT
    : m_IsOpen(false)
    , m_MountName()
    , m_Path()
    , m_Handle()
{
    // 何もしない
}

HostFsFile::~HostFsFile() NN_NOEXCEPT
{
    this->Close();
}

const ::std::string& HostFsFile::GetPath() const NN_NOEXCEPT
{
    return m_Path;
}

::nn::fs::FileHandle HostFsFile::GetHandle() const NN_NOEXCEPT
{
    return m_Handle;
}

::nn::Result HostFsFile::Open(const ::std::string& path, int mode) NN_NOEXCEPT
{
    // オープン済みだった場合に備えてクローズ
    this->Close();

    ::std::string safePath = path;

    ::std::replace(safePath.begin(), safePath.end(), '/', '\\');

    const size_t index = safePath.find_last_of('\\');

    const bool hasFileName =
        ((0 < safePath.size()) && (index < safePath.size() - 1));

    const ::std::string rootPath =
        hasFileName ? safePath.substr(0, index + 1) : safePath;

    const ::std::string mountName = GetHostFsMountName();

    auto needsRollback = true;

    NN_RESULT_DO(::nn::fs::MountHost(mountName.c_str(), rootPath.c_str()));
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            ::nn::fs::Unmount(mountName.c_str());
        }
    };

    const ::std::string filePath =
        mountName + ":/" + (hasFileName ? safePath.substr(index + 1) : "");

    if ((mode & ::nn::fs::OpenMode_Write) != 0)
    {
        const ::nn::Result result = ::nn::fs::CreateFile(filePath.c_str(), 0);

        NN_RESULT_THROW_UNLESS(
            result.IsSuccess() ||
            ::nn::fs::ResultPathAlreadyExists::Includes(result),
            result);
    }

    NN_RESULT_DO(::nn::fs::OpenFile(&m_Handle, filePath.c_str(), mode));

    needsRollback = false;

    m_IsOpen = true;

    m_MountName = mountName;

    m_Path = safePath;

    NN_RESULT_SUCCESS;
}

void HostFsFile::Close() NN_NOEXCEPT
{
    if (m_IsOpen)
    {
        ::nn::fs::CloseFile(m_Handle);

        ::nn::fs::Unmount(m_MountName.c_str());

        m_IsOpen = false;

        m_MountName.clear();

        m_Path.clear();

        m_Handle = ::nn::fs::FileHandle();
    }
}

bool OpenFile(IFile* pFile, const ::std::string& path, int mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFile);

    const ::nn::Result result = pFile->Open(path.c_str(), mode);

    if (result.IsFailure())
    {
        PrintErrorCode(ErrorCode::FileOpenFailure, result, path);
    }

    return result.IsSuccess();
}

bool SetFileSize(const IFile& file, int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(size, 0);

    const ::nn::Result result = ::nn::fs::SetFileSize(file.GetHandle(), size);

    if (result.IsFailure())
    {
        PrintErrorCode(
            ErrorCode::FileSizeChangeFailure, result, file.GetPath());
    }

    return result.IsSuccess();
}

bool GetFileSize(int64_t* pOutValue, const IFile& file) NN_NOEXCEPT
{
    const ::nn::Result result =
        ::nn::fs::GetFileSize(pOutValue, file.GetHandle());

    if (result.IsFailure())
    {
        PrintErrorCode(
            ErrorCode::FileSizeAcquisitionFailure, result, file.GetPath());
    }

    return result.IsSuccess();
}

bool ReadFile(
    size_t* pOutSize, char* buffer, size_t bufferSize,
    const IFile& file, int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    int64_t fileSize = 0;

    COMMAND_DO(GetFileSize(&fileSize, file));

    if (fileSize == 0)
    {
        *pOutSize = 0;
    }
    else
    {
        const ::nn::Result result =
            ::nn::fs::ReadFile(
                pOutSize, file.GetHandle(), offset, buffer, bufferSize);

        if (result.IsFailure())
        {
            PrintErrorCode(
                ErrorCode::FileReadingFailure, result, file.GetPath());

            return false;
        }
    }

    return true;
}

bool WriteFile(
    const IFile& file, int64_t offset,
    const char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    const ::nn::Result result = ::nn::fs::WriteFile(
        file.GetHandle(), offset, buffer, bufferSize, ::nn::fs::WriteOption());

    if (result.IsFailure())
    {
        PrintErrorCode(ErrorCode::FileWritingFailure, result, file.GetPath());
    }

    return result.IsSuccess();
}

bool FlushFile(const IFile& file) NN_NOEXCEPT
{
    const ::nn::Result result = ::nn::fs::FlushFile(file.GetHandle());

    if (result.IsFailure())
    {
        PrintErrorCode(ErrorCode::FileFlushingFailure, result, file.GetPath());
    }

    return result.IsSuccess();
}
