﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <string>
#include <utility>
#include <vector>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Backlight.h>

#include "SettingsManager_Backlight.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< 設定情報ヘッダのキー
const char* const SettingNameBacklightSettings = "backlight_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 自動輝度調整を行うか否かを表すフラグのキー
const char* const KeyAutomationFlag = "automation_flag";

//!< 輝度設定値のキー
const char* const KeyBrightness = "brightness";

//!< 輝度設定値 (VR モード用) のキー
const char* const KeyBrightnessVr = "vr_brightness";

//!< 自動輝度調整を行うか否かを表すフラグをエクスポートします。
bool ExportAutomationFlag(
    Node* pDictNode,
    const ::nn::settings::system::BacklightSettingsEx& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyAutomationFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::BacklightFlag::Automatic>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyAutomationFlag, ::std::move(node)));

    return true;
}

//!< 自動輝度調整を行うか否かを表すフラグをインポートします。
bool ImportAutomationFlag(
    ::nn::settings::system::BacklightSettingsEx* pOutValue,
    const Node& dictNode,
    const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const char* const key = KeyAutomationFlag;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->flags.Set<
        ::nn::settings::system::BacklightFlag::Automatic>(value);

    return true;
}

//!< 輝度設定値をエクスポートします。
bool ExportBrightness(
    Node* pDictNode,
    const ::nn::settings::system::BacklightSettingsEx& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyBrightness);

        auto currentValueNode = Node::CreateDoubleNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                static_cast<double>(
                    static_cast<int>(settings.brightness * 1000)) / 1000));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateDoubleNode();

        COMMAND_DO(defaultValueNode.SetValue(0.624));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyBrightness, ::std::move(node)));

    return true;
}

//!< 輝度設定値 (VR モード用) をエクスポートします。
bool ExportBrightnessVr(
    Node* pDictNode,
    const ::nn::settings::system::BacklightSettingsEx& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyBrightnessVr);

        auto currentValueNode = Node::CreateDoubleNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                static_cast<double>(
                    static_cast<int>(settings.brightnessVr * 1000)) / 1000));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateDoubleNode();

        COMMAND_DO(defaultValueNode.SetValue(0.843));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyBrightnessVr, ::std::move(node)));

    return true;
}

//!< 輝度設定値をインポートします。
bool ImportBrightness(
    ::nn::settings::system::BacklightSettingsEx* pOutValue,
    const Node& dictNode,
    const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const char* const key = KeyBrightness;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = double();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->brightness =
           static_cast<float>(static_cast<int>(value * 1000)) / 1000;

    return true;
}

//!< 輝度設定値 (VR モード用) をインポートします。
bool ImportBrightnessVr(
    ::nn::settings::system::BacklightSettingsEx* pOutValue,
    const Node& dictNode,
    const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const char* const key = KeyBrightnessVr;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = double();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->brightnessVr =
           static_cast<float>(static_cast<int>(value * 1000)) / 1000;

    return true;
}

} // namespace

bool IsSettingNameBacklightSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameBacklightSettings);
}

bool ExportBacklightSettings(Node* pNode) NN_NOEXCEPT
{
    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameBacklightSettings);

        ::nn::settings::system::BacklightSettingsEx settings = {};

        ::nn::settings::system::GetBacklightSettings(&settings);

        COMMAND_DO(ExportAutomationFlag(&node, settings));

        COMMAND_DO(ExportBrightness(&node, settings));

        COMMAND_DO(ExportBrightnessVr(&node, settings));
    }

    COMMAND_DO(
        pNode->AppendMember(
            SettingNameBacklightSettings, ::std::move(node)));

    return true;
}

bool ImportBacklightSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameBacklightSettings);

    ::std::vector<::std::string> keys;

    COMMAND_DO(node.GetKeys(&keys));

    ::nn::settings::system::BacklightSettingsEx settings = {};

    ::nn::settings::system::GetBacklightSettings(&settings);

    COMMAND_DO(ImportAutomationFlag(&settings, node, keys));

    COMMAND_DO(ImportBrightness(&settings, node, keys));

    COMMAND_DO(ImportBrightnessVr(&settings, node, keys));

    ::nn::settings::system::SetBacklightSettings(settings);

    return true;
}
