﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cctype>
#include <sstream>
#include <string>

#include <nn/nn_FirmwareVersion.h>
#include <nn/crypto.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/fs.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/erpt.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/erpt/erpt_Report.h>
#include <nn/erpt/erpt_TypesPrivate.h>
#include <nn/erpt/server/erpt_Keys.h>
#include <nn/erpt/server/erpt_ServerTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "../DevMenuCommand_Log.h"
#include "DevMenuCommand_ErrorExport.h"
#include "DevMenuCommand_ErrorDump.h"

#if defined(_USE_MATH_DEFINES)
// DeMenu が _USE_MATH_DEFINES を定義するため、nn/msgpack.h の include によって DevMenuCommand in DevMenu ビルド時に重複定義になってしまうことの W/A
#undef _USE_MATH_DEFINES
#endif

#include <nn/msgpack.h>
#ifdef NN_BUILD_CONFIG_OS_WIN
#include <nn/nn_Windows.h> // msgpack.h が Windows.h を include していることに対するワークアラウンド
#endif

using namespace nn;

namespace errors {

namespace
{
    struct ErrorReportBinaryHeader
    {
        Bit32 magicNumber;
        Bit8 version;
        Bit8 flags;
        Bit8 reserved[2]; // padding 対策
        struct
        {
            Bit16 reportCount;
            Bit8 aes128Key[16];
            Bit8 aes128Iv[16];
            ToolVersion toolVersion; // Export に使用したツールのバージョン(5byte)
            Bit8 reserved[151];  // 256 - HashSize(=32) * 2 + 2 = 190 byte までデータを持てるのでギリギリまで確保しておく
            Bit8 unavailable[66];
        }
        rsa;
        NN_STATIC_ASSERT(sizeof(ErrorReportBinaryHeader::rsa) == 256);
    };

    const Bit32 ErrorReportBinaryHeaderMagicNumber = 0x11265963;
    const Bit8 ErrorReportBinaryHeaderVersion = 2;
    const size_t ErrorReportBinaryAlignedBytes = 4;
    const size_t ErrorReportBinaryRsaKeyBytes = 2048 / 8;

    struct ErrorReportBinaryBlockHeader
    {
        Bit8 reportId[erpt::ReportIdLength];
        Bit32 reportSize;
        Bit32 blockSize;
    };

    // ErrorReportBinaryHeaderVersion = 2 以降
    struct ErrorReportBinaryBlockHeader2
    {
        Bit8 reportId[erpt::ReportIdLength];
        Bit32 reportSize;
        Bit32 blockSize;
        Bit8 metaData[erpt::ReportMetaDataLength];
    };

    // 公開鍵成分
    static const uint8_t g_ErrorReportBinaryRsaPublicModulus[] =
    {
        0xe0, 0x6a, 0x52, 0x7f, 0x4c, 0xe0, 0xee, 0x4a, 0x4f, 0x66, 0xcc, 0xe6, 0x7b, 0x23, 0xaa, 0x3b,
        0x89, 0xb9, 0x65, 0xec, 0xc1, 0x5d, 0xac, 0xaa, 0x1e, 0x79, 0x8c, 0x69, 0x22, 0x87, 0x74, 0x01,
        0x14, 0xcb, 0xf9, 0x2d, 0x8d, 0x41, 0x60, 0xbf, 0x1d, 0x98, 0x0e, 0x68, 0x9d, 0x69, 0x8f, 0x6d,
        0x5c, 0xb8, 0xa7, 0x1d, 0x07, 0x36, 0x92, 0xff, 0x29, 0x52, 0x1b, 0x83, 0x90, 0x08, 0xa6, 0x67,
        0x29, 0xdd, 0xe3, 0xbf, 0x4b, 0x9c, 0x9b, 0x25, 0x2e, 0x39, 0x90, 0x7c, 0xf3, 0x82, 0x95, 0x45,
        0x22, 0x94, 0xa6, 0x0f, 0x00, 0xe3, 0x72, 0xbf, 0x6a, 0x6c, 0x34, 0x02, 0xcc, 0x1b, 0xb3, 0x72,
        0xce, 0x03, 0xbb, 0x09, 0x75, 0x4b, 0xa3, 0x4a, 0x0b, 0xa1, 0x1c, 0x13, 0xcc, 0x76, 0x91, 0x33,
        0x3e, 0x36, 0x96, 0xfd, 0xf0, 0xf1, 0xeb, 0xe5, 0x76, 0x8d, 0x0a, 0x48, 0xb9, 0x0a, 0xcd, 0xcf,
        0x31, 0xd8, 0xae, 0x87, 0x71, 0x53, 0x8c, 0xe3, 0x17, 0x67, 0x30, 0x3e, 0x15, 0x2b, 0x51, 0xe1,
        0xd4, 0xba, 0x80, 0x3c, 0xf0, 0x6f, 0xe5, 0x53, 0x31, 0x3d, 0xa6, 0x80, 0xc8, 0x70, 0xf4, 0x5d,
        0xcb, 0x2b, 0x1c, 0x58, 0xd5, 0x5a, 0x14, 0xc4, 0xa2, 0x35, 0x42, 0x5e, 0x04, 0xec, 0x81, 0xf6,
        0xda, 0x16, 0xeb, 0x2f, 0x9a, 0x23, 0xf7, 0xb5, 0xb3, 0xb1, 0xef, 0xa6, 0xa3, 0xac, 0x21, 0x6d,
        0xf6, 0xa7, 0xd8, 0xcc, 0xc6, 0x72, 0xde, 0x46, 0x2a, 0x3f, 0x91, 0xe1, 0x0a, 0x56, 0x05, 0x59,
        0x5e, 0x42, 0xe7, 0xa8, 0x24, 0x36, 0x04, 0x0a, 0xb5, 0xb8, 0x4e, 0x66, 0xdc, 0x1b, 0xf5, 0x33,
        0x41, 0xb5, 0xc1, 0x4a, 0xa5, 0x49, 0xcb, 0x59, 0x2b, 0xc3, 0x9b, 0x6b, 0x21, 0x68, 0x3f, 0x4f,
        0x37, 0x47, 0x71, 0x56, 0x23, 0x10, 0xee, 0xc9, 0xef, 0xaf, 0x3b, 0x50, 0xac, 0xaf, 0xae, 0x1f
    };

    static const uint8_t g_ErrorReportBinaryRsaPublicExponent[] =
    {
        0x01,0x00,0x01
    };

    inline size_t GetErrorReportBinaryAlignedSize(size_t size)
    {
        return size + ErrorReportBinaryAlignedBytes - (size % ErrorReportBinaryAlignedBytes);
    }

    Result GenerateErrorReportBinaryHeader(ErrorReportBinaryHeader* pHeader, Bit16 reportCount, Bit8 flags)
    {
        pHeader->magicNumber = ErrorReportBinaryHeaderMagicNumber;
        pHeader->version = ErrorReportBinaryHeaderVersion;
        pHeader->flags = flags;

        std::memset(&(pHeader->rsa), 0x0, sizeof(pHeader->rsa));
        pHeader->rsa.reportCount = reportCount;
        pHeader->rsa.toolVersion = errors::GetToolVersion();

        if( pHeader->flags & ErrorReportBinaryHeaderFlag_AesEncrypted )
        {
            os::GenerateRandomBytes(pHeader->rsa.aes128Key, sizeof(pHeader->rsa.aes128Key));
            os::GenerateRandomBytes(pHeader->rsa.aes128Iv, sizeof(pHeader->rsa.aes128Iv));
        }

        NN_RESULT_SUCCESS;
    }

    Result HidePrivateInfo(uint8_t* pReportData, int64_t reportSize)
    {
        nn::msgpack::MpWalker mpWalker;
        mpWalker.Init(pReportData, static_cast<size_t>(reportSize));

        const char* pStr;
        uint32_t size;

        const char* privateKeys[] = {
            "ApplicationTitle",
            "ApplicationVersion",
            "RunningApplicationTitle",
            "RunningApplicationVersion",
            "CurrentIPAddress",
            "SubnetMask",
            "GatewayIPAddress",
            "PriorityDNSIPAddress",
            "AlternateDNSIPAddress",
            "ProxyIPAddress",
            "GlobalIPAddress",
        };

        for( auto key : privateKeys )
        {
            pStr = nullptr;
            mpWalker.Find(key).GetString(&pStr, &size);
            if( pStr != nullptr )
            {
                std::memset(const_cast<char*>(pStr), '*', size);
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result WriteErrorReportBlock(fs::FileHandle handle,
        ErrorReportBinaryHeader& header, nn::erpt::ReportInfo& reportInfo, int64_t* pOffset, crypto::Aes128CtrEncryptor& encryptor)
    {
        nn::erpt::Report report;
        NN_RESULT_DO(report.Open(reportInfo.reportId));

        int64_t reportSize;
        NN_RESULT_DO(report.GetSize(&reportSize));

        auto bodySize = GetErrorReportBinaryAlignedSize(static_cast<size_t>(reportSize));
        auto blockSize = sizeof(ErrorReportBinaryBlockHeader2) + bodySize;

        std::unique_ptr<uint8_t[]> reportData(new uint8_t[static_cast<uint32_t>(blockSize)]);
        NN_RESULT_THROW_UNLESS(reportData != nullptr, nn::os::ResultOutOfMemory());
        std::memset(reportData.get(), 0x0, blockSize);

        ErrorReportBinaryBlockHeader2 *pBlockHeader = reinterpret_cast<ErrorReportBinaryBlockHeader2*>(reportData.get());
        std::memcpy(pBlockHeader->reportId, reportInfo.reportId.u.id, erpt::ReportIdLength);
        pBlockHeader->reportSize = static_cast<Bit32>(reportSize);
        pBlockHeader->blockSize = static_cast<Bit32>(blockSize);
        std::memcpy(pBlockHeader->metaData, reportInfo.reportMetaData.userData, sizeof(pBlockHeader->metaData));

        uint32_t readCount;
        NN_RESULT_DO(report.Read(&readCount, reportData.get() + sizeof(ErrorReportBinaryBlockHeader2), static_cast<uint32_t>(reportSize)));
        NN_RESULT_THROW_UNLESS(reportSize == readCount, nn::erpt::ResultOutOfArraySpace());

        // SDK 開発者に見せなくて良い情報を隠蔽する
        NN_RESULT_DO(HidePrivateInfo(reportData.get() + sizeof(ErrorReportBinaryBlockHeader2), reportSize));

        // AES CTR モードで暗号化
        if( header.flags & ErrorReportBinaryHeaderFlag_AesEncrypted )
        {
            std::unique_ptr<uint8_t[]> encryptedData(new uint8_t[static_cast<uint32_t>(blockSize)]);
            NN_RESULT_THROW_UNLESS(encryptedData != nullptr, nn::os::ResultOutOfMemory());

            encryptor.Update(encryptedData.get(), blockSize, reportData.get(), blockSize);

            NN_RESULT_DO(fs::WriteFile(handle, *pOffset, encryptedData.get(), blockSize, fs::WriteOption()));
        }
        else
        {
            NN_RESULT_DO(fs::WriteFile(handle, *pOffset, reportData.get(), blockSize, fs::WriteOption()));
        }
        NN_RESULT_DO(fs::FlushFile(handle));
        *pOffset += blockSize;

        NN_RESULT_SUCCESS;
    }

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)

    // 秘密鍵成分
    static const uint8_t g_ErrorReportBinaryRsaPrivateExponent[] =
    {
        0x80, 0xd9, 0xeb, 0x46, 0x36, 0x61, 0x0b, 0xd0, 0xd4, 0xe8, 0x44, 0x8b, 0xad, 0x76, 0x5b, 0xde,
        0xd5, 0x84, 0x02, 0xd6, 0xbe, 0x6b, 0xc7, 0xa0, 0xe1, 0xe0, 0x8e, 0x53, 0x2b, 0x45, 0x34, 0xc1,
        0x0e, 0x2c, 0xd3, 0x84, 0x69, 0x0f, 0x79, 0xba, 0x22, 0x06, 0xd5, 0x77, 0x11, 0xb2, 0xb3, 0x6c,
        0x6a, 0x89, 0x87, 0x42, 0x2e, 0x8b, 0x12, 0x39, 0xb9, 0x82, 0xf0, 0x53, 0x48, 0xd6, 0xfe, 0x9c,
        0x4b, 0x38, 0xa0, 0xf4, 0x8c, 0x38, 0x7e, 0x08, 0x0a, 0x13, 0x79, 0x37, 0x95, 0x25, 0x8c, 0x09,
        0x67, 0x02, 0x8c, 0x9a, 0x1e, 0xbb, 0x79, 0x7e, 0x18, 0x5f, 0xca, 0x39, 0x34, 0xf2, 0x4e, 0xc5,
        0x83, 0x2e, 0x37, 0x2d, 0x06, 0x11, 0x08, 0xcf, 0x91, 0xff, 0xce, 0x3c, 0x3b, 0xf0, 0x74, 0x0c,
        0x45, 0x13, 0x58, 0xfc, 0xb3, 0x64, 0x64, 0x3a, 0x9b, 0x27, 0xfe, 0x18, 0xa5, 0x4e, 0x63, 0xbe,
        0x19, 0x34, 0xb0, 0x5c, 0x05, 0xef, 0x59, 0x3d, 0xc7, 0x3e, 0x3c, 0x1b, 0x03, 0xc7, 0x16, 0x43,
        0x6f, 0x59, 0x80, 0xe0, 0xe0, 0x1b, 0x4d, 0x2b, 0x65, 0xa7, 0x15, 0x0d, 0x93, 0x3a, 0x6d, 0x10,
        0x62, 0xdb, 0x24, 0x99, 0x20, 0xe9, 0x37, 0x8c, 0x84, 0x27, 0x8f, 0xc5, 0xdb, 0xb4, 0xa7, 0xc7,
        0xf2, 0x7c, 0x80, 0xdd, 0x83, 0x33, 0x85, 0x63, 0x46, 0x38, 0xf9, 0xfb, 0xf8, 0xe6, 0x57, 0x5d,
        0x6c, 0x90, 0x87, 0xa0, 0x4e, 0x08, 0x6a, 0x32, 0x45, 0x5f, 0x78, 0x60, 0xf0, 0x5e, 0xc4, 0x4a,
        0xbf, 0xfa, 0xb0, 0xd2, 0xf8, 0x35, 0x71, 0xeb, 0x32, 0x7f, 0x44, 0x47, 0x97, 0xad, 0xb9, 0x4f,
        0x7b, 0xb3, 0xf9, 0xf1, 0xea, 0xf3, 0x7a, 0x3f, 0xbd, 0x2c, 0x09, 0xe1, 0x0a, 0xfe, 0x97, 0x0b,
        0x7c, 0x09, 0x70, 0x55, 0x6f, 0xd1, 0xcb, 0x69, 0x5d, 0x33, 0x97, 0xea, 0x01, 0xba, 0x22, 0x69
    };

    Result DecryptErrorReportBlock( void* pData, size_t dataSize, crypto::Aes128CtrDecryptor& decryptor )
    {
        std::unique_ptr<uint8_t[]> decryptedData(new uint8_t[static_cast<uint32_t>(dataSize)]);
        NN_RESULT_THROW_UNLESS(decryptedData != nullptr, nn::os::ResultOutOfMemory());

        decryptor.Update( decryptedData.get(), dataSize, pData, dataSize );
        std::memcpy( pData, decryptedData.get(), dataSize );

        NN_RESULT_SUCCESS;
    }

    Result DumpErrorReportBlock(fs::FileHandle handle, ErrorReportBinaryHeader& header,
        int64_t* pOffset, crypto::Aes128CtrDecryptor& decryptor)
    {
        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, handle));

        size_t blockHeaderSize = header.version <= 1 ? sizeof(ErrorReportBinaryBlockHeader) : sizeof(ErrorReportBinaryBlockHeader2);

        ErrorReportBinaryBlockHeader2 blockHeader;
        NN_RESULT_THROW_UNLESS(*pOffset + static_cast<int>(blockHeaderSize) <= fileSize, fs::ResultOutOfRange());
        NN_RESULT_DO(fs::ReadFile(handle, *pOffset, &blockHeader, blockHeaderSize));
        *pOffset += blockHeaderSize;

        // ブロックヘッダの復号化
        if( header.flags & ErrorReportBinaryHeaderFlag_AesEncrypted )
        {
            NN_RESULT_DO(DecryptErrorReportBlock(&blockHeader, blockHeaderSize, decryptor));
        }

        auto bodySize = blockHeader.blockSize - blockHeaderSize;
        NN_RESULT_THROW_UNLESS(*pOffset + static_cast<int>(bodySize) <= fileSize, fs::ResultOutOfRange());

        std::unique_ptr<uint8_t[]> reportData(new uint8_t[static_cast<uint32_t>(bodySize)]);
        NN_RESULT_THROW_UNLESS(reportData != nullptr, nn::os::ResultOutOfMemory());
        NN_RESULT_DO(fs::ReadFile(handle, *pOffset, reportData.get(), bodySize));
        *pOffset += bodySize;

        // ブロック本体の復号化
        if( header.flags & ErrorReportBinaryHeaderFlag_AesEncrypted )
        {
            NN_RESULT_DO(DecryptErrorReportBlock(reportData.get(), bodySize, decryptor));
        }

        erpt::ReportId *pReportId = reinterpret_cast<erpt::ReportId*>(blockHeader.reportId);
        erpt::ReportMetaData metaData;
        if( header.version <= 1 )
        {
            std::memset(metaData.userData, 0, sizeof(metaData.userData));
        }
        else
        {
            std::memcpy(metaData.userData, blockHeader.metaData, sizeof(metaData.userData));
        }
        NN_RESULT_DO(errors::DumpReportCore(*pReportId, metaData, reportData.get(), blockHeader.reportSize));

        NN_RESULT_SUCCESS;
    }
#endif
} //~anonymous namespace

ToolVersion GetToolVersion()
{
    ToolVersion toolVersion;
    toolVersion.major = NN_FIRMWARE_VERSION_MAJOR;
    toolVersion.minor = NN_FIRMWARE_VERSION_MINOR;
    toolVersion.micro = NN_FIRMWARE_VERSION_MICRO;
    toolVersion.majorRelStep = NN_FIRMWARE_VERSION_MAJORRELSTEP;
    toolVersion.minorRelStep = NN_FIRMWARE_VERSION_MINORRELSTEP;
    return toolVersion;
}

int GetToolVersionString(char* outStr, size_t outStrLength, const ToolVersion& toolVersion)
{
    return util::SNPrintf(outStr, outStrLength, "%u.%u.%u-%u.%u",
        toolVersion.major, toolVersion.minor, toolVersion.micro,
        toolVersion.majorRelStep, toolVersion.minorRelStep);
}

Result ExportReport(const char* pathString, const nn::erpt::ReportList& reportList, ErrorReportBinaryHeaderFlag flags)
{
    // 作成前に削除
    NN_RESULT_TRY(fs::DeleteFile(pathString))
        NN_RESULT_CATCH(fs::ResultPathNotFound) {}
    NN_RESULT_END_TRY
    NN_RESULT_DO(fs::CreateFile(pathString, 0));

    fs::FileHandle handle;
    int64_t offset = 0;
    NN_RESULT_DO(fs::OpenFile(&handle, pathString, fs::OpenMode_Write | fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

    // ヘッダを作成
    ErrorReportBinaryHeader header;
    NN_RESULT_DO(GenerateErrorReportBinaryHeader(&header, static_cast<Bit16>(reportList.reportCount), flags));
    offset += sizeof(header);

    crypto::Aes128CtrEncryptor aesEncryptor;
    if( header.flags & ErrorReportBinaryHeaderFlag_AesEncrypted )
    {
        aesEncryptor.Initialize(header.rsa.aes128Key, sizeof(header.rsa.aes128Key),
            header.rsa.aes128Iv, sizeof(header.rsa.aes128Iv));
    }

    // レポートデータの書き出し
    for( uint32_t i = 0; i < reportList.reportCount; ++i )
    {
        auto reportInfo = reportList.Report[i];
        NN_RESULT_DO(WriteErrorReportBlock(handle, header, reportInfo, &offset, aesEncryptor));
    }

    // RSA OAEP でヘッダの RSA フィールドを暗号化
    if( header.flags & ErrorReportBinaryHeaderFlag_RsaEncrypted )
    {
        uint8_t seed[32];
        os::GenerateRandomBytes(seed, sizeof(seed));
        uint8_t buffer[sizeof(header.rsa)];

        crypto::RsaOaepEncryptor<ErrorReportBinaryRsaKeyBytes, crypto::Sha256Generator> rsaEncryptor;
        rsaEncryptor.Initialize(g_ErrorReportBinaryRsaPublicModulus, ErrorReportBinaryRsaKeyBytes,
            g_ErrorReportBinaryRsaPublicExponent, sizeof(g_ErrorReportBinaryRsaPublicExponent));
        rsaEncryptor.Encrypt(buffer, sizeof(buffer),
            &header.rsa, sizeof(header.rsa) - sizeof(header.rsa.unavailable), seed, sizeof(seed));

        std::memcpy(&header.rsa, buffer, sizeof(header.rsa));
    }

    // ヘッダの書き出し
    NN_RESULT_DO(fs::WriteFile(handle, 0, &header, sizeof(header), fs::WriteOption()));
    NN_RESULT_DO(fs::FlushFile(handle));
    NN_RESULT_SUCCESS;
}

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)

Result DecryptReport(bool* outValue, const char* inputPathString, const char* outputPathString)
{
    fs::FileHandle handle;
    NN_RESULT_DO(fs::OpenFile(&handle, inputPathString, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

    int64_t fileSize;
    NN_RESULT_DO(fs::GetFileSize(&fileSize, handle));

    if( fileSize <= sizeof(ErrorReportBinaryHeader) )
    {
        DEVMENUCOMMAND_LOG("%s is invalid.\n", inputPathString);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    std::unique_ptr<uint8_t[]> inputData(new uint8_t[static_cast<uint32_t>(fileSize)]);
    NN_RESULT_THROW_UNLESS(inputData != nullptr, nn::os::ResultOutOfMemory());

    NN_RESULT_DO(fs::ReadFile(handle, 0, inputData.get(), static_cast<size_t>(fileSize)));
    fs::CloseFile(handle);

    fs::DeleteFile(outputPathString);
    NN_RESULT_DO(fs::CreateFile(outputPathString, 0));
    NN_RESULT_DO(fs::OpenFile(&handle, outputPathString, fs::OpenMode_Write | fs::OpenMode_AllowAppend));

    ErrorReportBinaryHeader *pHeader = reinterpret_cast<ErrorReportBinaryHeader*>(inputData.get());
    if( pHeader->magicNumber != ErrorReportBinaryHeaderMagicNumber )
    {
        DEVMENUCOMMAND_LOG("%s is invalid.\n", inputPathString);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    if( pHeader->flags & ErrorReportBinaryHeaderFlag_RsaEncrypted )
    {
        const auto plainSize = sizeof(pHeader->rsa) - sizeof(pHeader->rsa.unavailable);
        uint8_t buffer[plainSize];

        crypto::RsaOaepDecryptor<ErrorReportBinaryRsaKeyBytes, crypto::Sha256Generator> decryptor;
        decryptor.Initialize(g_ErrorReportBinaryRsaPublicModulus, ErrorReportBinaryRsaKeyBytes,
            g_ErrorReportBinaryRsaPrivateExponent, ErrorReportBinaryRsaKeyBytes);
        decryptor.Decrypt(buffer, sizeof(buffer), &pHeader->rsa, sizeof(pHeader->rsa));

        std::memcpy(&pHeader->rsa, buffer, plainSize);
        std::memset(&pHeader->rsa.unavailable, 0x0, sizeof(pHeader->rsa.unavailable));
    }

    if( pHeader->flags & ErrorReportBinaryHeaderFlag_AesEncrypted )
    {
        size_t decryptSize = static_cast<size_t>(fileSize) - sizeof(ErrorReportBinaryHeader);

        std::unique_ptr<uint8_t[]> decryptData(new uint8_t[static_cast<uint32_t>(decryptSize)]);
        NN_RESULT_THROW_UNLESS(decryptData != nullptr, nn::os::ResultOutOfMemory());

        crypto::Aes128CtrDecryptor decryptor;
        decryptor.Initialize(pHeader->rsa.aes128Key, sizeof(pHeader->rsa.aes128Key),
            pHeader->rsa.aes128Iv, sizeof(pHeader->rsa.aes128Iv));
        decryptor.Update(decryptData.get(), decryptSize, inputData.get() + sizeof(ErrorReportBinaryHeader), decryptSize);

        std::memcpy(inputData.get() + sizeof(ErrorReportBinaryHeader), decryptData.get(), decryptSize);
        std::memset(pHeader->rsa.aes128Key, 0x0, sizeof(pHeader->rsa.aes128Key));
        std::memset(pHeader->rsa.aes128Iv, 0x0, sizeof(pHeader->rsa.aes128Iv));
    }

    pHeader->flags = 0;

    NN_RESULT_DO(fs::WriteFile(handle, 0, inputData.get(), static_cast<size_t>(fileSize), fs::WriteOption()));
    NN_RESULT_DO(fs::FlushFile(handle));

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result DecryptAndDumpReport(bool* outValue, const char* inputFilePath)
{
    fs::FileHandle handle;
    NN_RESULT_DO(fs::OpenFile(&handle, inputFilePath, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

    int64_t fileSize;
    NN_RESULT_DO(fs::GetFileSize(&fileSize, handle));

    if( fileSize <= sizeof(errors::ErrorReportBinaryHeader) )
    {
        DEVMENUCOMMAND_LOG("%s is invalid.\n", inputFilePath);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    int64_t offset = 0;

    // ヘッダの読み込み
    errors::ErrorReportBinaryHeader header;
    NN_RESULT_DO(fs::ReadFile(handle, offset, &header, sizeof(header)));
    offset += sizeof(header);

    if( header.magicNumber != ErrorReportBinaryHeaderMagicNumber )
    {
        DEVMENUCOMMAND_LOG("%s is invalid.\n", inputFilePath);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    // ヘッダの RSA フィールドを復号化
    if( header.flags & ErrorReportBinaryHeaderFlag_RsaEncrypted )
    {
        const auto plainSize = sizeof(header.rsa) - sizeof(header.rsa.unavailable);
        uint8_t buffer[plainSize];

        crypto::RsaOaepDecryptor<ErrorReportBinaryRsaKeyBytes, crypto::Sha256Generator> decryptor;
        decryptor.Initialize(g_ErrorReportBinaryRsaPublicModulus, ErrorReportBinaryRsaKeyBytes,
            g_ErrorReportBinaryRsaPrivateExponent, ErrorReportBinaryRsaKeyBytes);
        decryptor.Decrypt(buffer, sizeof(buffer), &header.rsa, sizeof(header.rsa));

        std::memcpy(&header.rsa, buffer, plainSize);
    }

    if( header.version >= 2 )
    {
        char toolVersionStr[64];
        GetToolVersionString(toolVersionStr, sizeof(toolVersionStr), header.rsa.toolVersion);
        DEVMENUCOMMAND_LOG("Exported by DevMenuCommand ver. %s\n", toolVersionStr);
    }

    crypto::Aes128CtrDecryptor decryptor;
    decryptor.Initialize(header.rsa.aes128Key, sizeof(header.rsa.aes128Key),
        header.rsa.aes128Iv, sizeof(header.rsa.aes128Iv));

    DEVMENUCOMMAND_LOG("ReportCount: %d\n", header.rsa.reportCount);
    for( int i = 0; i < header.rsa.reportCount; i++ )
    {
        NN_RESULT_DO(DumpErrorReportBlock(handle, header, &offset, decryptor));
    }

    *outValue = true;
    NN_RESULT_SUCCESS;
}

#endif

}
