﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommand_ValueCommandBuilder.h"

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include "DevMenuCommand_CommandImplUtility.h"

namespace nn { namespace devmenucommand {

namespace detail {

Result ValueCommandBuilderBase::SetValueImpl(bool* outValue, const std::string& value) const NN_NOEXCEPT
{
    auto result = DoSetValue(value);
    if (!result)
    {
        NN_LOG("error: %s\n", result.message.c_str());
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    std::string v;
    if (DoGetValue(&v))
    {
        NN_LOG("set %s = %s\n", GetName().c_str(), v.c_str());
    }
    auto&& warnRestart = [this]
    {
        switch (GetReflectionTime())
        {
            case ReflectionTime::NeedsRelaunchApplication:
            {
                NN_LOG("Please restart the running application (if running) to apply this change.\n");
                return;
            }
            case ReflectionTime::NeedsRebootSystem:
            {
                NN_LOG("Please restart your target to apply this change.\n");
                return;
            }
            default: return;
        }
    };
    warnRestart();
    *outValue = true;
    NN_RESULT_SUCCESS;
}

Result ValueCommandBuilderBase::GetValueImpl(bool* outValue) const NN_NOEXCEPT
{
    std::string v;
    auto result = DoGetValue(&v);
    if (!result)
    {
        NN_LOG("%s\n", result.message.c_str());
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    NN_LOG("%s = %s\n", GetName().c_str(), v.c_str());
    *outValue = true;
    NN_RESULT_SUCCESS;
}

std::function<Result(bool* outValue, const Option&)> ValueCommandBuilderBase::MakeSetterFunction() const NN_NOEXCEPT
{
    return [p = this->shared_from_this()](bool* outValue, const Option& option)
    {
        return p->SetValueImpl(outValue, option.GetTarget());
    };
}

std::function<Result(bool* outValue, const Option&)> ValueCommandBuilderBase::MakeDefaultSetterFunction(const std::string& value) const NN_NOEXCEPT
{
    return [p = this->shared_from_this(), value](bool* outValue, const Option&)
    {
        return p->SetValueImpl(outValue, value);
    };
}

std::function<Result(bool* outValue, const Option&)> ValueCommandBuilderBase::MakeGetterFunction() const NN_NOEXCEPT
{
    return [p = this->shared_from_this()](bool* outValue, const Option&)
    {
        return p->GetValueImpl(outValue);
    };
}

SubCommand ValueCommandBuilderBase::MakeGetterSubCommand(const std::string& name) const NN_NOEXCEPT
{
    return
    {
        name,
        MakeGetterFunction(),
        GetGetterVisibility(),
        IsDeprecated(),
        "",
    };
}

SubCommand ValueCommandBuilderBase::MakeSetterSubCommand(const std::string& name) const NN_NOEXCEPT
{
    return
    {
        name,
        MakeSetterFunction(),
        GetSetterVisibility(),
        IsDeprecated(),
        this->GetArgumentHelpMessage(),
    };
}

std::vector<SubCommand> ValueCommandBuilderBase::MakeSubCommands() const NN_NOEXCEPT
{
    std::vector<SubCommand> ret;
    ret.push_back(MakeGetterSubCommand("get-" + this->GetName()));
    ret.push_back(MakeSetterSubCommand("set-" + this->GetName()));
    return ret;
}

std::vector<SubCommand> ValueCommandBuilder<bool>::MakeSubCommands() const NN_NOEXCEPT
{
    auto ret = Derived::MakeSubCommands();
    const auto& name = this->GetName();
    const std::string Suffix = "-enabled";
    if (auto pPrefix = RemoveSuffix(name, "-enabled"))
    {
        auto&& targetName = *pPrefix;
        ret.push_back(
        {
            "enable-" + targetName,
            MakeDefaultSetterFunction("true"),
            GetSetterVisibility(),
            true,
            "",
        });
        ret.push_back(
        {
            "disable-" + targetName,
            MakeDefaultSetterFunction("false"),
            GetSetterVisibility(),
            true,
            "",
        });
    }
    return ret;
}

} // detail

}}
