﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <string>
#include <functional>
#include <vector>
#include <utility>
#include <type_traits>
#include <tuple>
#include <nn/util/util_Optional.h>
#include <nn/nn_Assert.h>
#include "DevMenuCommand_CommandTypes.h"
#include "DevMenuCommand_CommandImplUtility.h"
#include "DevMenuCommand_ValueSerializers.h"

// for OptionData
#include <map>

namespace nn { namespace devmenucommand {

struct OptionData
{
    std::map<std::string, std::string> attributes;
    std::vector<std::string> args;

    ProcessResultT<std::string> ExtractAttribute(const std::vector<std::string>& list) const
    {
        for (auto&& key : list)
        {
            auto it = this->attributes.find(key);
            if (it != attributes.end())
            {
                return it->second;
            }
        }
        return ProcessResultT<std::string>::MakeFailure("not found option " + JoinString(list.begin(), list.end(), ","));
    }

    ProcessResultT<std::string> HasAttribute(const std::vector<std::string>& list) const
    {
        for (auto&& key : list)
        {
            auto it = this->attributes.find(key);
            if (it != attributes.end())
            {
                return std::string("true");
            }
        }
        return std::string("false");
    }

    ProcessResultT<std::string> ExtractArgument(int index) const
    {
        if (!(index < static_cast<int>(args.size())))
        {
            return ProcessResultT<std::string>::MakeFailure("no argument");
        }
        return args[index];
    }

};

template <typename Data>
struct ExtractorTag {};
template <typename Data, typename F> // F: (Data, const std::string&) -> String
using ExtractorWrapper = Wrapper<F, ExtractorTag<Data>>;

template <typename String = std::string>
class ExtractorHolder
{
private:

    std::function<ProcessResultT<String>(const OptionData&)> m_Extractor;

public:

    template <typename F>
    void SetExtractor(F&& f)
    {
        this->m_Extractor = std::forward<F>(f);
    }

    ProcessResultT<String> Extract(const OptionData& data) const
    {
        if (!m_Extractor)
        {
            return MakeProcessResultFailure("internal error: extractor is not set");
        }
        return m_Extractor(data);
    }

    // utility

    template <typename F>
    void SetProperty(const ExtractorWrapper<OptionData, F>& h)
    {
        SetExtractor(h.value);
    }

};

struct ParserTag {};
template <typename F> // F: (T*, String, const std::string& valueName) -> ProcessResult
using ParserWrapper = Wrapper<F, ParserTag>;

template <typename F>
inline auto MakeParser(F&& f)
{
    return Wrap<ParserTag>(std::forward<F>(f));
}

struct ValueToStringTag {};
template <typename F> // F: (T) -> String
using ValueToStringTagWrapper = Wrapper<F, ValueToStringTag>;

template <typename F>
inline auto MakeValueToString(F&& f)
{
    return Wrap<ValueToStringTag>(std::forward<F>(f));
}

struct ValidatorTag {};
template <typename F> // F: (T) -> ProcessResult
using ValidatorWrapper = Wrapper<F, ValidatorTag>;

template <typename F>
inline auto MakeValidator(F&& f, const std::string& errorMessage)
{
    return Wrap<ValidatorTag>([f, errorMessage](auto&& x, const std::string& valueName) -> ProcessResult
    {
        if (!f(x))
        {
            return MakeProcessResultFailure(valueName + " " + errorMessage);
        }
        return ProcessResultSuccess;
    });
}

struct ArgumentHelpMessageHolder
{
    std::function<std::string()> getArgumentHelpMessage;
};

template <typename T, typename String = std::string>
class ParserHolder
{
private:

    std::function<ProcessResult(T* pOut, const String&, const std::string& valueName)> m_Parser;

public:

    template <typename F>
    void SetParser(F&& f)
    {
        this->m_Parser = std::forward<F>(f);
    }

    ProcessResultT<T> Parse(const String& s, const std::string& valueName) const
    {
        T value;
        auto r = m_Parser(&value, s, valueName);
        if (!r)
        {
            return r;
        }
        return value;
    }

    // utility

    template <typename F>
    void SetProperty(const ParserWrapper<F>& h)
    {
        SetParser(h.value);
    }

    template <typename F>
    void SetProperty(const ValueToStringTagWrapper<F>&)
    {
        // nop
    }

};

struct ParameterStringTag {};
using ParameterSrtingWrapper = Wrapper<std::string, ParameterStringTag>;

inline auto MakeParameterString(const std::string& s)
{
    return Wrap<ParameterStringTag>(s);
}

struct HelpStringTag {};
template <typename F> // (std::string) -> std::string
using HelpSrtingWrapper = Wrapper<F, HelpStringTag>;

template <typename F>
inline auto MakeHelpString(F&& f)
{
    return Wrap<HelpStringTag>(std::forward<F>(f));
}

template <typename T, typename String = std::string>
class Parameter
    : public ExtractorHolder<String>
    , public ParserHolder<T, String>
{
private:

    using ExtractorHolder = ExtractorHolder<String>;
    using ParserHolder = ParserHolder<T, String>;

    std::string m_ParameterString = "value";
    std::function<std::string(const std::string&)> m_GetHelpString;
    std::vector<std::function<ProcessResult(const T&, const std::string& valueName)>> m_Validators;
    std::function<std::string()> m_GetArgumentHelpMessage;

public:

    using Type = T;

    using ExtractorHolder::SetProperty;
    using ParserHolder::SetProperty;

    ProcessResultT<T> GetValue(const OptionData& data) const
    {
        auto pString = ExtractorHolder::Extract(data);
        if (!pString)
        {
            return pString;
        }
        auto pValue = ParserHolder::Parse(*pString, m_ParameterString);
        if (!pValue)
        {
            return pValue;
        }
        auto&& value = *pValue;
        for (auto&& validator : m_Validators)
        {
            auto r = validator(value, m_ParameterString);
            if (!r)
            {
                return r;
            }
        }
        return value;
    }

    void SetParameterString(const std::string& s)
    {
        this->m_ParameterString = s;
    }

    template <typename F>
    void SetHelpString(F&& f)
    {
        this->m_GetHelpString = std::forward<F>(f);
    }

    template <typename F>
    void AddValidator(F&& f)
    {
        this->m_Validators.push_back(std::forward<F>(f));
    }

    std::string GetHelpString() const
    {
        auto s = m_GetArgumentHelpMessage ? m_GetArgumentHelpMessage() : "<value>";
        if (!m_GetHelpString)
        {
            return s;
        }
        return m_GetHelpString(s);
    }

    // utility

    void SetProperty(const ParameterSrtingWrapper& h)
    {
        SetParameterString(h.value);
    }

    template <typename F>
    void SetProperty(const HelpSrtingWrapper<F>& h)
    {
        SetHelpString(h.value);
    }

    template <typename F>
    void SetProperty(const ValidatorWrapper<F>& h)
    {
        AddValidator(h.value);
    }

    void SetProperty(const ArgumentHelpMessageHolder& h)
    {
        this->m_GetArgumentHelpMessage = h.getArgumentHelpMessage;
    }

};

template <typename... Types>
using Parameters = std::tuple<Parameter<Types>...>;

struct ProcessorTag {};
template <typename F> // F: (T...) -> ProcessResult
using ProcessorWrapper = Wrapper<F, ProcessorTag>;

template <typename... Types>
class Processor
{
private:

    Parameters<Types...> m_Parameters;
    std::function<ProcessResult(const Types&...)> m_F;

    // InvokeImpl(data, f, p1, p2, ..., pN)
    // {
    //     auto v1 = p1.GetValue(data);
    //     if (!v1) { return v1; } // error
    //     auto v2 = p2.GetValue(data);
    //     if (!v2) { return v2; } // error
    //     ...
    //     auto vn = pN.GetValue(data);
    //     if (!vn) { return vN; } // error
    //     return f(*v1, *v2, ..., *vN);
    // }

    template <typename F>
    ProcessResult InvokeImpl(const OptionData&, F&& f) const
    {
        return std::forward<F>(f)();
    }

    template <typename F, typename Head, typename... Tail>
    ProcessResult InvokeImpl(const OptionData& data, F&& f, const Head& head, const Tail&... tail) const
    {
        auto pValue = head.GetValue(data);
        if (!pValue)
        {
            return pValue;
        }
        return InvokeImpl(data, [&](auto&&... args){ return std::forward<F>(f)(*pValue, args...); }, tail...);
    }

public:

    void SetParameters(const Parameters<Types...>& parameters)
    {
        this->m_Parameters = parameters;
    }

    template <typename F>
    void SetFucntion(F&& f)
    {
        // ここでエラーが出た場合、処理関数の引数の型や数・順番がパラメータと一致しているかを確認する。
        this->m_F = std::forward<F>(f);
    }

    ProcessResult operator()(const OptionData& data) const
    {
        return ApplyTuple([&](auto&&... args){ return InvokeImpl(data, m_F, args...); }, m_Parameters);
    }

    template <typename F>
    void SetProperty(const ProcessorWrapper<F>& wrapper)
    {
        SetFucntion(wrapper.value);
    }

    std::string GetHelpMessage() const
    {
        std::string s;
        auto f = [&s](auto&& parameter) -> void
        {
            s += parameter.GetHelpString();
            s += " ";
        };
        ApplyTuple([&](auto&&... args){ ForeachArgs(f, args...); }, m_Parameters);
        return s;
    }

};

inline auto MakeOptionDataAttributeExtractor(std::vector<std::string> list)
{
    return Wrap<ExtractorTag<OptionData>>([list = std::move(list)](auto&& data)
    {
        return data.ExtractAttribute(list);
    });
}

inline auto MakeOptionDataAttributeFlagExtractor(std::vector<std::string> list)
{
    return Wrap<ExtractorTag<OptionData>>([list = std::move(list)](auto&& data)
    {
        return data.HasAttribute(list);
    });
}

inline auto MakeOptionDataArgumentExtractor(int index = 0)
{
    return Wrap<ExtractorTag<OptionData>>([index](auto&& data)
    {
        return data.ExtractArgument(index);
    });
}

inline auto GetDefaultSerializer() NN_NOEXCEPT
{
    return MakeValueToString([](const auto& x) -> std::string
    {
        using T = typename std::decay<decltype(x)>::type;
        return DefaultSerializer<T>::Serialize(x);
    });
};

inline auto GetDefaultDeserializer() NN_NOEXCEPT
{
    return MakeParser([](auto* pOut, const std::string& s, const std::string& valueName) -> ProcessResult
    {
        using T = std::decay_t<decltype(*pOut)>;
        return DefaultSerializer<T>::Deserialize(pOut, s, valueName);
    });
};

template <typename T, typename... Properties>
inline auto MakeParameter(Properties&&... properties)
{
    auto ret = Parameter<T>{};
    ret.SetProperty(GetDefaultDeserializer());
    SetProperties(ret, std::forward<Properties>(properties)...);
    return ret;
}

}}
