﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "DevMenuCommand_CommandTypes.h"
#include "DevMenuCommand_ValueCommandBuilder.h"
#include "DevMenuCommand_ValueSerializers.h"
#include "DevMenuCommand_CommandImplUtility.h"

#include <utility>
#include <string>
#include <functional>
#include <tuple>
#include <vector>
#include <memory>
#include <type_traits>

namespace nn { namespace devmenucommand {

template <typename T, typename... Args>
inline std::shared_ptr<ICommandBuilder> MakeValueCommandBuilder(const std::string& name, Args&&... args) NN_NOEXCEPT
{
    using Builder = detail::ValueCommandBuilder<T>;
    auto p = std::make_shared<Builder>();
    p->SetName(name);
    p->SetProperty(GetDefaultSerializer());
    p->SetProperty(GetDefaultDeserializer());
    p->SetProperty(ArgumentHelpMessageHolder{DefaultSerializer<T>::GetArgumentHelpMessage});
    ApplyRecursively([p = p.get()](auto&& v) { p->SetProperty(v); }, std::forward<Args>(args)...);
    return p;
}

namespace abbreviation {

/**
    @brief 指定した型の設定または取得をするコマンドの生成器を作成する。
    @tparam T 値の型
    @tparam Args 追加の属性
    @param[in] name 値の名前
    @param[in] args 追加の属性値
    @return 生成器を返す。

    @details
     T 型の値に対する以下のコマンドを生成する。

     - set-<name> 値
     - get-<name>
*/
template <typename T, typename... Args>
inline std::shared_ptr<ICommandBuilder> ValueCommand(const std::string& name, Args&&... args) NN_NOEXCEPT
{
    return MakeValueCommandBuilder<T>(name, std::forward<Args>(args)...);
}

/**
    @brief bool 値の設定または取得をするコマンドの生成器を作成する。
    @tparam Args 追加の属性
    @param[in] name 値の名前
    @param[in] args 追加の属性値
    @return 生成器を返す。

    @details
     以下のコマンドを生成する生成器を作成する。

     - set-<name> <true|false>
     - get-<name>

     さらに、name が "<value_name>-enabled" の形式の場合、以下のコマンドを追加で生成する。

     - enable-<value_name>
     - disable-<value_name>
*/
template <typename... Args>
inline std::shared_ptr<ICommandBuilder> BoolCommand(const std::string& name, Args&&... args) NN_NOEXCEPT
{
    return ValueCommand<bool>(name, std::forward<Args>(args)...);
}

/**
    @brief int 値の設定または取得をするコマンドの生成器を作成する。
    @tparam Args 追加の属性
    @param[in] name 値の名前
    @param[in] args 追加の属性値
    @return 生成器を返す。
*/
template <typename... Args>
inline std::shared_ptr<ICommandBuilder> IntCommand(const std::string& name, Args&&... args) NN_NOEXCEPT
{
    return ValueCommand<int>(name, std::forward<Args>(args)...);
}

/**
    @brief 文字列の設定または取得をするコマンドの生成器を作成する。
    @tparam Args 追加の属性
    @param[in] name 値の名前
    @param[in] args 追加の属性値
    @return 生成器を返す。
*/
template <typename... Args>
inline std::shared_ptr<ICommandBuilder> StringCommand(const std::string& name, Args&&... args) NN_NOEXCEPT
{
    return ValueCommand<std::string>(name, std::forward<Args>(args)...);
}

/**
    @brief 設定コマンドのヘルプに表示される文字列を変更する。
    @param s 設定コマンドのヘルプに表示される文字列
    @return 追加属性を返す。
*/
inline auto ArgumentString(const std::string& s) NN_NOEXCEPT
{
    return ArgumentHelpMessageHolder{[s]() { return s; }};
}

}

}}
