﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "DevMenuCommand_CommandTypes.h"
#include "DevMenuCommand_CommandBuilder.h"
#include "DevMenuCommand_ValueCommandBuilder.h"
#include "DevMenuCommand_CommandImplUtility.h"

#include <utility>
#include <string>
#include <type_traits>

namespace nn { namespace devmenucommand {

namespace detail {

template <typename Map>
inline auto MakeValueToStringForValueMap(Map&& map) NN_NOEXCEPT
{
    return MakeValueToString([map = std::forward<Map>(map)](const auto& x) -> std::string
    {
        auto it = std::find_if(map.begin(), map.end(), [&x](auto&& p){ return p.second == x; });
        if (it == map.end())
        {
            return "unknown value (" + ToString(x) + ")";
        }
        return it->first;
    });
}

template <typename Map>
inline auto MakeParserForValueMap(Map&& map) NN_NOEXCEPT
{
    return MakeParser([map = std::forward<Map>(map)](auto* pOut, const std::string& s, const std::string& valueName) -> ProcessResult
    {
        auto it = std::find_if(map.begin(), map.end(), [&s](auto&& p){ return p.first == s; });
        if (it == map.end())
        {
            return MakeProcessResultFailure(valueName + " must be one of [" + JoinString(std::begin(map), std::end(map), "|", [](auto&& p){ return p.first; }) + "]");
        }
        *pOut = it->second;
        return ProcessResultSuccess;
    });
}

} // detail

namespace abbreviation {

/**
    @brief いずれかの選択であることを強制する。
    @tparam T 値の型
    @param list 選択可能な値のリスト
*/
template <typename T>
inline auto EnumMap(const std::vector<std::pair<std::string, T>>& map)
{
    return std::make_tuple(
        detail::MakeParserForValueMap(map),
        detail::MakeValueToStringForValueMap(map),
        ArgumentString("<" + JoinString(std::begin(map), std::end(map), "|", [](auto&& p){ return p.first; }) + ">")
    );
}

template <typename Int, typename... Args>
inline auto EnumCommand(const std::string& name, const std::vector<std::pair<std::string, Int>>& map, Args&&... args) NN_NOEXCEPT
{
    return ValueCommand<Int>(name, EnumMap(map), std::forward<Args>(args)...);
}

template <typename Int, typename... Args>
inline auto Enum(const std::vector<std::pair<std::string, Int>>& map, Args&&... args) NN_NOEXCEPT
{
    return Param<Int>(EnumMap(map), std::forward<Args>(args)...);
}

}

}}
