﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "DevMenuCommand_CommandTypes.h"
#include "DevMenuCommand_ValueCommandBuilder.h"

#include <string>
#include <tuple>
#include <nn/util/util_IFunction.h>
#include <nn/bconfig/bconfig_Api.h>

namespace nn { namespace devmenucommand {

template <typename F>
inline auto MakeBootConfigValueGetter(F&& f) NN_NOEXCEPT
{
    return MakeValueGetter([f](auto* pOut) -> ProcessResult
    {
        nn::bconfig::BootConfig bootConfigData;
        nn::bconfig::LoadBootConfig(&bootConfigData);
        *pOut = f(&bootConfigData);
        return ProcessResultSuccess;
    });
}

template <typename F>
inline auto MakeBootConfigValueSetter(F&& f) NN_NOEXCEPT
{
    return MakeValueSetter([f](auto&& x) -> ProcessResult
    {
        nn::bconfig::BootConfig bootConfigData;
        nn::bconfig::LoadBootConfig(&bootConfigData);
        f(&bootConfigData, x);
        nn::bconfig::SaveBootConfig(&bootConfigData);
        return ProcessResultSuccess;
    });
}

namespace abbreviation {

/**
    @brief BootConfig 内のデータへの取得・設定を行うことを指定する。
    @tparam Getter 取得関数の型
    @tparam Setter 設定関数の型
    @param getter 取得関数
    @param setter 設定関数

    @pre getter(const nn::bconfig::BootConfig*) が取得したい値を返す
    @pre 設定したい値 x に対し setter(nn::bconfig::BootConfig*, x) で設定ができる
*/
template <typename Getter, typename Setter>
inline auto BootConfig(Getter&& getter, Setter&& setter) NN_NOEXCEPT
{
    return std::make_tuple(MakeBootConfigValueGetter(std::forward<Getter>(getter)), MakeBootConfigValueSetter(std::forward<Setter>(setter)));
}

}

}}
