﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommandAlbum_ProgramOption.h"

#include <cstring>
#include <nn/util/util_StringUtil.h>
#include <nn/nn_Log.h>

#define ALBUMSYNC_TEST_STRING(expected,value)    \
    (TestStringEquality((expected), (value), static_cast<int>(std::strlen(expected) + 1)))

namespace album {
namespace detail {

    char ConvertToLowerChar(char c) NN_NOEXCEPT
    {
        if(c >= 'A' && c <= 'Z')
        {
            return c - 'A' + 'a';
        }
        return c;
    }

    // 大文字小文字は無視する
    bool TestStringEquality(const char* expected, const char* value, size_t size) NN_NOEXCEPT
    {
        for(size_t i = 0; i < size; i++)
        {
            char a = expected[i];
            char b = value[i];
            if(a == '\0' && b == '\0')
            {
                return true;
            }
            else if(a == '\0' || b == '\0')
            {
                break;
            }

            a = ConvertToLowerChar(a);
            b = ConvertToLowerChar(b);

            if(a != b)
            {
                break;
            }
        }
        return false;
    }

    enum class ParseResult
    {
        Success,
        NotMatched,
        Error,
    };

    // <StorageValue> ::= "NAND" | "SD"
    ParseResult TryParseStorageValue(nn::fs::ImageDirectoryId* pOutValue, const char* pStorage) NN_NOEXCEPT
    {
        auto value = nn::fs::ImageDirectoryId::Nand;
        if (ALBUMSYNC_TEST_STRING("NAND", pStorage))
        {
            value = nn::fs::ImageDirectoryId::Nand;
        }
        else if (ALBUMSYNC_TEST_STRING("builtin", pStorage))
        {
            value = nn::fs::ImageDirectoryId::Nand;
        }
        else if (ALBUMSYNC_TEST_STRING("SD", pStorage))
        {
            value = nn::fs::ImageDirectoryId::SdCard;
        }
        else if (ALBUMSYNC_TEST_STRING("sdcard", pStorage))
        {
            value = nn::fs::ImageDirectoryId::SdCard;
        }
        else
        {
            return ParseResult::NotMatched;
        }
        *pOutValue = value;
        return ParseResult::Success;
    }

    // <Storage> ::= "--storage" <StorageValue>
    ParseResult TryParseStorage(nn::fs::ImageDirectoryId* pOutValue, const Option& option) NN_NOEXCEPT
    {
        // 省略形
        if (option.GetValue("--nand"))
        {
            *pOutValue = nn::fs::ImageDirectoryId::Nand;
            return ParseResult::Success;
        }
        if (option.GetValue("--sd"))
        {
            *pOutValue = nn::fs::ImageDirectoryId::SdCard;
            return ParseResult::Success;
        }
        // --storage <value> のパース
        auto pStorage    = option.GetValue("--storage");
        auto pStorageLen = pStorage ? std::strlen(pStorage) : 0;
        if (pStorageLen == 0)
        {
            return ParseResult::NotMatched;
        }

        auto value = nn::fs::ImageDirectoryId::Nand;
        if(TryParseStorageValue(&value, pStorage) != ParseResult::Success)
        {
            NN_LOG("storage name required after '--storage'.\n");
            return ParseResult::Error;
        }
        *pOutValue = value;
        return ParseResult::Success;
    }

    // <HostDirectoryValue> ::= string
    ParseResult TryParseHostDirectoryValue(char* buffer, size_t bufferSize, const char* pDirectory) NN_NOEXCEPT
    {
        if (nn::util::Strnlen(pDirectory, static_cast<int>(bufferSize)) >= static_cast<int>(bufferSize))
        {
            NN_LOG("host directory path is too long\n");
            return ParseResult::Error;
        }

        nn::util::Strlcpy(buffer, pDirectory, static_cast<int>(bufferSize));
        return ParseResult::Success;
    }

    // <HostDirectory> ::= "--directory" <HostDirectoryValue>
    ParseResult TryParseHostDirectory(char* buffer, size_t bufferSize, const Option& option) NN_NOEXCEPT
    {
        auto pDirectory    = option.GetValue("--directory");
        if (!pDirectory)
        {
            pDirectory = option.GetValue("--dir");
        }
        auto pDirectoryLen = pDirectory ? std::strlen(pDirectory) : 0;
        if (pDirectoryLen == 0)
        {
            return ParseResult::NotMatched;
        }

        if (TryParseHostDirectoryValue(buffer, bufferSize, pDirectory) != ParseResult::Success)
        {
            NN_LOG("host directory path is required after '--directory'.\n");
            return ParseResult::Error;
        }
        return ParseResult::Success;
    }

    // 単独指定オプション（引数なし）
    ParseResult TryParseSimpleOption(bool* pOutValue, const char* pString, const Option& option) NN_NOEXCEPT
    {
        if (!option.HasKey(pString))
        {
            return ParseResult::NotMatched;
        }
        *pOutValue = true;
        return ParseResult::Success;
    }

    bool Validate(const ProgramOption& opts) NN_NOEXCEPT
    {
        auto action = opts.GetAction();
        if (false
            || action == Action_Upload
            || action == Action_Download
            || action == Action_List
            || action == Action_Clean
            )
        {
            if (!opts.GetStorage())
            {
                NN_LOG("'--storage' is required.\n");
                return false;
            }
        }
        if (action == Action_Upload || action == Action_Download)
        {
            if (!opts.GetHostDirectory())
            {
                NN_LOG("'--directory' is required.\n");
                return false;
            }
            if (opts.IsExtraAlbumFileOnly())
            {
                NN_LOG("'--extra' is only available for list or clean command.\n");
                return false;
            }
            if (opts.IsRegularAlbumFileOnly())
            {
                NN_LOG("'--regular' is only available for list or clean command.\n");
                return false;
            }
        }

        // 安全のため directory にルート直下を指定させない
        if (const char* path = opts.GetHostDirectory())
        {
            // TORIAEZU:
            // 雑に判定
            int length = nn::util::Strnlen(path, ProgramOption::PathSizeMax);
            int delimiterCount = 0;
            bool endWithDelimiter = false;
            for (int i = 0; i < length; i++)
            {
                char c = path[i];
                if(c == '/' || c == '\\')
                {
                    delimiterCount++;
                    endWithDelimiter = true;
                }
                else
                {
                    endWithDelimiter = false;
                }
            }
            if (delimiterCount == 0 || (delimiterCount == 1 && endWithDelimiter))
            {
                NN_LOG("directory path must not be a root path.\n");
                NN_LOG("  path: %s\n", path);
                return false;
            }
        }
        return true;
    }

}   // namespace detail


// <ProgramOption> ::= <Action> (<Storage> | <HostDirectory> | <Force>)*
// ただし <Storage>, <HostDirectory>, <Force> の指定は最大 1 回ずつまで。
bool ProgramOption::TryParseArgument(const Option& option) NN_NOEXCEPT
{
    // --storage オプション
    {
        nn::fs::ImageDirectoryId storage;
        auto result = detail::TryParseStorage(&storage, option);
        if (result == detail::ParseResult::Error)
        {
            return false;
        }
        else if (result == detail::ParseResult::Success)
        {
            if (m_Storage)
            {
                NN_LOG("--storage option can't be specified twice.\n");
                return false;
            }
            this->m_Storage = storage;
        }
    }
    // --directory オプション
    {
        auto result = detail::TryParseHostDirectory(m_HostDirectory, sizeof(m_HostDirectory), option);
        if (result == detail::ParseResult::Error)
        {
            return false;
        }
        else if (result == detail::ParseResult::Success)
        {
            if (m_IsHostDirectorySpecified)
            {
                NN_LOG("--directory option can't be specified twice.\n");
                return false;
            }
            this->m_IsHostDirectorySpecified = true;
        }
    }
    // --check-empty オプション
    {
        bool value = false;
        auto result = detail::TryParseSimpleOption(&value, "--check-empty", option);
        if (result == detail::ParseResult::Success)
        {
            if (m_IsEmptyCheckRequired)
            {
                NN_LOG("--check-empty option can't be specified twice.\n");
                return false;
            }
            this->m_IsEmptyCheckRequired = value;
        }
    }
    // --force オプション
    {
        bool value = false;
        auto result = detail::TryParseSimpleOption(&value, "--force", option);
        if (result == detail::ParseResult::Success)
        {
            if (m_IsForceEnabled)
            {
                NN_LOG("--force option can't be specified twice.\n");
                return false;
            }
            this->m_IsForceEnabled = value;
            this->m_IsSkipEnabled = false;
        }
    }

    // --extra オプション
    {
        bool value = false;
        auto result = detail::TryParseSimpleOption(&value, "--extra", option);
        if (result == detail::ParseResult::Success)
        {
            if (m_IsExtraAlbumFileOnly)
            {
                NN_LOG("--extra option can't be specified twice.\n");
                return false;
            }
            this->m_IsExtraAlbumFileOnly = value;
        }
    }

    // --regular オプション
    {
        bool value = false;
        auto result = detail::TryParseSimpleOption(&value, "--regular", option);
        if (result == detail::ParseResult::Success)
        {
            if (m_IsRegularAlbumFileOnly)
            {
                NN_LOG("--regular option can't be specified twice.\n");
                return false;
            }
            this->m_IsRegularAlbumFileOnly = value;
        }
    }

    // --raw オプション
    {
        bool value = false;
        auto result = detail::TryParseSimpleOption(&value, "--raw", option);
        if (result == detail::ParseResult::Success)
        {
            if (m_IsRawModeEnabled)
            {
                NN_LOG("--raw option can't be specified twice.\n");
                return false;
            }
            this->m_IsRawModeEnabled = value;
        }
    }

    // オプションのない最後尾引数
    std::strncpy(this->m_LastArgument, option.GetTarget(option.GetTargetCount() - 1), sizeof(this->m_LastArgument) - 1);
    this->m_ArgumentCount = option.GetTargetCount();

    // オプションの組合せを検査
    return detail::Validate(*this);

}   // NOLINT(impl/function_size)

}  // namespace album

