﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommandAlbum_ExecuteCleanAction.h"

#include <vector>
#include <string>

#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>

#include "DevMenuCommandAlbum_StorageUtility.h"
#include "DevMenuCommandAlbum_ParseFilename.h"

namespace album {

namespace {

    bool g_IsRebootRequired = false;

    nn::Result DeleteFilesRecursivelyImpl(int64_t& totalCount, const std::string& path, const ProgramOption& opts)
    {
        nn::fs::DirectoryHandle hDir = {};
        NN_RESULT_DO(nn::fs::OpenDirectory(&hDir, path.c_str(), nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(hDir);
        };

        int64_t count = 0;
        NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&count, hDir));
        if(count == 0)
        {
            NN_RESULT_SUCCESS;
        }

        std::vector<nn::fs::DirectoryEntry> srcEntryList;
        srcEntryList.resize(static_cast<size_t>(count));

        int64_t n = 0;
        NN_RESULT_DO(nn::fs::ReadDirectory(&n, srcEntryList.data(), hDir, static_cast<int64_t>(srcEntryList.size())));

        int64_t deleteCount = 0;
        for (auto& e : srcEntryList)
        {
            std::string entryPath = path + "/" + e.name;

            // ディレクトリの場合
            if (e.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                NN_RESULT_DO(DeleteFilesRecursivelyImpl(totalCount, entryPath, opts));
                auto result = nn::fs::DeleteDirectory(entryPath.c_str());
                if (result.IsSuccess())
                {
                    g_IsRebootRequired = true;
                    NN_LOG("Deleting directory ... %s/\n", entryPath.c_str() + 7);
                }
                continue;
            }

            // ファイルの場合
            if (opts.IsExtraAlbumFileOnly() && !IsExtraAlbumFile(e.name))
            {
                NN_LOG("Skipping file ... %s\n", e.name);
                continue;
            }
            if (opts.IsRegularAlbumFileOnly() && !IsRegularAlbumFile(e.name))
            {
                NN_LOG("Skipping file ... %s\n", e.name);
                continue;
            }
            NN_LOG("Deleting file ... %s\n", e.name);
            NN_RESULT_DO(nn::fs::DeleteFile(entryPath.c_str()));
            g_IsRebootRequired = true;
            ++deleteCount;
        }

        totalCount += deleteCount;
        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteAllFilesRecursively(const char* mountName, const ProgramOption& opts) NN_NOEXCEPT
    {
        std::string rootPath = std::string(mountName) + ":/";

        int64_t totalCount = 0;
        NN_RESULT_DO( DeleteFilesRecursivelyImpl(totalCount, rootPath, opts) );

        NN_LOG("Delete %d files\n", static_cast<int>(totalCount));
        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteAllFiles(const char* mountName) NN_NOEXCEPT
    {
        std::string rootPath = std::string(mountName) + ":/";

        nn::fs::DirectoryHandle hDir = {};
        NN_RESULT_DO(nn::fs::OpenDirectory(&hDir, rootPath.c_str(), nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(hDir);
        };
        int64_t count = 0;
        NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&count, hDir));
        if (count == 0)
        {
            NN_RESULT_SUCCESS;
        }

        std::vector<nn::fs::DirectoryEntry> entryList;
        entryList.resize(static_cast<size_t>(count));

        int deleteCount = 0;
        int64_t n = 0;
        NN_RESULT_DO(nn::fs::ReadDirectory(&n, entryList.data(), hDir, static_cast<int64_t>(entryList.size())));

        for (auto& e : entryList)
        {
            std::string path = rootPath + "/" + e.name;
            if (e.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                NN_LOG("Deleting directory ... %s/\n", path.c_str() + 7);
                NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(path.c_str()));
                g_IsRebootRequired = true;
                ++deleteCount;
            }
            else
            {
                NN_LOG("Deleting file ... %s\n", e.name);
                NN_RESULT_DO(nn::fs::DeleteFile(path.c_str()));
                g_IsRebootRequired = true;
                ++deleteCount;
            }
        }

        NN_LOG("Deleted %d files/directories.\n", deleteCount);
        NN_RESULT_SUCCESS;
    }

}   // namespace

bool ExecuteCleanAction(const ProgramOption& opts) NN_NOEXCEPT
{
    g_IsRebootRequired = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (g_IsRebootRequired)
        {
            NN_LOG("\x1b[31m=============================================================\x1b[0m\n");
            NN_LOG("\x1b[31m*** Please reboot the target to allow changes to take effect.\x1b[0m\n");
            NN_LOG("\x1b[31m=============================================================\x1b[0m\n");
        }
    };

    NN_LOG("Cleaning Album: storage=\x22%s\x22\n", opts.GetStorageName());

    if (opts.IsEmptyCheckRequired())
    {
        NN_LOG("Failed. '--check-empty' option is not available for clean command.\n");
        return false;
    }
    if (!MountTargetStorage(*opts.GetStorage()))
    {
        NN_LOG("Cleaning Album ... cannot mount target storage.\n");
        return false;
    }
    NN_UTIL_SCOPE_EXIT
    {
        UnmountTargetStorage();
    };
    NN_UTIL_SCOPE_EXIT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        switch (*opts.GetStorage())
        {
            case nn::fs::ImageDirectoryId::Nand:
            {
                nn::capsrv::RefreshAlbumCache(nn::capsrv::AlbumStorage_Nand);
                break;
            }
            case nn::fs::ImageDirectoryId::SdCard:
            {
                nn::capsrv::RefreshAlbumCache(nn::capsrv::AlbumStorage_Sd);
                break;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
    };

    if (opts.IsExtraAlbumFileOnly() || opts.IsRegularAlbumFileOnly())
    {
        if (!DeleteAllFilesRecursively("TARG", opts).IsSuccess())
        {
            NN_LOG("Cleaning Album ... failed to complete\n");
            return false;
        }
    }
    else
    {
        if (!DeleteAllFiles("TARG").IsSuccess())
        {
            NN_LOG("Cleaning Album ... failed to complete\n");
            return false;
        }
    }
    NN_LOG("Cleaning Album ... completed.\n");
    return true;
}

}   // namespace album

