﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <curl/curl.h>

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/ssl/ssl_Context.h>

namespace accounts {

int UrlEncode(char *dst, size_t dstSize, const char* src, size_t srcSize) NN_NOEXCEPT;

class SimpleDownloader
{
    NN_STATIC_ASSERT(LIBCURL_VERSION_NUM >= 0x072a01); // サポートバージョン

private:
    CURL* m_CurlHandle;
    bool m_Initialized;

    struct SslCtxFunctionContext
    {
        bool isCalled;
        nn::Result result;
        void* userData;

        static const SslCtxFunctionContext Default;
    } m_SslCtxFunctionContext;

    struct
    {
        void* address;
        size_t total;
        size_t used;
        std::atomic<bool> failure;
    } m_BufferInfo;

    void Reset() NN_NOEXCEPT;
    CURLcode SetDefault() NN_NOEXCEPT;
    CURLcode SetUrl(const char* url) NN_NOEXCEPT;
    CURLcode SetHttpPost(const char* patchData) NN_NOEXCEPT;
    CURLcode SetDebugMode() NN_NOEXCEPT;

    static size_t HeaderFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT;
    static size_t WriteFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT;
    static int XferInfoFunction(void* pContext, curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow) NN_NOEXCEPT;
    static CURLcode DefaultSslCtxFunction(CURL* pCurl, void* pSslContext, SslCtxFunctionContext* pFunctionContext) NN_NOEXCEPT;

public:
    explicit SimpleDownloader(CURL* curlHandle) NN_NOEXCEPT;

    void Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    void Setup(const char* url, const char* postData, bool verbose) NN_NOEXCEPT;
    CURLcode Invoke(size_t* pOutSizeActual) NN_NOEXCEPT;
    long GetHttpCode() const NN_NOEXCEPT;
};

// リクエストを解析

template <size_t UriBaseSize>
class UriParser
{
private:
    const char (&m_UriBase)[UriBaseSize];

protected:
    struct UriLookupEntry
    {
        const char* path;
        bool found;
        explicit UriLookupEntry(const char* p) NN_NOEXCEPT : path(p) , found(false) { }
        NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT { return found; }
        bool CanAccept(const char* key, size_t keyLength) const NN_NOEXCEPT
        {
            return !(*this) && (std::strncmp(path, key, keyLength) == 0);
        }
        void MarkAccepted() NN_NOEXCEPT { found = true; }
    };

    virtual void UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT = 0;
    void Update(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        UpdateImpl(key, keyLength, value, valueLength);
    }

public:
    explicit UriParser(const char (&uriBase)[UriBaseSize]) NN_NOEXCEPT
        : m_UriBase(uriBase)
    {
    }

#define ACCOUNTS_RETURN_FALSE_UNLESS(cond) \
    do \
    { \
        if (!NN_STATIC_CONDITION(cond)) \
        { \
            return false; \
        } \
    } while(NN_STATIC_CONDITION(false))

    bool Parse(const char* document, size_t bufferSize) NN_NOEXCEPT
    {
        const auto BaseLength = strnlen(m_UriBase, UriBaseSize);
        ACCOUNTS_RETURN_FALSE_UNLESS(BaseLength < UriBaseSize);

        const auto DocumentLength = strnlen(document, bufferSize);
        ACCOUNTS_RETURN_FALSE_UNLESS(DocumentLength < bufferSize);

        ACCOUNTS_RETURN_FALSE_UNLESS(strncmp(m_UriBase, document, BaseLength) == 0);

        if (!(DocumentLength + 1 > BaseLength))
        {
            return false;
        }
        ACCOUNTS_RETURN_FALSE_UNLESS(document[BaseLength] == '?' || document[BaseLength] == '#');

        auto key = &document[BaseLength + 1];
        while (key && key - document <= static_cast<int>(DocumentLength))
        {
            auto p = strchr(key, '&');

            auto tokenLength = (p? p - key : DocumentLength - (key - document));
            ACCOUNTS_RETURN_FALSE_UNLESS(tokenLength > 0);
            auto tokenEnd = key + tokenLength;

            auto keyEnd = strchr(key, '=');
            ACCOUNTS_RETURN_FALSE_UNLESS(keyEnd);

            auto keylength = keyEnd - key;
            ACCOUNTS_RETURN_FALSE_UNLESS(keylength > 0);

            auto value = keyEnd + 1;
            Update(key, keylength, value, tokenEnd - value);

            key = (p ? p + 1 : nullptr);
        }
        return true;
    }
#undef ACCOUNTS_RETURN_FALSE_UNLESS
};

} // ~namespace nn::accounts
