﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>
#include <vector>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_Result.h>
#include <nn/ns/ns_SdCardApi.h>
#include <nn/time.h>
#include <nn/time/time_ApiForMenu.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Json.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_SaveDataCommand.h"

using namespace nn;
using namespace nne;

namespace
{
    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " savedata list\n"
        "       " DEVMENUCOMMAND_NAME " savedata delete <savedata_id>\n"
        "       " DEVMENUCOMMAND_NAME " savedata delete --savedata-id <savedata_id>\n"
        "       " DEVMENUCOMMAND_NAME " savedata delete --application-id <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " savedata delete all\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " savedata list-detail\n"
        "       " DEVMENUCOMMAND_NAME " savedata corrupt <savedata_id>\n"
#endif
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    Result SaveDataListCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        std::unique_ptr<fs::SaveDataIterator> iter;
        NN_RESULT_DO(fs::OpenSaveDataIterator(&iter, fs::SaveDataSpaceId::User));

        for (;;)
        {
            fs::SaveDataInfo info;
            int64_t count;
            NN_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                break;
            }

            NN_LOG("id: 0x%016llx\n", info.saveDataId);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    bool IsSaveDataDeleted( std::vector<nn::fs::SaveDataId>& deletedSaveDataIdList, fs::SaveDataId saveDataId )
    {
        auto result = std::find( deletedSaveDataIdList.begin(), deletedSaveDataIdList.end(), saveDataId );
        if ( result != deletedSaveDataIdList.end() )
        {
            return true;
        }
        return false;
    }

    Result DeleteSaveData( bool* outValue, std::vector<nn::fs::SaveDataId>& deletedSaveDataIdList, const char* saveDataId )
    {
        // セーブデータIDを指定して削除
        fs::SaveDataId id = { STR_TO_ULL( saveDataId, NULL, 16 ) };

        if ( !IsSaveDataDeleted( deletedSaveDataIdList, id ) )
        {
            NN_RESULT_TRY( fs::DeleteSaveData( id ) )
                NN_RESULT_CATCH( fs::ResultTargetNotFound )
                {
                    NN_LOG( "Save data id 0x%016llx is not found.\n", id );
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY

            NN_LOG( "deleted id: 0x%016llx\n", id );
            deletedSaveDataIdList.push_back( id );
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeleteSaveDataAll( bool* outValue, std::vector<nn::fs::SaveDataId>& deletedSaveDataIdList, const char* applicationId )
    {
        std::vector<nn::fs::SaveDataId> saveDataIdList;
        std::unique_ptr< fs::SaveDataIterator > iter;
        NN_RESULT_DO( fs::OpenSaveDataIterator( &iter, fs::SaveDataSpaceId::User ) );

        for (;;)
        {
            fs::SaveDataInfo info;
            int64_t count;
            NN_RESULT_DO( iter->ReadSaveDataInfo( &count, &info, 1 ) );
            if ( count == 0 )
            {
                break;
            }

            // アプリケーションIDの指定がない場合は、全てのセーブデータを削除する。
            if ( applicationId == nullptr )
            {
                saveDataIdList.push_back( info.saveDataId );
            }
            else
            {
                // アプリケーションIDに紐付く全てのセーブデータを削除する。
                ncm::ApplicationId id = { STR_TO_ULL( applicationId, NULL, 16 ) };
                if ( info.applicationId == id )
                {
                    if ( info.saveDataType == fs::SaveDataType::Account ||
                        info.saveDataType == fs::SaveDataType::Bcat ||
                        info.saveDataType == fs::SaveDataType::Device ||
                        info.saveDataType == fs::SaveDataType::Cache )
                    {
                        saveDataIdList.push_back( info.saveDataId );
                    }
                }
            }
        }

        if ( saveDataIdList.size() == 0 )
        {
            if ( applicationId == nullptr )
            {
                NN_LOG( "Save data is not found.\n" );
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
            else
            {
                ncm::ApplicationId id = { STR_TO_ULL( applicationId, NULL, 16 ) };
                NN_LOG( "Save data associated with application id 0x%016llx is not found.\n", id );
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }

        for( auto saveDataId : saveDataIdList )
        {
            NN_RESULT_DO( fs::DeleteSaveData( saveDataId ) );
            NN_LOG( "deleted id: 0x%016llx\n", saveDataId );
            deletedSaveDataIdList.push_back( saveDataId );
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeleteSaveDataAll( bool* outValue, std::vector<nn::fs::SaveDataId>& deletedSaveDataIdList )
    {
        const char* applicationId = nullptr;
        NN_RESULT_DO( DeleteSaveDataAll( outValue, deletedSaveDataIdList, applicationId ) );
        NN_RESULT_SUCCESS;
    }

    Result SaveDataDeleteCommand(bool* outValue, const Option& option)
    {
        std::vector<nn::fs::SaveDataId> deletedSaveDataIdList;

        auto target = option.GetTarget();
        if ( ( std::strlen( target ) == 0 && !option.HasKey( "--application-id" ) && !option.HasKey( "--savedata-id" ) ) ||
           ( option.HasKey( "--application-id" ) && std::strlen( option.GetValue( "--application-id" ) ) == 0 ) ||
           ( option.HasKey( "--savedata-id" ) && std::strlen( option.GetValue( "--savedata-id" ) ) == 0 ) )
        {
            NN_LOG("usage: " DEVMENUCOMMAND_NAME " savedata delete <savedata_id>\n");
            NN_LOG("       " DEVMENUCOMMAND_NAME " savedata delete --savedata-id <savedata_id>\n");
            NN_LOG("       " DEVMENUCOMMAND_NAME " savedata delete --application-id <application_id>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        // 全てのセーブデータを削除
        if ( std::strcmp( target, "all" ) == 0 )
        {
            NN_RESULT_DO( DeleteSaveDataAll( outValue, deletedSaveDataIdList ) );
            NN_RESULT_SUCCESS;
        }

        // アプリケーションIDに紐付くセーブデータを削除
        if ( option.HasKey( "--application-id" ) )
        {
            auto applicationId = option.GetValue( "--application-id" );
            NN_RESULT_DO( DeleteSaveDataAll( outValue, deletedSaveDataIdList, applicationId ) );
        }

        // セーブデータIDを指定してセーブデータを削除
        if ( option.HasKey( "--savedata-id" ) )
        {
            auto saveDataId = option.GetValue( "--savedata-id" );
            NN_RESULT_DO( DeleteSaveData( outValue, deletedSaveDataIdList, saveDataId ) );
        }
        else if ( std::strlen( target ) != 0 )
        {
            NN_RESULT_DO( DeleteSaveData( outValue, deletedSaveDataIdList, target ) );
        }

        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    const char* GetSaveDataTypeString(fs::SaveDataType type)
    {
        switch (type)
        {
        case fs::SaveDataType::Account: return "Account";
        case fs::SaveDataType::Bcat: return "Bcat";
        case fs::SaveDataType::Device: return "Device";
        case fs::SaveDataType::System: return "System";
        case fs::SaveDataType::Cache: return "Cache";
        case fs::SaveDataType::Temporary: return "Temporary";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* GetSaveDataSpaceIdString(fs::SaveDataSpaceId spaceId)
    {
        switch (spaceId)
        {
        case fs::SaveDataSpaceId::System: return "System";
        case fs::SaveDataSpaceId::User: return "User";
        case fs::SaveDataSpaceId::SdUser: return "SdUser";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    Result ListDetailCommand(bool* outValue, const Option&)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ NN_ABORT_UNLESS_RESULT_SUCCESS(time::Finalize()); };

        rapidjson::Document document;
        document.SetArray();


        auto iterate = [&document](fs::SaveDataSpaceId spaceId)->Result
        {
            std::unique_ptr<fs::SaveDataIterator> iter;
            NN_RESULT_DO(fs::OpenSaveDataIterator(&iter, spaceId));

            for (;;)
            {
                fs::SaveDataInfo info;
                rapidjson::Value item(rapidjson::kObjectType);
                int64_t count;
                NN_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));
                if (count == 0)
                {
                    break;
                }

                item.AddMember("id", rapidjson::Value(devmenuUtil::GetBit64String(info.saveDataId).data, document.GetAllocator()), document.GetAllocator());
                item.AddMember("type", rapidjson::Value(GetSaveDataTypeString(info.saveDataType), document.GetAllocator()), document.GetAllocator());
                item.AddMember("size", rapidjson::Value(info.saveDataSize), document.GetAllocator());

                int64_t saveDataAvaiableSize;
                NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetSaveDataAvailableSize(&saveDataAvaiableSize, spaceId, info.saveDataId));
                item.AddMember("availableSize", rapidjson::Value(saveDataAvaiableSize), document.GetAllocator());
                int64_t journalSize;
                NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetSaveDataJournalSize(&journalSize, spaceId, info.saveDataId));
                item.AddMember("journalSize", rapidjson::Value(journalSize), document.GetAllocator());

                uint32_t flags;
                auto result = fs::GetSaveDataFlags(&flags, spaceId, info.saveDataId);
                if (result.IsFailure())
                {
                    item.AddMember("flagsFailureResult", rapidjson::Value(result.GetInnerValueForDebug()), document.GetAllocator());
                }
                else
                {
                    item.AddMember("flags", rapidjson::Value(flags), document.GetAllocator());
                }

                item.AddMember("spaceId", rapidjson::Value(GetSaveDataSpaceIdString(spaceId), document.GetAllocator()), document.GetAllocator());
                if (info.applicationId != ncm::ApplicationId::GetInvalidId())
                {
                    item.AddMember("applicationId", rapidjson::Value(devmenuUtil::GetBit64String(info.applicationId.value).data, document.GetAllocator()), document.GetAllocator());
                }
                if (info.systemSaveDataId != fs::InvalidSystemSaveDataId)
                {
                    item.AddMember("systemId", rapidjson::Value(devmenuUtil::GetBit64String(info.systemSaveDataId).data, document.GetAllocator()), document.GetAllocator());
                }
                if (info.saveDataUserId != fs::InvalidUserId)
                {
                    item.AddMember("userId0", rapidjson::Value(devmenuUtil::GetBit64String(info.saveDataUserId._data[0]).data, document.GetAllocator()), document.GetAllocator());
                    item.AddMember("userId1", rapidjson::Value(devmenuUtil::GetBit64String(info.saveDataUserId._data[1]).data, document.GetAllocator()), document.GetAllocator());
                }

                Bit64 ownerId;
                NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetSaveDataOwnerId(&ownerId, spaceId, info.saveDataId));
                item.AddMember("ownerId", rapidjson::Value(devmenuUtil::GetBit64String(ownerId).data, document.GetAllocator()), document.GetAllocator());

                time::PosixTime timestamp;
                NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetSaveDataTimeStamp(&timestamp, spaceId, info.saveDataId));

                // fs::GetSaveDataTimeStamp が未実装なので、現時点では timestamp の値は 0 が必ず返る。
                // 意味のない値を表示しても紛らわしいので、有効な値以外は表示しないように判定を加えておく。
                // 実装されたら判定文は不要になる：http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-30752
                if( timestamp >= time::InputPosixTimeMin && timestamp <= time::InputPosixTimeMax )
                {
                    item.AddMember("timestamp", rapidjson::Value(timestamp.value), document.GetAllocator());
                }

                document.PushBack(item, document.GetAllocator());
            }


            NN_RESULT_SUCCESS;
        };

        NN_RESULT_DO(iterate(fs::SaveDataSpaceId::User));
        if (ns::CheckSdCardMountStatus().IsSuccess())
        {
            NN_RESULT_DO(iterate(fs::SaveDataSpaceId::SdUser));
        }
        NN_RESULT_DO(iterate(fs::SaveDataSpaceId::System));

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());


        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CorruptCommand(bool* outValue, const Option& option)
    {
        NN_RESULT_DO(fs::CorruptSaveDataForDebug(std::strtoull(option.GetTarget(), nullptr, 16)));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    const SubCommand g_SubCommands[] =
    {
        { "list", SaveDataListCommand },
        { "delete", SaveDataDeleteCommand },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "list-detail", ListDetailCommand },
        { "corrupt", CorruptCommand },
#endif
    };
}

Result SaveDataCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu savedata command. See '" DEVMENUCOMMAND_NAME " savedata --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
