﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <string>
#include <vector>
#include <cstdlib>
#include <sstream>
#include <iomanip>

#include <nn/account.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_ApiForPrivate.h>
#include <nn/olsc/olsc_ApiForSystemService.h>
#include <nn/olsc/olsc_DaemonController.h>
#include <nn/olsc/olsc_DataTransferPolicyApi.h>
#include <nn/olsc/olsc_RemoteStorageController.h>
#include <nn/olsc/olsc_TransferTaskListController.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/time/time_TimeZoneApi.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Eula.h"
#include "DevMenuCommand_Json.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Language.h"
#include "DevMenuCommand_OlscCommand.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_StorageId.h"
#include "DevMenuCommand_StrToUll.h"

using namespace nn;
using namespace nne;

#define NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(s) \
    do {    \
        auto nn_dmc_olsc_command_result = (s);          \
        if(nn_dmc_olsc_command_result.IsFailure()) {    \
            NN_LOG("Result: %08x\n", nn_dmc_olsc_command_result.GetInnerValueForDebug());   \
            return false;   \
        }                   \
    } while(NN_STATIC_CONDITION(false))

//------------------------------------------------------------------------------------------------

namespace {

    struct SubCommand
    {
        std::string name;
        bool(*function)(const Option&);
    };


    Result GetUidFromIndex(util::optional<account::Uid>* outValue, int userIndex)
    {
        auto index = static_cast<size_t>(userIndex);

        std::vector<account::Uid> uidList(account::UserCountMax);
        int count;
        NN_RESULT_DO(account::ListAllUsers(&count, uidList.data(), static_cast<int>(uidList.size())));
        if (count == 0)
        {
            *outValue = util::nullopt;
            NN_RESULT_SUCCESS;
        }
        uidList.resize(static_cast<size_t>(count));

        if (index >= uidList.size())
        {
            *outValue = util::nullopt;
            NN_RESULT_SUCCESS;
        }

        *outValue = uidList[index];
        NN_RESULT_SUCCESS;
    }

    bool DeleteAllTransferTask(const Option&)
    {
        olsc::DeleteAllTransferTask();
        return true;
    }

    const char* GetTaskKindString(olsc::TransferTaskKind kind)
    {
        switch (kind)
        {
            case olsc::TransferTaskKind::Upload:
                return "Upload";
            case olsc::TransferTaskKind::Download:
                return "Download";
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* GetDataStatusString(olsc::DataStatus status)
    {
        switch (status)
        {
        case olsc::DataStatus::Fixed:
            return "Fixed";
        case olsc::DataStatus::Differential_Uploading:
            return "DefferentialUploading";
        case olsc::DataStatus::Uploading:
            return "Uploading";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    util::optional<olsc::TransferTaskKind> ConvertToTransferTaskKind(const std::string& kindString)
    {
        if (kindString == "ul")
        {
            return olsc::TransferTaskKind::Upload;
        }
        else if (kindString == "dl")
        {
            return olsc::TransferTaskKind::Download;
        }
        return util::nullopt;
    }

    util::optional<bool> ConvertToBoolean(const std::string& booleanString)
    {
        if (booleanString == "true")
        {
            return true;
        }
        else if (booleanString == "false")
        {
            return false;
        }
        return util::nullopt;
    }

    const char* GetTaskStatusString(olsc::TransferTaskStatus status)
    {
        switch (status)
        {
        case olsc::TransferTaskStatus::Completed:
            return "Completed";
        case olsc::TransferTaskStatus::Error:
            return "Error";
        case olsc::TransferTaskStatus::Runnable:
            return "Runnable";
        case olsc::TransferTaskStatus::Running:
            return "Running";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* GetTaskRankString(olsc::TransferTaskRank rank)
    {
        switch (rank)
        {
        case olsc::TransferTaskRank::Basic:
            return "Basic";
        case olsc::TransferTaskRank::Postponed:
            return "Postponed";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    rapidjson::StringBuffer& GenerateTransferTaskInfoListJson(rapidjson::StringBuffer& buffer, const olsc::TransferTaskInfo infoList[], int count)
    {
        rapidjson::Document document;
        document.SetObject();
        auto& allocator = document.GetAllocator();

        document.AddMember("count", rapidjson::Value(count), allocator);

        rapidjson::Value list(rapidjson::kArrayType);

        for (int i = 0; i < count; ++i)
        {
            auto& info = infoList[i];
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("Id",
                rapidjson::Value(devmenuUtil::GetBit64String(info.id).data, allocator), allocator);
            item.AddMember("applicationId",
                rapidjson::Value(devmenuUtil::GetBit64String(info.applicationId.value).data, allocator), allocator);

            char tmp[64];
            util::TSNPrintf(tmp, sizeof(tmp), "%016llx-%016llx", info.uid._data[0], info.uid._data[1]);
            item.AddMember("uid", rapidjson::Value(tmp, allocator), allocator);

            item.AddMember("status", rapidjson::Value(GetTaskStatusString(info.status), allocator), allocator);
            item.AddMember("kind", rapidjson::Value(GetTaskKindString(info.type), allocator), allocator);
            item.AddMember("created", rapidjson::Value(devmenuUtil::GetDateTimeString(tmp, sizeof(tmp), info.createdTime), allocator), allocator);
            item.AddMember("rank", rapidjson::Value(GetTaskRankString(info.rank), allocator), allocator);

            list.PushBack(item, allocator);
        }

        document.AddMember("list", list, allocator);

        buffer.Clear();
        rapidjson::PrettyWriter<rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    rapidjson::StringBuffer& GenerateTransferTaskErrorInfoListJson(rapidjson::StringBuffer& buffer, const olsc::TransferTaskErrorInfo infoList[], int count)
    {
        rapidjson::Document document;
        document.SetObject();
        auto& allocator = document.GetAllocator();

        document.AddMember("count", rapidjson::Value(count), allocator);

        rapidjson::Value list(rapidjson::kArrayType);

        for (int i = 0; i < count; ++i)
        {
            auto& info = infoList[i];
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("applicationId",
                rapidjson::Value(devmenuUtil::GetBit64String(info.applicationId.value).data, allocator), allocator);

            char tmp[64];
            util::TSNPrintf(tmp, sizeof(tmp), "%016llx-%016llx", info.uid._data[0], info.uid._data[1]);
            item.AddMember("uid", rapidjson::Value(tmp, allocator), allocator);
            item.AddMember("result", rapidjson::Value(devmenuUtil::GetBit32String(info.lastResult.GetInnerValueForDebug()).data, allocator), allocator);
            item.AddMember("kind", rapidjson::Value(GetTaskKindString(info.kind), allocator), allocator);
            item.AddMember("retryable", rapidjson::Value((info.isRetryable)), allocator);
            item.AddMember("registered", rapidjson::Value(devmenuUtil::GetDateTimeString(tmp, sizeof(tmp), info.registeredTime), allocator), allocator);

            list.PushBack(item, allocator);
        }

        document.AddMember("list", list, allocator);

        buffer.Clear();
        rapidjson::PrettyWriter<rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    rapidjson::StringBuffer& GenerateDataInfoListJson(rapidjson::StringBuffer& buffer, const olsc::DataInfo infoList[], int count)
    {
        rapidjson::Document document;
        document.SetObject();
        auto& allocator = document.GetAllocator();

        document.AddMember("count", rapidjson::Value(count), allocator);

        rapidjson::Value list(rapidjson::kArrayType);

        for (int i = 0; i < count; ++i)
        {
            auto& info = infoList[i];
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("id", rapidjson::Value(info.id), allocator);
            item.AddMember("applicationId",
                rapidjson::Value(devmenuUtil::GetBit64String(info.appId.value).data, allocator), allocator);

            char tmp[64];
            util::TSNPrintf(tmp, sizeof(tmp), "%016llx-%016llx", info.uid._data[0], info.uid._data[1]);
            item.AddMember("uid", rapidjson::Value(tmp, allocator), allocator);
            item.AddMember("uploaded", rapidjson::Value(devmenuUtil::GetDateTimeString(tmp, sizeof(tmp), info.uploadedTime), allocator), allocator);
            item.AddMember("updated", rapidjson::Value(devmenuUtil::GetDateTimeString(tmp, sizeof(tmp), info.updatedTime), allocator), allocator);
            item.AddMember("size", rapidjson::Value(info.size), allocator);
            item.AddMember("status", rapidjson::Value(GetDataStatusString(info.status), allocator), allocator);
            list.PushBack(item, allocator);
        }

        document.AddMember("list", list, allocator);

        buffer.Clear();
        rapidjson::PrettyWriter<rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    rapidjson::StringBuffer& GenerateSeriesInfoJson(rapidjson::StringBuffer& buffer, const olsc::srv::SeriesInfo si)
    {
        rapidjson::Document document;
        document.SetObject();
        auto& allocator = document.GetAllocator();

        std::stringstream seriesId;
        seriesId << std::hex << std::setw(16) << std::setfill('0') << si.seriesId;
        document.AddMember("seriesId", rapidjson::Value(seriesId.str().c_str(), allocator), allocator);

        std::stringstream commitId;
        commitId << std::hex << std::setw(16) << std::setfill('0') << si.commitId;
        document.AddMember("commitId", rapidjson::Value(commitId.str().c_str(), allocator), allocator);

        buffer.Clear();
        rapidjson::PrettyWriter<rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    const char* GetDataDownloadRestrictionString(olsc::DataDownloadRestriction dr)
    {
        switch (dr)
        {
        case nn::olsc::DataDownloadRestriction::None:
            return "None";
        case nn::olsc::DataDownloadRestriction::OnlyRepairProcess:
            return "OnlyRepairProcess";
        case nn::olsc::DataDownloadRestriction::Restricted:
            return "Restricted";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* GetDataUploadRestrictionString(olsc::DataUploadRestriction du)
    {
        switch (du)
        {
        case nn::olsc::DataUploadRestriction::None:
            return "None";
        case nn::olsc::DataUploadRestriction::Restricted:
            return "Restricted";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    rapidjson::StringBuffer& GenerateDataTransferPolicyJson(rapidjson::StringBuffer& buffer, const olsc::DataTransferPolicy& policy)
    {
        rapidjson::Document document;
        document.SetObject();
        auto& allocator = document.GetAllocator();

        document.AddMember("DataDownloadRestriction", rapidjson::Value(GetDataDownloadRestrictionString(policy.downloadRestriction), allocator), allocator);

        document.AddMember("DataUploadRestriction", rapidjson::Value(GetDataUploadRestrictionString(policy.uploadRestriction), allocator), allocator);

        buffer.Clear();
        rapidjson::PrettyWriter<rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    bool ListTransferTask(const Option&)
    {
        olsc::TransferTaskListController tc;
        olsc::OpenTransferTaskListController(&tc);

        auto taskCount = tc.GetTransferTaskCount();
        std::unique_ptr<olsc::TransferTaskInfo[]> buffer(new olsc::TransferTaskInfo[taskCount]);

        auto getCount = tc.ListTransferTask(buffer.get(), taskCount, 0);
        NN_ABORT_UNLESS(taskCount == getCount);

        rapidjson::StringBuffer jsonString;
        GenerateTransferTaskInfoListJson(jsonString, buffer.get(), taskCount);
        NN_LOG("%s\n", jsonString.GetString());

        return true;
    }

    bool ListTransferTaskErrorInfo(const Option&)
    {
        auto maxCount = olsc::GetTransferTaskErrorInfoCount();

        std::unique_ptr<olsc::TransferTaskErrorInfo[]> buffer(new olsc::TransferTaskErrorInfo[maxCount]);
        auto listedCount = olsc::ListTransferTaskErrorInfo(buffer.get(), maxCount, 0);

        NN_ABORT_UNLESS(listedCount == maxCount);

        rapidjson::StringBuffer jsonString;
        GenerateTransferTaskErrorInfoListJson(jsonString, buffer.get(), maxCount);
        NN_LOG("%s\n", jsonString.GetString());
        return true;
    }

    bool RegisterTransferTaskErrorInfo(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);
            return false;
        }

        olsc::TransferTaskErrorInfo errorInfo = {};
        errorInfo.uid = *uid;

        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            NN_LOG("--app option is required\n");
            return false;
        }
        errorInfo.applicationId = { STR_TO_ULL(appIdString, nullptr, 16) };

        std::string kindString = option.GetValue("--kind");
        if (kindString.length() == 0)
        {
            NN_LOG("--kind option is required\n");
            return false;
        }
        auto kind = ConvertToTransferTaskKind(kindString);
        if (!kind)
        {
            NN_LOG("--kind option must be 'ul' or 'dl'\n");
            return false;
        }
        errorInfo.kind = *kind;

        auto retryableString = option.GetValue("--retryable");
        if (retryableString == nullptr)
        {
            NN_LOG("--retryable option is required\n");
            return false;
        }
        auto retryable = ConvertToBoolean(retryableString);
        if (!retryable)
        {
            NN_LOG("--retryable option must be 'true' or 'false'\n");
        }
        errorInfo.isRetryable = *retryable;

        auto resultString = option.GetValue("--result");
        if (resultString == nullptr)
        {
            NN_LOG("--result option is required\n");
            return false;
        }
        errorInfo.lastResult = result::detail::ConstructResult(std::strtoul(resultString, nullptr, 16));

        olsc::RegisterTransferTaskErrorInfo(errorInfo);
        return true;
    }

    bool DeleteAllTransferTaskErrorInfo(const Option&)
    {
        olsc::DeleteAllTransferTaskErrorInfo();
        return true;
    }

    bool DeleteTransferTaskErrorInfo(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);
            return false;
        }

        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            NN_LOG("--app option is required");
            return false;
        }
        ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };

        olsc::RemoveTransferTaskErrorInfo(*uid, appId);
        return true;
    }

    bool RegisterUploadTask(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);
            return false;
        }

        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            NN_LOG("--app option is required");
            return false;
        }
        ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };

        olsc::RemoteStorageController rc;
        olsc::OpenRemoteStorageController(&rc, *uid);

        olsc::TransferTaskId taskId;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(rc.RegisterUploadSaveDataTransferTask(&taskId, appId));

        NN_LOG("Upload task registered.\n");
        NN_LOG("  TaskId: %016llx.\n", taskId);

        return true;
    }

    bool RegisterDownloadTask(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);
            return false;
        }

        auto dataIdString = option.GetValue("--data");
        if (!dataIdString)
        {
            NN_LOG("--data option is required");
            return false;
        }

        olsc::TransferTaskId taskId;
        olsc::DataId dataId = { STR_TO_ULL(dataIdString, nullptr, 16) };
        olsc::RemoteStorageController rc;
        olsc::OpenRemoteStorageController(&rc, *uid);
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(rc.RegisterDownloadSaveDataTransferTask(&taskId, dataId));

        NN_LOG("Download task registered.\n");
        NN_LOG("  TaskId: %016llx.\n", taskId);
        return true;
    }

    void PrintTransferTaskProgress(const olsc::TransferTaskProgress& progress) NN_NOEXCEPT
    {
        NN_LOG("TotalSize    : %zu\n", progress.totalSize);
        NN_LOG("CompletedSize: %zu\n", progress.completedSize);
        NN_LOG("Throughput   : %zu\n", progress.throughput);
    }

    bool GetTransferTaskProgress(const Option& option)
    {
        auto ttidString = option.GetValue("--task");
        if (ttidString == nullptr)
        {
            NN_LOG("--task option is required");
            return false;
        }
        olsc::TransferTaskId taskId = STR_TO_ULL(ttidString, nullptr, 16);

        olsc::TransferTaskListController tc;
        olsc::OpenTransferTaskListController(&tc);

        olsc::TransferTaskProgress progress;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(tc.GetTransferTaskProgress(&progress, taskId));

        PrintTransferTaskProgress(progress);
        return true;
    }

    bool DeleteTransferTask(const Option& option)
    {
        auto ttidString = option.GetValue("--task");
        if (ttidString == nullptr)
        {
            NN_LOG("--task option is required");
            return false;
        }
        olsc::TransferTaskId taskId = STR_TO_ULL(ttidString, nullptr, 16);

        olsc::TransferTaskListController tc;
        olsc::OpenTransferTaskListController(&tc);

        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(tc.DeleteTransferTask(taskId));

        return true;
    }

    bool ListDataInfoCache(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        olsc::RemoteStorageController rc;
        olsc::OpenRemoteStorageController(&rc, *uid);

        auto cacheCount = rc.GetDataInfoCount();

        std::unique_ptr<olsc::DataInfo[]> listBuffer(new olsc::DataInfo[cacheCount]);
        auto listedCount = rc.ListDataInfo(listBuffer.get(), cacheCount, 0);

        rapidjson::StringBuffer jsonString;
        GenerateDataInfoListJson(jsonString, listBuffer.get(), listedCount);
        NN_LOG("%s\n", jsonString.GetString());

        return true;
    }

    bool UpdateDataInfoCache(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        olsc::RemoteStorageController rc;
        olsc::OpenRemoteStorageController(&rc, *uid);
        olsc::AsyncRequest request;

        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(rc.RequestUpdateDataInfoCacheAsync(&request));
        }
        else
        {
            ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };
            NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(rc.RequestUpdateDataInfoCacheAsync(&request, appId));
        }

        os::SystemEvent requestEvent;
        request.GetEvent(&requestEvent);
        if (!requestEvent.TimedWait(nn::TimeSpan::FromSeconds(30)))
        {
            NN_LOG("Request timeout\n");
            return false;
        }

        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(request.GetResult());
        return true;
    }

    bool GetSeriesInfo(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            NN_LOG("--app option is required");
            return false;
        }

        ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };

        olsc::srv::SeriesInfo si;
        olsc::GetSeriesInfo(&si, *uid, appId);

        rapidjson::StringBuffer jsonString;
        GenerateSeriesInfoJson(jsonString, si);
        NN_LOG("%s\n", jsonString.GetString());

        return true;
    }

    bool DeleteSeriesInfo(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            olsc::DeleteAllSeriesInfo(*uid);
        }
        else
        {
            ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };
            olsc::DeleteSeriesInfo(*uid, appId);
        }

        return true;
    }

    bool DeleteSdaInfoCache(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            olsc::DeleteAllSaveDataArchiveInfoCache(*uid);
            return true;
        }
        else
        {
            ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };
            auto rc = olsc::OpenRemoteStorageController(*uid);
            olsc::DataInfo di;
            if (rc.GetDataInfo(&di, appId))
            {
                rc.DeleteDataInfo(di.id);
            }

            return true;
        }

    }

    bool DeleteSdaInfo(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        auto dataIdString = option.GetValue("--data");
        if (dataIdString == nullptr)
        {
            NN_LOG("--data option is required");
            return false;
        }

        olsc::DataId dataId = STR_TO_ULL(dataIdString, nullptr, 10);

        auto rc = olsc::OpenRemoteStorageController(*uid);

        olsc::AsyncRequest asyncResult;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(rc.RequestDeleteData(&asyncResult, dataId));
        os::SystemEvent asyncEvent;
        asyncResult.GetEvent(&asyncEvent);
        asyncEvent.Wait();

        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(asyncResult.GetResult());
        return true;
    }

    bool SetAutoUploadSetting(const Option& option, bool isEnabled)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        auto dc = olsc::OpenDaemonController();
        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            dc.SetGlobalAutoUploadSetting(*uid, isEnabled);
        }
        else
        {
            ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };
            dc.SetApplicationAutoUploadSetting(*uid, appId, isEnabled);
        }

        return true;
    }

    bool EnableAutoUploadSetting(const Option& option)
    {
        return SetAutoUploadSetting(option, true);
    }

    bool DisableAutoUploadSetting(const Option& option)
    {
        return SetAutoUploadSetting(option, false);
    }

    bool GetAutoUploadSetting(const Option& option)
    {
        auto userIndexString = option.GetValue("--user");
        if (userIndexString == nullptr)
        {
            NN_LOG("--user option is required");
            return false;
        }
        auto userIndex = std::strtol(userIndexString, nullptr, 10);

        util::optional<account::Uid> uid;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(GetUidFromIndex(&uid, userIndex));

        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            NN_LOG("--app option is required\n");
            return false;
        }
        ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };

        auto dc = olsc::OpenDaemonController();
        NN_LOG("%s\n", dc.IsAutoUploadEnabled(*uid, appId) ? "True" : "False");

        return true;
    }

    bool UpdateDataTransferPolicyCache(const Option& option)
    {
        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            NN_LOG("--app option is required\n");
            return false;
        }
        ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };

        olsc::AsyncRequest request;
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(olsc::RequestUpdateDataTransferPolicy(&request, appId));

        os::SystemEvent requestEvent;
        request.GetEvent(&requestEvent);
        requestEvent.Wait();
        NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED(request.GetResult());

        return true;
    }

    bool GetDataTransferPolicyCache(const Option& option)
    {
        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            NN_LOG("--app option is required\n");
            return false;
        }
        ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };

        olsc::DataTransferPolicy policy;
        if (!olsc::GetDataTransferPolicy(&policy, appId))
        {
            return false;
        }

        rapidjson::StringBuffer jsonString;
        GenerateDataTransferPolicyJson(jsonString, policy);
        NN_LOG("%s\n", jsonString.GetString());
        return true;
    }

    bool DeleteDataTransferPolicyCache(const Option& option)
    {
        auto appIdString = option.GetValue("--app");
        if (appIdString == nullptr)
        {
            NN_LOG("--app option is required\n");
            return false;
        }
        ApplicationId appId = { STR_TO_ULL(appIdString, nullptr, 16) };

        olsc::DeleteDataTransferPolicy(appId);

        return true;
    }

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " olsc register-upload-task --user <account_index> --app <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " olsc register-download-task --user <account_index> --data <data_id>\n"
        "       " DEVMENUCOMMAND_NAME " olsc list-transfer-task\n"
        "       " DEVMENUCOMMAND_NAME " olsc delete-transfer-task --task <transfer_task_id>\n"
        "       " DEVMENUCOMMAND_NAME " olsc delete-all-transfer-task\n"
        "       " DEVMENUCOMMAND_NAME " olsc update-datainfo-cache --user <account_index> [--app <application_id>]\n"
        "       " DEVMENUCOMMAND_NAME " olsc list-datainfo-cache --user <account_index>\n"
        "       " DEVMENUCOMMAND_NAME " olsc get-series-info --user <account_index> --app <application_id>"
        "       " DEVMENUCOMMAND_NAME " olsc delete-series-info --user <account_index> [--app <application_id>]\n"
        "       " DEVMENUCOMMAND_NAME " olsc delete-sda-info-cache --user <account_index> [--app <application_id>]\n"
        "       " DEVMENUCOMMAND_NAME " olsc delete-sda-info --user <account_index> --data <data_id>\n"
        "       " DEVMENUCOMMAND_NAME " olsc list-transfer-task-error-info\n"
        "       " DEVMENUCOMMAND_NAME " olsc register-transfer-task-error-info --user <account_index> --app <application_id> --kind <task_kind> --retryable <retryable> --result <result_value>\n"
        "       " DEVMENUCOMMAND_NAME " olsc delete-transfer-task-error-info --user <account_index> --app <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " olsc delete-all-transfer-task-error-info\n"
        "       " DEVMENUCOMMAND_NAME " olsc enable-auto-upload-setting --user <account_index> [--app <application_id>]\n"
        "       " DEVMENUCOMMAND_NAME " olsc disable-auto-upload-setting --user <account_index> [--app <application_id>]\n"
        "       " DEVMENUCOMMAND_NAME " olsc list-auto-upload-setting --user <account_index> --app <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " olsc update-data-transfer-policy-cache --app <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " olsc get-data-transfer-policy-cache --app <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " olsc delete-data-transfer-policy-cache --app <application_id>\n"
        "       "
        ""; // 終端

    const SubCommand g_SubCommands[] =
    {
        { "register-upload-task",                           RegisterUploadTask                          },
        { "register-download-task",                         RegisterDownloadTask                        },
        { "delete-all-transfer-task",                       DeleteAllTransferTask                       },
        { "list-transfer-task",                             ListTransferTask                            },
        { "get-transfer-task-progress",                     GetTransferTaskProgress                     },
        { "delete-transfer-task",                           DeleteTransferTask                          },
        { "update-datainfo-cache",                          UpdateDataInfoCache                         },
        { "list-datainfo-cache",                            ListDataInfoCache                           },
        { "delete-series-info",                             DeleteSeriesInfo                            },
        { "delete-sda-info-cache",                          DeleteSdaInfoCache                          },
        { "list-transfer-task-error-info",                  ListTransferTaskErrorInfo                   },
        { "register-transfer-task-error-info",              RegisterTransferTaskErrorInfo               },
        { "delete-transfer-task-error-info",                DeleteTransferTaskErrorInfo                 },
        { "delete-all-transfer-task-error-info",            DeleteAllTransferTaskErrorInfo              },
        { "enable-auto-upload-setting",                     EnableAutoUploadSetting                     },
        { "disable-auto-upload-setting",                    DisableAutoUploadSetting                    },
        { "get-auto-upload-setting",                        GetAutoUploadSetting                        },
        { "get-series-info",                                GetSeriesInfo                               },
        { "delete-sda-info",                                DeleteSdaInfo                               },
        { "update-data-transfer-policy-cache",              UpdateDataTransferPolicyCache               },
        { "get-data-transfer-policy-cache",                 GetDataTransferPolicyCache                  },
        { "delete-data-transfer-policy-cache",              DeleteDataTransferPolicyCache               },
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result OlscCommand(bool* outValue, const Option& option)
{
    account::InitializeForAdministrator();
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::InitializeForSystem());
    NN_UTIL_SCOPE_EXIT{ time::Finalize(); };

    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            *outValue = subCommand.function(option);
            NN_RESULT_SUCCESS;
        }
    }

    NN_LOG("'%s' is not a DevMenu olsc command. See '" DEVMENUCOMMAND_NAME " olsc --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}

#undef NN_DEVMENUCOMMAND_OLSC_RETURN_FALSE_IF_FAILED
