﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_StaticAssert.h>
#include <nn/account.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/nd/nd_ApiForDebug.h>
#include <nn/nd/nd_ApiForSystem.h>
#include <nn/nd/detail/nd_DataConverter.h>
#include <nn/ndd.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_Optional.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_NeighborDetectionCommand.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"

using namespace nn;

namespace
{
    const int NameFormatLength = 20;

    util::optional<account::Uid> GetUserAccountId(int index) NN_NOEXCEPT
    {
        account::InitializeForAdministrator();
        int count;
        account::Uid users[account::UserCountMax];
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&count, users, NN_ARRAY_SIZE(users)));
        if( index >= count )
        {
            DEVMENUCOMMAND_LOG("The number of registered accounts is %d. Index %d is out of range.\n", count, index);
            return nullptr;
        }
        return users[index];
    }

    util::optional<account::Uid> GetUserAccountId(const Option& option) NN_NOEXCEPT
    {
        auto indexStr = option.GetValue("--account");
        if( indexStr == nullptr || std::strlen(indexStr) == 0 )
        {
            DEVMENUCOMMAND_LOG("Provide index value for --account option.\n");
            return nullptr;
        }
        char* endPtr;
        int index = static_cast<int>(std::strtoull(indexStr, &endPtr, 10));
        if( *endPtr != '\0' )
        {
            DEVMENUCOMMAND_LOG("\"%s\" is invalid for --account option. Provide a number.\n", indexStr);
            return nullptr;
        }
        return GetUserAccountId(index);
    }

    void DumpSendDataDescription(const ndd::SendDataDescription& data) NN_NOEXCEPT
    {
        DEVMENUCOMMAND_LOG("[SendData]\n");
        DEVMENUCOMMAND_LOG("    %-*s : ", NameFormatLength, "DataId");
        for( size_t i = 0; i < NN_ARRAY_SIZE(ndd::DataId::raw); i++ )
        {
            DEVMENUCOMMAND_LOG("%02x", data.dataId.raw[i]);
        }
        DEVMENUCOMMAND_LOG("\n");
        DEVMENUCOMMAND_LOG("    %-*s : %u\n", NameFormatLength, "Data Size", data.dataSize);
        auto pPayload = reinterpret_cast<const nd::detail::Payload*>(data.data);
        DEVMENUCOMMAND_LOG("    %-*s : %u\n", NameFormatLength, "Version", pPayload->version);
        DEVMENUCOMMAND_LOG("    %-*s : %s\n", NameFormatLength, "Language", pPayload->languageCode.string);
        util::Uuid uuid;
        std::memcpy(uuid.data, pPayload->localUserId._data, sizeof(uuid.data));
        char idStr[util::Uuid::StringSize];
        DEVMENUCOMMAND_LOG("    %-*s : %s\n", NameFormatLength, "LocalUserId", uuid.ToString(idStr, sizeof(idStr)));
        if( pPayload->hasNetworkUserId )
        {
            DEVMENUCOMMAND_LOG("    %-*s : 0x%016llx\n", NameFormatLength, "NetworkUserId", pPayload->networkUserId._id);
        }
        else
        {
            DEVMENUCOMMAND_LOG("    %-*s : None\n", NameFormatLength, "NetworkUserId");
        }
        DEVMENUCOMMAND_LOG("    %-*s : %s\n", NameFormatLength, "Nickname", pPayload->nickName.name);
        DEVMENUCOMMAND_LOG("    %-*s : Count = %d\n", NameFormatLength, "Recently Played App", pPayload->recentlyPlayedApplicationCount);
        for( int i = 0; i < pPayload->recentlyPlayedApplicationCount; i++ )
        {
            DEVMENUCOMMAND_LOG("    %-*s   [%d] 0x%016llx\n", NameFormatLength, "", i, pPayload->recentlyPlayedApplication[i]);
        }
        DEVMENUCOMMAND_LOG("    %-*s : Length = %zu\n", NameFormatLength, "SystemData", pPayload->systemDataSize);
    }

    void ShowRebootLog() NN_NOEXCEPT
    {
        DEVMENUCOMMAND_LOG("*** Reboot the target to allow the change to take effect.\n");
    }

    bool IsServiceEnabled() NN_NOEXCEPT
    {
        bool enabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&enabled, sizeof(enabled), "nd", "service_enabled");
        if( size != sizeof(enabled) )
        {
            return false;
        }
        return enabled;
    }

    bool IsDevelopmentFunctionEnabled() NN_NOEXCEPT
    {
        bool enabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&enabled, sizeof(enabled), "nd", "development_function_enabled");
        if( size != sizeof(enabled) )
        {
            return false;
        }
        return enabled;
    }

    int32_t GetSendDataIdRefreshIntervalSeconds() NN_NOEXCEPT
    {
        int32_t value;
        auto size = settings::fwdbg::GetSettingsItemValue(&value, sizeof(value), "nd", "send_data_id_refresh_interval_seconds");
        if( size != sizeof(value) )
        {
            // fwdbg設定が読み込めなかった場合のデフォルト値は8h。参考：Programs\Iris\Sources\Libraries\nd\detail\nd_FirmwareDebugSettings.cpp
            return static_cast<int32_t>(nn::TimeSpan::FromHours(8).GetSeconds());
        }
        return value;
    }

    bool InfoCommand(const Option&) NN_NOEXCEPT
    {
        if( !IsServiceEnabled() )
        {
            DEVMENUCOMMAND_LOG("Neighbor Detecition service is disabled. Use 'nd enable-service' command to enable the service.\n");
            return true;
        }
        ndd::Initialize();
        nd::InitializeForSystem();
        DEVMENUCOMMAND_LOG("[General]\n");
        DEVMENUCOMMAND_LOG("    %-*s : %s\n", NameFormatLength, "Development Function", IsDevelopmentFunctionEnabled() ? "Enabled" : "Disabled");
        DEVMENUCOMMAND_LOG("    %-*s : %s\n", NameFormatLength, "Auto Communication", ndd::IsAutoCommunicationEnabled() ? "Enabled" : "Disabled");
        DEVMENUCOMMAND_LOG("    %-*s : %s\n", NameFormatLength, "Network", ndd::IsNetworkActive() ? "Active" : "Inactive");
        DEVMENUCOMMAND_LOG("    %-*s : Oldest = %u, Current = %u, Next = %u\n", NameFormatLength, "Receive Data Counter",
            ndd::GetOldestReceiveDataCounter(), ndd::GetCurrentReceiveDataCounter(), ndd::GetNextReceiveDataCounter());
        DEVMENUCOMMAND_LOG("    %-*s : Every %d seconds\n", NameFormatLength, "Send Data Id Refresh", GetSendDataIdRefreshIntervalSeconds());
        DEVMENUCOMMAND_LOG("    %-*s : ", NameFormatLength, "Sender");
        account::Uid uid;
        auto exists = nd::GetSender(&uid);
        if( !exists )
        {
            DEVMENUCOMMAND_LOG("None\n");
            return true;
        }
        DEVMENUCOMMAND_LOG("Uid<0x%016llx-0x%016llx>\n", uid._data[0], uid._data[1]);
        ndd::SendDataDescription sendData;
        auto sendDataCount = ndd::GetSendData(&sendData, 0, 1);
        if( sendDataCount == 0 )
        {
            return true;
        }
        DumpSendDataDescription(sendData);
        return true;
    }

    bool AccountInfoCommand(const Option& option) NN_NOEXCEPT
    {
        auto perAccountFunc = [](const account::Uid& uid)
        {
            DEVMENUCOMMAND_LOG("Information of Uid<0x%016llx-0x%016llx>\n", uid._data[0], uid._data[1]);
            DEVMENUCOMMAND_LOG("[General]\n");
            account::Uid sender;
            auto senderExists = nd::GetSender(&sender);
            DEVMENUCOMMAND_LOG("    %-*s : %s\n", NameFormatLength, "Is Current Sender", (senderExists && sender == uid) ? "Yes" : "No");
            ndd::SendDataDescription sendData;
            nd::GetSendDataForDebug(&sendData, uid);
            DumpSendDataDescription(sendData);
        };

        if( option.HasKey("--account") )
        {
            auto uid = GetUserAccountId(option);
            if( !uid )
            {
                return false;
            }
            account::InitializeForAdministrator();
            nd::InitializeForSystem();
            perAccountFunc(*uid);
            return true;
        }
        else
        {
            account::InitializeForAdministrator();
            nd::InitializeForSystem();
            int count;
            account::Uid users[account::UserCountMax];
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&count, users, NN_ARRAY_SIZE(users)));
            for( int i = 0; i < count; i++ )
            {
                perAccountFunc(users[i]);
                DEVMENUCOMMAND_LOG("--------------------------------------------------\n");
            }
            return true;
        }
    }

    bool EnableServiceCommand(const Option& option) NN_NOEXCEPT
    {
        bool enable = true;
        if(option.HasKey("--ndd-only"))
        {
            DEVMENUCOMMAND_LOG("ndd-only\n");
            settings::fwdbg::SetSettingsItemValue("ndd", "service_enabled", &enable, sizeof(enable));
        }
        else
        {
            settings::fwdbg::SetSettingsItemValue("nd", "service_enabled", &enable, sizeof(enable));
            settings::fwdbg::SetSettingsItemValue("ndd", "service_enabled", &enable, sizeof(enable));
        }
        ShowRebootLog();
        return true;
    }

    bool DisableServiceCommand(const Option& option) NN_NOEXCEPT
    {
        bool enable = false;
        if(option.HasKey("--ndd-only"))
        {
            DEVMENUCOMMAND_LOG("ndd-only\n");
            settings::fwdbg::SetSettingsItemValue("ndd", "service_enabled", &enable, sizeof(enable));
        }
        else
        {
            settings::fwdbg::SetSettingsItemValue("nd", "service_enabled", &enable, sizeof(enable));
            settings::fwdbg::SetSettingsItemValue("ndd", "service_enabled", &enable, sizeof(enable));
        }
        ShowRebootLog();
        return true;
    }

    bool EnableSetAutoCommunicationCommand(const Option&) NN_NOEXCEPT
    {
        ndd::Initialize();
        ndd::EnableAutoCommunication();
        return true;
    }

    bool DisableSetAutoCommunicationCommand(const Option&) NN_NOEXCEPT
    {
        ndd::Initialize();
        ndd::DisableAutoCommunication();
        return true;
    }

    bool EnableDevelopmentFunctionCommand(const Option&) NN_NOEXCEPT
    {
        bool enable = true;
        settings::fwdbg::SetSettingsItemValue("nd", "development_function_enabled", &enable, sizeof(enable));
        ShowRebootLog();
        return true;
    }

    bool DisableDevelopmentFunctionCommand(const Option&) NN_NOEXCEPT
    {
        bool enable = false;
        settings::fwdbg::SetSettingsItemValue("nd", "development_function_enabled", &enable, sizeof(enable));
        ShowRebootLog();
        return true;
    }

    bool SetSenderCommand(const Option& option) NN_NOEXCEPT
    {
        auto uid = GetUserAccountId(option);
        if( !uid )
        {
            return false;
        }
        nd::InitializeForSystem();
        nd::SetSender(*uid);
        return true;
    }

    bool SetSystemDataCommand(const Option& option) NN_NOEXCEPT
    {
        auto uid = GetUserAccountId(option);
        if( !uid )
        {
            return false;
        }

        auto target = option.GetTarget();
        size_t targetLength = (target == nullptr) ? 0u : std::strlen(target);
        if( targetLength == 0u )
        {
            DEVMENUCOMMAND_LOG("<data> is empty.\n");
            return false;
        }
        if( !option.HasKey("--format") )
        {
            DEVMENUCOMMAND_LOG("--format option is required.\n");
            return false;
        }

        auto format = std::string(option.GetValue("--format"));
        if( format == "string" )
        {
            if( targetLength > nd::SystemDataSizeMax )
            {
                DEVMENUCOMMAND_LOG("The length of <data> must be less than or equal to %zu.\n", nd::SystemDataSizeMax);
                return false;
            }
            nd::InitializeForSystem();
            nd::SetSystemData(*uid, target, std::min(nd::SystemDataSizeMax, targetLength));
            return true;
        }
        else if( format == "base64" )
        {
            Bit8 data[nd::SystemDataSizeMax];
            size_t dataLength;
            auto status = util::Base64::FromBase64String(&dataLength, data, sizeof(data), target, util::Base64::Mode_Normal);
            switch( status )
            {
            case util::Base64::Status::Status_Success:
                {
                    nd::InitializeForSystem();
                    nd::SetSystemData(*uid, data, dataLength);
                }
                return true;
            case util::Base64::Status::Status_BufferFull:
                {
                    DEVMENUCOMMANDSYSTEM_LOG("Failed to decode <data> due to BufferFull. <data> is too long.\n");
                }
                return false;
            case util::Base64::Status::Status_BadData:
                {
                    DEVMENUCOMMANDSYSTEM_LOG("Failed to decode <data> due to BadData.\n");
                }
                return false;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        else
        {
            DEVMENUCOMMAND_LOG("\"%s\" is invalid for --format option. Please provide \"base64\" or \"string\".\n", format.c_str());
            return false;
        }
    }

    bool SetSendDataIdRefreshIntervalCommand(const Option& option) NN_NOEXCEPT
    {
        auto target = option.GetTarget();
        char* endPtr;
        int seconds = static_cast<int>(std::strtoull(target, &endPtr, 10));
        if( *endPtr != '\0' || seconds < 10 )
        {
            DEVMENUCOMMAND_LOG("\"%s\" is invalid for <seconds>. Provide a number larger than or equal to 10.\n", target);
            return false;
        }
        settings::fwdbg::SetSettingsItemValue("nd", "send_data_id_refresh_interval_seconds", &seconds, sizeof(seconds));
        ShowRebootLog();
        return true;
    }

    bool ClearSystemDataCommand(const Option& option) NN_NOEXCEPT
    {
        auto uid = GetUserAccountId(option);
        if( !uid )
        {
            return false;
        }
        nd::InitializeForSystem();
        nd::ClearSystemData(*uid);
        return true;
    }

    bool ClearReceiveDataCommand(const Option&) NN_NOEXCEPT
    {
        ndd::Initialize();
        ndd::ClearReceiveData();

        nd::InitializeForSystem();
        account::InitializeForAdministrator();
        int count;
        account::Uid users[account::UserCountMax];
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&count, users, NN_ARRAY_SIZE(users)));
        for( int i = 0; i < count; i++ )
        {
            nd::ClearReceiveCounterForDebug(users[i]);
        }
        return true;
    }

    bool ClearReceiveDataFilterCommand(const Option&) NN_NOEXCEPT
    {
        ndd::Initialize();
        ndd::ClearDataIdFilter();
        return true;
    }

    struct SubCommand
    {
        std::string name;
        bool(*function)(const Option&);
    };

    const char HelpMessage[] =
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "usage: " DEVMENUCOMMAND_NAME " nd info\n"
        "       " DEVMENUCOMMAND_NAME " nd account-info [--account <index>]\n"
        "       " DEVMENUCOMMAND_NAME " nd enable-service\n"
        "       " DEVMENUCOMMAND_NAME " nd disable-service\n"
        "       " DEVMENUCOMMAND_NAME " nd enable-auto-communication\n"
        "       " DEVMENUCOMMAND_NAME " nd disable-auto-communication\n"
        "       " DEVMENUCOMMAND_NAME " nd enable-development-function\n"
        "       " DEVMENUCOMMAND_NAME " nd disable-development-function\n"
        "       " DEVMENUCOMMAND_NAME " nd set-sender --account <index>\n"
        "       " DEVMENUCOMMAND_NAME " nd set-system-data --account <index> --format <string|base64> <data>\n"
        "       " DEVMENUCOMMAND_NAME " nd set-send-data-id-refresh-interval <seconds>\n"
        "       " DEVMENUCOMMAND_NAME " nd clear-system-data --account <index>\n"
        "       " DEVMENUCOMMAND_NAME " nd clear-receive-data\n"
        "       " DEVMENUCOMMAND_NAME " nd clear-receive-data-filter\n"
#endif
        ""; // 終端

    const SubCommand g_SubCommands[] =
    {
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "info", InfoCommand },
        { "account-info", AccountInfoCommand },
        { "enable-service", EnableServiceCommand },
        { "disable-service", DisableServiceCommand },
        { "enable-auto-communication", EnableSetAutoCommunicationCommand },
        { "disable-auto-communication", DisableSetAutoCommunicationCommand },
        { "enable-development-function", EnableDevelopmentFunctionCommand },
        { "disable-development-function", DisableDevelopmentFunctionCommand },
        { "set-sender", SetSenderCommand },
        { "set-system-data", SetSystemDataCommand },
        { "set-send-data-id-refresh-interval", SetSendDataIdRefreshIntervalCommand },
        { "clear-system-data", ClearSystemDataCommand },
        { "clear-receive-data", ClearReceiveDataCommand },
        { "clear-receive-data-filter", ClearReceiveDataFilterCommand },
#endif
    };
}

#include <limits>

Result NeighborDetectionCommand(bool* outValue, const Option& option) NN_NOEXCEPT
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            *outValue = subCommand.function(option);
            NN_RESULT_SUCCESS;
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu nd command. See '" DEVMENUCOMMAND_NAME " nd --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
