﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/time.h>

#include "DevMenuCommand_Eula.h"

#define EULA_TRACE NN_LOG

namespace
{
    //!< DevMenu での同意バージョン.本来の経由での同意と区別できるようプリインより小さい 0.1 を指定
    const uint16_t EulaVersionMajor = 0;
    const uint16_t EulaVersionMinor = 1;
    const uint32_t DevMenuEulaVersion = ((EulaVersionMajor << 16) | EulaVersionMinor);
}

//!< EULA 同意
void AcceptEulaInDevVersion() NN_NOEXCEPT
{
    nn::settings::system::RegionCode regionCode;
    nn::settings::system::GetRegionCode(&regionCode);

    std::unique_ptr<nn::settings::system::EulaVersion[]> pEulaVers(new nn::settings::system::EulaVersion[nn::settings::system::EulaVersionCountMax]);
    int count = nn::settings::system::GetEulaVersions(pEulaVers.get(), nn::settings::system::EulaVersionCountMax);

    for (int i = 0; i < count; ++i)
    {
        nn::settings::system::EulaVersion& eulaVer = pEulaVers[i];
        if (eulaVer.regionCode == regionCode)
        {
            uint16_t major = eulaVer.version >> 16;
            uint16_t minor = 0x0000FFFF & eulaVer.version;
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
            NN_UNUSED(major);
            NN_UNUSED(minor);
#endif
            EULA_TRACE("EULA has been already accepted. (v%d.%d)\n", major, minor);
            return;
        }
    }

    // pEulaVers の有効要素の後ろに追加する
    NN_ABORT_UNLESS_LESS(count, nn::settings::system::EulaVersionCountMax);

    nn::settings::system::EulaVersion& eulaVer = pEulaVers[count];
    eulaVer.version = DevMenuEulaVersion;
    eulaVer.regionCode = regionCode;
    eulaVer.clockType = nn::settings::system::EulaVersionClockType_SteadyClock;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&eulaVer.steadyClock));

    nn::settings::system::SetEulaVersions(pEulaVers.get(), count + 1);
    EULA_TRACE("Accept EULA(v%d.%d)\n", EulaVersionMajor, EulaVersionMinor);
}

//!< EULA 非同意
void RejectEulaOfDevVersion() NN_NOEXCEPT
{
    nn::settings::system::RegionCode regionCode;
    nn::settings::system::GetRegionCode(&regionCode);

    std::unique_ptr<nn::settings::system::EulaVersion[]> pCurrentEulaVers(new nn::settings::system::EulaVersion[nn::settings::system::EulaVersionCountMax]);
    const int currentCount = nn::settings::system::GetEulaVersions(pCurrentEulaVers.get(), nn::settings::system::EulaVersionCountMax);
    if (currentCount == 0)
    {
        EULA_TRACE("Noting accepted EULA.\n");
        return;
    }

    int writeCount = 0;
    std::unique_ptr<nn::settings::system::EulaVersion[]> pWriteEulaVers(new nn::settings::system::EulaVersion[currentCount]);

    for (int i = 0; i < currentCount; ++i)
    {
        nn::settings::system::EulaVersion& eulaVer = pCurrentEulaVers[i];
        // DevMenu で同意した version でないものだけ残す
        if (eulaVer.version != DevMenuEulaVersion)
        {
            pWriteEulaVers[writeCount++] = eulaVer;
        }
    }

    if (currentCount == writeCount)
    {
        EULA_TRACE("Noting accepted EULA(v%d.%d).\n", EulaVersionMajor, EulaVersionMinor);
        return;
    }

    nn::settings::system::SetEulaVersions(pWriteEulaVers.get(), writeCount);
    EULA_TRACE("Reject EULA(v%d.%d)\n", EulaVersionMajor, EulaVersionMinor);
}

