﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <stack>
#include <string>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/nn_Abort.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/time/time_PosixTime.h>
#include <nn/ns/ns_SdCardSystemApi.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/time.h>
#include <nn/util/util_StringView.h>
#include <nn/idle/idle_SystemApi.h>

#include "DevMenuCommand_Common.h"

using namespace nn;

namespace devmenuUtil
{

namespace
{
    nn::Result CopyFile(const char* destination, const char* source, void* workBuffer, size_t workBufferSize, bool force) NN_NOEXCEPT
    {
        nn::fs::FileHandle sourceHandle;
        NN_RESULT_DO(nn::fs::OpenFile(&sourceHandle, source, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile(sourceHandle); };

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, sourceHandle));

        NN_RESULT_TRY(nn::fs::CreateFile(destination, 0))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
                if (!force)
                {
                    NN_RESULT_RETHROW;
                }
            }
        NN_RESULT_END_TRY;

        nn::fs::FileHandle destinationHandle;
        NN_RESULT_DO(nn::fs::OpenFile(&destinationHandle, destination, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile(destinationHandle); };

        NN_RESULT_DO(nn::fs::SetFileSize(destinationHandle, fileSize));

        for (int64_t offset = 0; offset < fileSize;)
        {
            const auto copySize = std::min(workBufferSize, static_cast<size_t>(fileSize - offset));
            NN_RESULT_DO(nn::fs::ReadFile(sourceHandle, offset, workBuffer, copySize));
            NN_RESULT_DO(nn::fs::WriteFile(destinationHandle, offset, workBuffer, copySize, nn::fs::WriteOption()));
            offset += copySize;
        }

        NN_RESULT_DO(nn::fs::FlushFile(destinationHandle));

        NN_RESULT_SUCCESS;
    }

    nn::Result CopyFile(std::vector<std::unique_ptr<ErrorFileInfo>>* pOutErrorFileInfoList, const char* destination, const char* source, void* workBuffer, size_t workBufferSize, bool force, bool skipsErrorFile) NN_NOEXCEPT
    {
        auto result = CopyFile(destination, source, workBuffer, workBufferSize, force);
        if (result.IsFailure())
        {
            auto errorInfo = std::unique_ptr<ErrorFileInfo>(new ErrorFileInfo);
            util::Strlcpy(errorInfo->path, source, static_cast<int>(std::strlen(source)) + 1);
            errorInfo->result = result;
            pOutErrorFileInfoList->push_back(std::move(errorInfo));
        }

        NN_RESULT_TRY(result);
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
                if (!force)
                {
                    NN_RESULT_RETHROW;
                }
            }
            NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
            {
                if (!skipsErrorFile)
                {
                    NN_RESULT_RETHROW;
                }
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    nn::Result CreateDirectory(std::vector<std::unique_ptr<ErrorFileInfo>>* pOutErrorFileInfoList, const char* path, bool force, bool skipsErrorFile)
    {
        auto result = nn::fs::CreateDirectory(path);
        if (result.IsFailure())
        {
            auto errorInfo = std::unique_ptr<ErrorFileInfo>(new ErrorFileInfo);
            util::Strlcpy(errorInfo->path, path, static_cast<int>(std::strlen(path)) + 1);
            errorInfo->result = result;
            pOutErrorFileInfoList->push_back(std::move(errorInfo));
        }

        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
                if (!force)
                {
                    NN_RESULT_RETHROW;
                }
            }
            NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
            {
                if (!skipsErrorFile)
                {
                    NN_RESULT_RETHROW;
                }
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    nn::Result CopyDirectory(std::vector<std::unique_ptr<ErrorFileInfo>>* pOutErrorFileInfoList, const char* destination, const char* source, void* workBuffer, size_t workBufferSize, bool force, bool skipsErrorFile)
    {
        const int rootLength = nn::util::Strnlen(source, nn::fs::EntryNameLengthMax);
        NN_RESULT_DO(WalkDirectory(source, [&](const char* directoryPath, const nn::fs::DirectoryEntry& entry) -> nn::Result
        {
            char sourceBuffer[nn::fs::EntryNameLengthMax + 1];
            NN_RESULT_THROW_UNLESS(nn::util::SNPrintf(sourceBuffer, sizeof(sourceBuffer), "%s/%s", directoryPath, entry.name) < sizeof(sourceBuffer), nn::fs::ResultTooLongPath());

            char destinationBuffer[nn::fs::EntryNameLengthMax + 1];
            NN_RESULT_THROW_UNLESS(nn::util::SNPrintf(destinationBuffer, sizeof(destinationBuffer), "%s/%s", destination, sourceBuffer + rootLength + 1) < sizeof(destinationBuffer), nn::fs::ResultTooLongPath());

            if (entry.directoryEntryType == nn::fs::DirectoryEntryType_File)
            {
                NN_RESULT_DO(CopyFile(pOutErrorFileInfoList, destinationBuffer, sourceBuffer, workBuffer, workBufferSize, force, skipsErrorFile));
            }
            else
            {
                NN_RESULT_TRY(CreateDirectory(pOutErrorFileInfoList, destinationBuffer, force, skipsErrorFile))
                    NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                    {
                        if (!force)
                        {
                            NN_RESULT_RETHROW;
                        }
                    }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_SUCCESS;
        }));

        NN_RESULT_SUCCESS;
    }

    bool IsPathSeparator(char c)
    {
        return c == '/' || c == '\\';
    }

    nn::Result GetEntryNamePointer(const char** out, const char* path) NN_NOEXCEPT
    {
        int pathLength = nn::util::Strnlen(path, nn::fs::EntryNameLengthMax);
        NN_RESULT_THROW_UNLESS(pathLength <= nn::fs::EntryNameLengthMax && path[pathLength] == '\0', nn::fs::ResultTooLongPath());

        auto pCharacter = &path[pathLength - 1];
        for (; pCharacter >= path && !IsPathSeparator(*pCharacter); pCharacter--);

        NN_RESULT_THROW_UNLESS(pCharacter >= path, nn::fs::ResultInvalidPathFormat());

        *out = pCharacter + 1;

        NN_RESULT_SUCCESS;
    }

    nn::Result ConcatPath(int* outLength, char* front, const char* rear, size_t frontBufferSize) NN_NOEXCEPT
    {
        int frontLength = nn::util::Strnlen(front, nn::fs::EntryNameLengthMax);
        NN_RESULT_THROW_UNLESS(frontLength <= nn::fs::EntryNameLengthMax && front[frontLength] == '\0', nn::fs::ResultTooLongPath());

        int rearLength = nn::util::Strnlen(rear, nn::fs::EntryNameLengthMax);
        NN_RESULT_THROW_UNLESS(rearLength <= nn::fs::EntryNameLengthMax && rear[rearLength] == '\0', nn::fs::ResultTooLongPath());

        const int pathLength = frontLength + 1 + rearLength;
        NN_RESULT_THROW_UNLESS(pathLength <= nn::fs::EntryNameLengthMax && pathLength < static_cast<int>(frontBufferSize), nn::fs::ResultTooLongPath());

        front[frontLength] = '/';
        std::memcpy(front + frontLength + 1, rear, rearLength);
        front[pathLength] = '\0';

        *outLength = pathLength;

        NN_RESULT_SUCCESS;
    }
}

bool IsAbsolutePath(const char* path) NN_NOEXCEPT
{
    util::string_view view(path);

    return view.length() >= 2 && view[1] == ':';
}

bool IsInsertedSdCardPath(const char* path) NN_NOEXCEPT
{
    const std::string pathStr(path);
    const auto storageString = std::string(SdCardMountName) + std::string(":");

    return pathStr.length() >= 3 && pathStr.substr(0, storageString.length()) == storageString;
}

bool IsNspFile(const char* path) NN_NOEXCEPT
{
    util::string_view view(path);

    auto pos = view.find_last_of(".");

    // 拡張子がないファイル
    if ( pos == util::string_view::npos )
    {
        return false;
    }
    // 拡張子が .XYZ の形でない
    if ( view.size() - pos != 4 )
    {
        return false;
    }

    // 拡張子を取り出し、比較
    util::string_view ext = view.substr(pos + 1, view.size() - pos );
    if ( ext != "nsp" )
    {
        return false;
    }

    return true;
}

bool IsProgramInstalled(const nn::ncm::ProgramId& programId, ncm::StorageId storageId) NN_NOEXCEPT
{
    nn::ncm::ContentMetaDatabase db;
    NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));

    std::vector<nn::ncm::ContentMetaKey> list;
    list.resize(1024);
    auto listCount = db.ListContentMeta(list.data(), static_cast<int>(list.size()), nn::ncm::ContentMetaType::Unknown);
    list.resize(static_cast<size_t>(listCount.listed));
    if (list.size() > 0)
    {
        for (auto& meta : list)
        {
            if (meta.id == programId.value)
            {
                return true;
            }
        }
    }

    return false;
}

Bit64String GetBit64String(nn::Bit64 bit) NN_NOEXCEPT
{
    Bit64String str;
    util::TSNPrintf(str.data, sizeof(str.data), "0x%016llx", bit);
    return str;
}

Bit32String GetBit32String(nn::Bit32 bit) NN_NOEXCEPT
{
    Bit32String str;
    util::TSNPrintf(str.data, sizeof(str.data), "0x%08x", bit);
    return str;
}

CalendarTimeString GetCalendarTimeString(const nn::time::CalendarTime& calendarTime) NN_NOEXCEPT
{
    CalendarTimeString str;
    util::TSNPrintf(str.data, sizeof(str.data), "%04d/%02d/%02d %02d:%02d:%02d",
        calendarTime.year,
        calendarTime.month,
        calendarTime.day,
        calendarTime.hour,
        calendarTime.minute,
        calendarTime.second);
    return str;
}

CalendarTimeString GetCalendarTimeString(const nn::time::PosixTime& time, nn::time::CalendarAdditionalInfo* info) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(time::IsInitialized());

    time::CalendarTime calendarTime;
    if (time::ToCalendarTime(&calendarTime, info, time).IsFailure())
    {
        CalendarTimeString str;
        util::Strlcpy(str.data, "(Invalid)", sizeof(str.data));
        return str;
    }
    else
    {
        return GetCalendarTimeString(calendarTime);
    }
}

const char* GetDateTimeString(char* out, int maxOutLen, const time::PosixTime& posixTime) NN_NOEXCEPT
{
    if (posixTime < time::InputPosixTimeMin || posixTime > time::InputPosixTimeMax)
    {
        util::TSNPrintf(out, maxOutLen, "Invalid posix time: %lld", posixTime.value);
        return out;
    }

    time::CalendarTime calendarTime;
    time::CalendarAdditionalInfo calendarAdditionalInfo;
    auto result = time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime);
    if (result.IsSuccess())
    {
        util::SNPrintf(out, maxOutLen, "%04d-%02d-%02d %02d:%02d:%02d (%s)",
            calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second,
            calendarAdditionalInfo.timeZone.standardTimeName);
    }
    else
    {
        util::TSNPrintf(out, maxOutLen, "Failed to convert: %08x", result.GetInnerValueForDebug());
    }
    return out;
}

util::optional<ncm::ContentMetaType> ParseDownloadableContentMetaType(const char* type) NN_NOEXCEPT
{
    if (std::strcmp(type, "Application") == 0) return ncm::ContentMetaType::Application;
    if (std::strcmp(type, "Patch") == 0) return ncm::ContentMetaType::Patch;
    if (std::strcmp(type, "AddOnContent") == 0) return ncm::ContentMetaType::AddOnContent;

    return util::nullopt;
}

const char* GetContentMetaTypeString(ncm::ContentMetaType type) NN_NOEXCEPT
{
    switch (type)
    {
    case ncm::ContentMetaType::SystemProgram: return "SystemProgram";
    case ncm::ContentMetaType::SystemData: return "SystemData";
    case ncm::ContentMetaType::SystemUpdate: return "SystemUpdate";
    case ncm::ContentMetaType::Application: return "Application";
    case ncm::ContentMetaType::Patch: return "Patch";
    case ncm::ContentMetaType::AddOnContent: return "AddOnContent";
    case ncm::ContentMetaType::Delta: return "Delta";
    case ncm::ContentMetaType::BootImagePackage: return "BootImagePackage";
    case ncm::ContentMetaType::BootImagePackageSafe: return "BootImagePackageSafe";
    case ncm::ContentMetaType::Unknown: return "Unknown";
    default: NN_UNEXPECTED_DEFAULT;
    }
}
const char* GetContentTypeString(ncm::ContentType type) NN_NOEXCEPT
{
    switch (type)
    {
    case ncm::ContentType::Meta: return "Meta";
    case ncm::ContentType::Program: return "Program";
    case ncm::ContentType::Data: return "Data";
    case ncm::ContentType::Control: return "Control";
    case ncm::ContentType::HtmlDocument: return "HtmlDocument";
    case ncm::ContentType::LegalInformation: return "LegalInformation";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

const char* GetStorageIdString(ncm::StorageId storage) NN_NOEXCEPT
{
    switch (storage)
    {
    case ncm::StorageId::None: return "None";
    case ncm::StorageId::Host: return "Host";
    case ncm::StorageId::GameCard: return "GameCard";
    case ncm::StorageId::BuiltInSystem: return "BuiltInSystem";
    case ncm::StorageId::BuiltInUser: return "BuiltInUser";
    case ncm::StorageId::SdCard: return "SdCard";
    default: NN_UNEXPECTED_DEFAULT;
    }
}
namespace
{
NN_OS_ALIGNAS_THREAD_STACK Bit8 s_Stack[16 * 1024];
Result s_Result;
}

nn::Result WaitInstallTaskExecute(ncm::InstallTaskBase* task, nn::TimeSpan detachSdCardAfter) NN_NOEXCEPT
{

    os::ThreadType thread;
    os::CreateThread(&thread, [](void* p) {
        s_Result = reinterpret_cast<ncm::InstallTaskBase*>(p)->Execute();
    }, task, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority);
    os::StartThread(&thread);

    os::TimerEvent detachSdCardTimer(os::EventClearMode_ManualClear);
    if (detachSdCardAfter != 0)
    {
        detachSdCardTimer.StartOneShot(detachSdCardAfter);
    }
    while (NN_STATIC_CONDITION(true))
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON) && !defined(NN_DEVMENULOTCHECK) && !defined(NN_DISABLE_IDLE_REPORT) // DevMenuLotcheck は idle:sys 権限がないので対象外
        nn::idle::ReportUserIsActive();
#endif
        auto isFinished = os::TryWaitAny(&thread) >= 0;

        auto progress = task->GetProgress();
        NN_UNUSED(progress); // Release ビルド時の変数未使用警告を抑制。
        NN_LOG("%lld / %lld\n", progress.installedSize, progress.totalSize);

        if (isFinished)
        {
            break;
        }

        auto index = os::TimedWaitAny(TimeSpan::FromSeconds(1), &thread, detachSdCardTimer.GetBase());
        if (index == 1)
        {
            ns::RemoveSdCard();
            detachSdCardTimer.Clear();
        }
    }

    os::WaitThread( &thread );
    os::DestroyThread( &thread );

    return s_Result;
}

nn::Result WaitInstallTaskExecute(ncm::ApplyDeltaTaskBase* task) NN_NOEXCEPT
{

    os::ThreadType thread;
    os::CreateThread(&thread, [](void* p) {
        s_Result = reinterpret_cast<ncm::ApplyDeltaTaskBase*>(p)->Execute();
    }, task, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority);
    os::StartThread(&thread);

    while (NN_STATIC_CONDITION(true))
    {
        auto isFinished = os::TryWaitAny(&thread) >= 0;

        auto progress = task->GetProgress();
        NN_UNUSED(progress); // Release ビルド時の変数未使用警告を抑制。
        NN_LOG("%lld / %lld\n", progress.appliedDeltaSize, progress.totalDeltaSize);

        if (isFinished)
        {
            break;
        }

#if defined(NN_BUILD_CONFIG_OS_HORIZON) && !defined(NN_DEVMENULOTCHECK) && !defined(NN_DISABLE_IDLE_REPORT) // DevMenuLotcheck は idle:sys 権限がないので対象外
        nn::idle::ReportUserIsActive();
#endif
        os::SleepThread(TimeSpan::FromSeconds(1));
    }

    os::WaitThread( &thread );
    os::DestroyThread( &thread );

    return s_Result;
}

ncm::ApplicationId GetApplicationIdTarget(const Option& option) NN_NOEXCEPT
{
    ncm::ApplicationId appId = { std::strtoull(option.GetTarget(), NULL, 16) };
    return appId;
}

es::RightsIdIncludingKeyId GetRightsIdTarget(const Option& option) NN_NOEXCEPT
{
    std::string str = std::string(option.GetTarget());

    // ID の先頭が 0x だった場合削除
    if (str.substr(0, 2) == "0x")
    {
        str = str.substr(2);
    }

    es::RightsIdIncludingKeyId rightsId = {};
    for (int i = 0; i < sizeof(es::RightsIdIncludingKeyId); i++)
    {
        rightsId._data[i] = static_cast<uint8_t>(std::strtol(str.substr(i * 2, 2).c_str(), NULL, 16));
    }

    return rightsId;
}

util::optional<bool> GetBooleanOptionValue(const Option& option, const char* key) NN_NOEXCEPT
{
    auto value = option.GetValue(key);

    if( value == nullptr )
    {
        return util::optional<bool>(nullptr);
    }
    else if( std::strncmp(value, "true", std::strlen("true")) == 0 )
    {
        return true;
    }
    else if( std::strncmp(value, "false", std::strlen("false")) == 0 )
    {
        return false;
    }
    else
    {
        return util::optional<bool>(nullptr);
    }
}

nn::Result ReadFile(size_t* outSize, const char* path, void* outBuffer, size_t bufferSize) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t fileSize;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    NN_RESULT_THROW_UNLESS(fileSize <= static_cast<int64_t>(bufferSize), nn::fs::ResultInvalidSize());

    NN_RESULT_DO(nn::fs::ReadFile(outSize, handle, 0, outBuffer, bufferSize));

    NN_RESULT_SUCCESS;
}

nn::Result WriteFile(const char* path, void* buffer, size_t bufferSize, bool force) NN_NOEXCEPT
{
    if (force)
    {
        NN_RESULT_TRY(nn::fs::DeleteFile(path))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_DO(nn::fs::CreateFile(path, bufferSize));

    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, buffer, bufferSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

nn::Result WalkDirectory(const char* path, std::function<nn::Result(const char*, const nn::fs::DirectoryEntry&)> callback) NN_NOEXCEPT
{
    const int pathLength = nn::util::Strnlen(path, nn::fs::EntryNameLengthMax);
    NN_RESULT_THROW_UNLESS(pathLength <= nn::fs::EntryNameLengthMax && path[pathLength] == '\0', nn::fs::ResultTooLongPath());

    char pathBuffer[nn::fs::EntryNameLengthMax + 1];
    std::memcpy(pathBuffer, path, pathLength);
    pathBuffer[pathLength] = '\0';

    struct Context
    {
        nn::fs::DirectoryHandle directoryHandle;
        int pathLength;
    };

    std::stack<Context> contexts;
    NN_UTIL_SCOPE_EXIT
    {
        while (!contexts.empty())
        {
            nn::fs::CloseDirectory(contexts.top().directoryHandle);
            contexts.pop();
        }
    };

    nn::fs::DirectoryHandle directoryHandle;
    NN_RESULT_DO(nn::fs::OpenDirectory(&directoryHandle, pathBuffer, nn::fs::OpenDirectoryMode_All));
    contexts.push({directoryHandle, pathLength});

    while (!contexts.empty())
    {
        nn::fs::DirectoryEntry directoryEntry;
        int64_t entryCount;
        NN_RESULT_DO(nn::fs::ReadDirectory(&entryCount, &directoryEntry, contexts.top().directoryHandle, 1));
        if (entryCount == 0)
        {
            nn::fs::CloseDirectory(contexts.top().directoryHandle);
            contexts.pop();
            if (!contexts.empty())
            {
                pathBuffer[contexts.top().pathLength] = '\0';
            }
        }
        else
        {
            NN_ASSERT(entryCount == 1);

            NN_RESULT_DO(callback(pathBuffer, directoryEntry));

            if (directoryEntry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                int nextPathLength;
                NN_RESULT_DO(ConcatPath(&nextPathLength, pathBuffer, directoryEntry.name, sizeof(pathBuffer)));

                NN_RESULT_DO(nn::fs::OpenDirectory(&directoryHandle, pathBuffer, nn::fs::OpenDirectoryMode_All));
                contexts.push({directoryHandle, nextPathLength});
            }
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result Copy(std::vector<std::unique_ptr<ErrorFileInfo>>* pOutErrorFileInfoList, const char* destination, const char* source, bool force, bool skipsErrorFile) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType sourceEntryType;
    NN_RESULT_DO(nn::fs::GetEntryType(&sourceEntryType, source));

    bool destinationNotFound = false;
    nn::fs::DirectoryEntryType destinationEntryType;
    NN_RESULT_TRY(nn::fs::GetEntryType(&destinationEntryType, destination))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            if (sourceEntryType == nn::fs::DirectoryEntryType_File)
            {
                NN_RESULT_RETHROW;
            }
            else
            {
                destinationNotFound = true;
            }
        }
    NN_RESULT_END_TRY;

    char pathBuffer[nn::fs::EntryNameLengthMax + 1];
    const size_t CopyBufferSize = 8 * 1024 * 1024;
    std::unique_ptr<nn::Bit8[]> copyBuffer(new nn::Bit8[CopyBufferSize]);

    if (sourceEntryType == nn::fs::DirectoryEntryType_File)
    {
        if (destinationEntryType == nn::fs::DirectoryEntryType_File)
        {
            NN_RESULT_DO(CopyFile(pOutErrorFileInfoList, destination, source, copyBuffer.get(), CopyBufferSize, force, skipsErrorFile));
        }
        else
        {
            const char* entryName;
            NN_RESULT_DO(GetEntryNamePointer(&entryName, source));
            NN_RESULT_THROW_UNLESS(nn::util::SNPrintf(pathBuffer, sizeof(pathBuffer), "%s/%s", destination, entryName) < sizeof(pathBuffer), nn::fs::ResultTooLongPath());
            NN_RESULT_DO(CopyFile(pOutErrorFileInfoList, pathBuffer, source, copyBuffer.get(), CopyBufferSize, force, skipsErrorFile));
        }
    }
    else
    {
        if (destinationNotFound)
        {
            NN_RESULT_DO(CreateDirectory(pOutErrorFileInfoList, destination, force, skipsErrorFile));
            NN_RESULT_DO(CopyDirectory(pOutErrorFileInfoList, destination, source, copyBuffer.get(), CopyBufferSize, force, skipsErrorFile));
        }
        else
        {
            const char* entryName;
            NN_RESULT_DO(GetEntryNamePointer(&entryName, source));
            NN_RESULT_THROW_UNLESS(destinationEntryType == nn::fs::DirectoryEntryType_Directory, nn::fs::ResultPathAlreadyExists());
            NN_RESULT_THROW_UNLESS(nn::util::SNPrintf(pathBuffer, sizeof(pathBuffer), "%s/%s", destination, entryName) < sizeof(pathBuffer), nn::fs::ResultTooLongPath());
            NN_RESULT_TRY(CreateDirectory(pOutErrorFileInfoList, pathBuffer, force, skipsErrorFile))
                NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                {
                    if (!force)
                    {
                        NN_RESULT_RETHROW;
                    }
                }
            NN_RESULT_END_TRY;
            NN_RESULT_DO(CopyDirectory(pOutErrorFileInfoList, pathBuffer, source, copyBuffer.get(), CopyBufferSize, force, skipsErrorFile));
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result CreateDirectoryRecursively(const char* path, bool isFilePath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    char currentPath[nn::fs::EntryNameLengthMax + 1] = {};
    bool root = true;

    char c = path[0];

    for (int i = 0; path[i] != '\0'; i++)
    {
        c = path[i];

        if (c == '/' || c == '\\')
        {
            if (!root)
            {
                NN_RESULT_TRY(nn::fs::CreateDirectory(currentPath))
                    NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                    {
                    }
                NN_RESULT_END_TRY;
            }
            else
            {
                root = false;
            }
        }

        currentPath[i] = c;
    }

    if (!root && !isFilePath && c != '/' && c != '\\')
    {
        NN_RESULT_TRY(nn::fs::CreateDirectory(currentPath))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
            }
        NN_RESULT_END_TRY;
    }

    return nn::ResultSuccess();
}

nn::ApplicationId GetSelfApplicationId() NN_NOEXCEPT
{
#if defined(NN_DEVMENUSYSTEM)
    #if defined(APPLICATION_BUILD)
        return nn::ApplicationId{ PROGRAM_ID_OF_DEVMENUSYSTEMAPP };
    #else
        return nn::ApplicationId{ PROGRAM_ID_OF_DEVMENUSYSTEM };
    #endif
#elif defined(NN_DEVMENU)
    #if defined(APPLICATION_BUILD)
        return nn::ApplicationId{ PROGRAM_ID_OF_DEVMENUAPP };
    #else
        return nn::ApplicationId{ PROGRAM_ID_OF_DEVMENU };
    #endif
#elif defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
    return nn::ApplicationId{ NN_TOOL_DEVMENUCOMMANDSYSTEM_PROGRAM_ID };
#elif defined(NN_TOOL_DEVMENUCOMMAND)
    return nn::ApplicationId{ NN_TOOL_DEVMENUCOMMAND_PROGRAM_ID };
#else
    return nn::ApplicationId{ 0x0ull }; // このケースで呼ばれることは無い想定
#endif
}

}

