﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <cctype>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TFormatString.h>
#include <nn/os.h>
#include <nn/fs.h>

#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include <nn/am/service/am_SystemProgramIds.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_ProgramIdNameMap.h"
#include "DevMenuCommand_AppletCommand.h"
#include "DevMenuCommand_HashUtil.h"
#include "DevMenuCommand_InstallUtil.h"

using namespace nn;

//------------------------------------------------------------------------------------------------

namespace {

    const char HelpMessage[] =
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "usage: " DEVMENUCOMMAND_NAME " applet list\n"
        "       " DEVMENUCOMMAND_NAME " applet install <absolute_nsp_path>\n"
        "       " DEVMENUCOMMAND_NAME " applet uninstall <applet_program_id>\n"
        "       " DEVMENUCOMMAND_NAME " applet digest\n"
        "       " DEVMENUCOMMAND_NAME " applet redirect-program --id <applet_program_id> --path <absolute_nca_path>\n"
        "       " DEVMENUCOMMAND_NAME " applet list-redirect-programs [--id <applet_program_id>]\n"
        "       " DEVMENUCOMMAND_NAME " applet set-shop-applet-program <applet_program_id>\n"
        "       " DEVMENUCOMMAND_NAME " applet get-shop-applet-program\n"
        "       "
#endif
        ""; // 終端

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    struct AppletTable
    {
        ncm::SystemProgramId programId;
        std::string programName;
    };

    // システムデータ ID 一覧
    const ncm::SystemProgramId ProgramId_CertStore              = { 0x0100000000000800 };
    const ncm::SystemProgramId ProgramId_ErrorMessage           = { 0x0100000000000801 };
    const ncm::SystemProgramId ProgramId_MiiModel               = { 0x0100000000000802 };
    const ncm::SystemProgramId ProgramId_BrowserDll             = { 0x0100000000000803 };
    const ncm::SystemProgramId ProgramId_Help                   = { 0x0100000000000804 };
    const ncm::SystemProgramId ProgramId_SharedFont             = { 0x0100000000000805 };
    const ncm::SystemProgramId ProgramId_NgWord                 = { 0x0100000000000806 };
    const ncm::SystemProgramId ProgramId_SsidList               = { 0x0100000000000807 };
    const ncm::SystemProgramId ProgramId_Dictionary             = { 0x0100000000000808 };
    const ncm::SystemProgramId ProgramId_SystemVersion          = { 0x0100000000000809 };
    const ncm::SystemProgramId ProgramId_AvatarImage            = { 0x010000000000080A };
    const ncm::SystemProgramId ProgramId_LocalNews              = { 0x010000000000080B };
    const ncm::SystemProgramId ProgramId_Eura                   = { 0x010000000000080C };
    const ncm::SystemProgramId ProgramId_UrlBlackList           = { 0x010000000000080D };
    const ncm::SystemProgramId ProgramId_TimeZoneBinary         = { 0x010000000000080E };
    const ncm::SystemProgramId ProgramId_BrowserCertStore       = { 0x010000000000080F };

    const ncm::SystemProgramId ProgramId_FontNintendoExtension  = { 0x0100000000000810 };
    const ncm::SystemProgramId ProgramId_FontStandard           = { 0x0100000000000811 };
    const ncm::SystemProgramId ProgramId_FontKorean             = { 0x0100000000000812 };
    const ncm::SystemProgramId ProgramId_FontChineseTraditional = { 0x0100000000000813 };
    const ncm::SystemProgramId ProgramId_FontChineseSimple      = { 0x0100000000000814 };
    const ncm::SystemProgramId ProgramId_FontBfcpx              = { 0x0100000000000815 };

    const ncm::SystemProgramId ProgramId_DataOceanShared        = { 0x010000000000081E };

    // TODO : 将来的には SdkProgramIds.json に取り込むべき？
    const AppletTable g_AppletTable[] =
    {
        // programId                                            // programName
        { nn::am::service::ProgramId_LibraryAppletAuth,         "LibraryAppletAuth"         }, // 簡易認証 LA
        { nn::am::service::ProgramId_LibraryAppletCabinet,      "LibraryAppletCabinet"      }, // amiibo 設定 LA
        { nn::am::service::ProgramId_LibraryAppletController,   "LibraryAppletController"   }, // コントローラー LA
        { nn::am::service::ProgramId_LibraryAppletDataErase,    "LibraryAppletDataErase"    }, // データ削除 LA
        { nn::am::service::ProgramId_LibraryAppletError,        "LibraryAppletError"        }, // エラービューアー LA
        { nn::am::service::ProgramId_LibraryAppletNetConnect,   "LibraryAppletNetConnect"   }, // ネット接続 LA
        { nn::am::service::ProgramId_LibraryAppletPlayerSelect, "LibraryAppletPlayerSelect" }, // プレイヤー選択 LA
        { nn::am::service::ProgramId_LibraryAppletSwkbd,        "LibraryAppletSwkbd"        }, // ソフトウェアキーボード LA
        { nn::am::service::ProgramId_LibraryAppletMiiEdit,      "LibraryAppletMiiEdit"      }, // Mii 編集 LA
        { nn::am::service::ProgramId_LibraryAppletWeb,          "LibraryAppletWeb"          }, // Web LA
        { nn::am::service::ProgramId_LibraryAppletShop,         "LibraryAppletShop"         }, // ショップ LA

        { nn::am::service::ProgramId_LibraryAppletPhotoViewer,  "LibraryAppletPhotoViewer"  }, // フォトビューアー LA
        { nn::am::service::ProgramId_LibraryAppletSet,          "LibraryAppletSet"          }, // 本体設定 LA
        { nn::am::service::ProgramId_LibraryAppletOfflineWeb,   "LibraryAppletOfflineWeb"   }, // Offline Web LA
        { nn::am::service::ProgramId_LibraryAppletLoginShare,   "LibraryAppletLoginShare"   }, // ログインシェア LA
        { nn::am::service::ProgramId_LibraryAppletWifiWebAuth,  "LibraryAppletWifiWebAuth"  }, // WiFi WebAuth  LA

        { nn::am::service::ProgramId_LibraryAppletMyPage,       "LibraryAppletMyPage"       }, // マイページ LA

        { nn::am::service::ProgramId_SystemAppletMenu,          "SystemAppletMenu"          }, // SA メニュー

        { nn::am::service::ProgramId_OverlayApplet,             "OverlayApplet"             }, // オーバーレイ表示 LA

        { nn::am::service::ProgramId_SystemApplication,         "Starter"                   }, // システムアプリケーション

        { nn::am::service::ProgramId_MaintenanceMenu,           "MaintenanceMenu"           }, // メンテナンスモード

        { nn::am::service::ProgramId_LibraryAppletDummyShop,    "LibraryAppletDummyShop"    }, // Dummy Shop LA

        { nn::am::service::ProgramId_LibraryAppletGift,         "LibraryAppletGift"         }, // Gift LA

        { nn::am::service::ProgramId_LibraryAppletUserMigration, "LibraryAppletUserMigration"}, // User Migration LA

        { nn::am::service::ProgramId_LibraryAppletEncounter,    "LibraryAppletEncounter"    }, // すれちがい LA

        { nn::am::service::ProgramId_LibraryAppletStory,        "LibraryAppletStory"        }, // セーブデータ同期 LA

        // システムデータも対応するため、このテーブルに加える
        { ProgramId_CertStore,                                  "CertStore"                 }, // NSS用証明書ストア
        { ProgramId_ErrorMessage,                               "ErrorMessage"              }, // エラーメッセージ
        { ProgramId_MiiModel,                                   "MiiModel"                  }, // Miiのモデルリソースデータ
        { ProgramId_BrowserDll,                                 "BrowserDll"                }, // Cruiserが提供するDLL
        { ProgramId_Help,                                       "Help"                      }, // NX本体の権利表記
        { ProgramId_SharedFont,                                 "SharedFont"                }, // 共有フォント
        { ProgramId_NgWord,                                     "NgWord"                    }, // NGワード
        { ProgramId_SsidList,                                   "SsidList"                  }, // SSIDリスト
        { ProgramId_Dictionary,                                 "Dictionary"                }, // 辞書
        { ProgramId_SystemVersion,                              "SystemVersion"             }, // 本体バージョン
        { ProgramId_AvatarImage,                                "AvatarImage"               }, // アカウントのアバター画像
        { ProgramId_LocalNews,                                  "LocalNews"                 }, // ローカルニュースのデータ
        { ProgramId_Eura,                                       "Eura"                      }, // EULA
        { ProgramId_UrlBlackList,                               "UrlBlackList"              }, // URLブラックリスト
        { ProgramId_TimeZoneBinary,                             "TimeZoneBinary"            }, // タイムゾーン
        { ProgramId_BrowserCertStore,                           "BrowserCertStore"          }, // Cruiser用証明書ストア
        { ProgramId_FontNintendoExtension,                      "FontNintendoExtension"     }, // 共有フォント 外字フォント
        { ProgramId_FontStandard,                               "FontStandard"              }, // 共有フォント 日米欧フォント
        { ProgramId_FontKorean,                                 "FontKorean"                }, // 共有フォント 韓国フォント
        { ProgramId_FontChineseTraditional,                     "FontChineseTraditional"    }, // 共有フォント 繁体字フォント
        { ProgramId_FontChineseSimple,                          "FontChineseSimple"         }, // 共有フォント 簡体字フォント
        { ProgramId_FontBfcpx,                                  "FontBfcpx"                 }, // 共有フォント bfcpx の集合
        { ProgramId_DataOceanShared,                            "DataOceanShared"           }, // Ocean 用共有データ
    };

    const size_t g_AppletTableSize = sizeof(g_AppletTable) / sizeof(AppletTable);

    bool IsAppletProgramId( Bit64 id )
    {
        for ( int i = 0; i < g_AppletTableSize; i++ )
        {
            if ( id == g_AppletTable[i].programId.value )
            {
                return true;
            }
        }
        return false;
    }
    const char* GetAppletProgramNameById( Bit64 id )
    {
        for ( int i = 0; i < g_AppletTableSize; i++ )
        {
            if ( id == g_AppletTable[i].programId.value )
            {
                return g_AppletTable[i].programName.c_str();
            }
        }
        return "Unknown";
    }
    Bit64 GetAppletProgramIdByName( const char* name )
    {
        const size_t NameLengthMax = 128;
        auto nameLength = strlen( name );
        if ( nameLength >= NameLengthMax )
        {
            return 0;
        }
        for ( int i = 0; i < g_AppletTableSize; i++ )
        {
            auto *p = g_AppletTable[i].programName.c_str();
            if ( strncmp( name, p, nameLength )  == 0 )
            {
                return g_AppletTable[i].programId.value;
            }
        }
        return 0;
    }
    void ConvertHashToStr(std::string &str, ncm::Hash hash)
    {
        str.clear();
        for (int i = 0; i < crypto::Sha256Generator::HashSize; i++)
        {
            char buffer[3];
            nn::util::TSNPrintf( buffer, sizeof(buffer), "%02x", hash.data[i] );
            str.append( buffer );
        }
    }

    Result GetAppletMetaKeyList(std::vector<ncm::ContentMetaKey>& list)
    {
        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInSystem));

        list.resize(1024);
        auto listCount = db.ListContentMeta( list.data(), static_cast<int>(list.size()), ncm::ContentMetaType::Unknown );
        list.resize(static_cast<size_t>(listCount.listed));
        if (list.size() > 0)
        {
            for ( auto meta = list.begin(); meta != list.end(); )
            {
                if ( !IsAppletProgramId( meta->id ) )
                {
                    meta = list.erase( meta );
                    continue;
                }
                meta++;
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result AppletListCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        std::vector<ncm::ContentMetaKey> list;
        NN_RESULT_DO(GetAppletMetaKeyList( list ));
        if ( list.size() > 0 )
        {
            NN_LOG("%d content meta(s) found.\n", list.size());
            NN_LOG("id                        ver.  type            name\n");
            //      0x0123456789abcdef  4294967295  SystemProgram   DevMenu
            for (auto& meta : list)
            {
                NN_LOG( "0x%016llx  %10u  %-14s  %s\n",
                    meta.id, meta.version, GetContentMetaTypeString(meta.type), GetAppletProgramNameById( meta.id ) );
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result AppletInstallCommand(bool* outValue, const Option& option)
    {
        auto nspPath = option.GetTarget();
        if (std::strlen(nspPath) == 0)
        {
            NN_LOG("usage: " DEVMENUCOMMAND_NAME " applet install <absolute_nsp_path>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (!devmenuUtil::IsAbsolutePath(nspPath))
        {
            NN_LOG("'%s' is not an absolute path.\n", nspPath);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, nspPath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        static const size_t BufferSize = 16 * 1024 * 1024;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);
        ncm::SubmissionPackageInstallTask task;
        NN_RESULT_DO(task.Initialize(file, ncm::StorageId::BuildInSystem, buffer.get(), BufferSize));

        NN_LOG("Preparing to install %s...\n", nspPath);
        NN_RESULT_DO(task.Prepare());
        // インストール前に Applet か SystemData かどうかをチェックする
        {
            const int Count = 2;
            const int Offset = 0;
            ncm::StorageContentMetaKey storageKeys[Count];
            int outCount;

            NN_RESULT_DO( task.ListContentMetaKey( &outCount, storageKeys, Count, Offset) );
            auto key = storageKeys[0].key;

            bool isApplet = true;

            // key が 1 つの場合以外
            if ( outCount != 1)
            {
                isApplet = false;
            }
            // Type が SystemProgram or SystemData でない
            else if ( key.type != ncm::ContentMetaType::SystemProgram  &&
                key.type != ncm::ContentMetaType::SystemData )
            {
                isApplet = false;
            }
            // Id が Applet のものでない
            else if ( !IsAppletProgramId( key.id ) )
            {
                isApplet = false;
            }

            if ( !isApplet )
            {
                NN_LOG( "This nsp is neither an applet program nor a system data.\n" );
                NN_RESULT_DO( task.Cleanup() );
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            ncm::ContentMetaDatabase db;
            ncm::ContentStorage storage;
            NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuiltInSystem));
            NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::BuiltInSystem));
            bool hasHigherVersion;
            NN_RESULT_DO(devmenuUtil::CheckHigherVersionAndUninstallIfNeeded(&hasHigherVersion, &db, &storage, key, option.HasKey("--force")));
            if (hasHigherVersion)
            {
                DEVMENUCOMMAND_LOG("Higher version exists, install failed.\n");
                DEVMENUCOMMAND_LOG("If you want to install, please use --force option.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_DO(task.Cleanup());
        NN_RESULT_DO(task.Prepare());
        NN_RESULT_DO(devmenuUtil::WaitInstallTaskExecute(&task));
        NN_RESULT_DO(task.Commit());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result AppletUninstallCommand(bool* outValue, const Option& option)
    {
        auto target = option.GetTarget();
        if (std::strlen(target) == 0)
        {
            NN_LOG( "usage: " DEVMENUCOMMAND_NAME " applet uninstall <applet_program_id> | <applet_name>\n" );
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ncm::ContentMetaDatabase db;
        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInSystem));
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::BuildInSystem));
        ncm::ContentManagerAccessor accessor(&db, &storage);

        char* endptr;
        Bit64 id = STR_TO_ULL(target, &endptr, 16);
        if (*endptr != '\0')
        {
            auto idByName = GetAppletProgramIdByName( target );
            if (idByName)
            {
                id = idByName;
            }
            else
            {
                NN_LOG("Unknown program name %s.\n", target);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        if ( !IsAppletProgramId( id ) )
        {
            NN_LOG("This id is neither an applet program id nor a system data id.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(accessor.DeleteAll(id));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result AppletDigestCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        std::vector<ncm::ContentMetaKey> list;
        NN_RESULT_DO(GetAppletMetaKeyList( list ));
        if ( list.size() > 0 )
        {
            NN_LOG("%d content meta(s) found.\n", list.size());
            NN_LOG("id                  digest\n");
            //      0x0123456789abcdef  ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff
            for (auto& meta : list)
            {
                ncm::Hash hash;
                std::string hashString;
                NN_RESULT_DO(devmenuUtil::GetApplicationRomIdHash( &hash, meta, ncm::StorageId::BuildInSystem ));
                ConvertHashToStr( hashString, hash );
                NN_LOG( "0x%016llx  %s\n", meta.id, hashString.data() );
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    const uint64_t g_BeginAppicationProgramId = 0x0100000000002800;
    const uint64_t g_EndAppicationProgramId   = 0xffffffffffffffff;
    uint64_t g_BeginAppletProgramId = 0;
    uint64_t g_EndAppletProgramId   = 0;

    bool IsPermittedAppletProgramId(ncm::ProgramId id) NN_NOEXCEPT
    {
        // アプレット用の ProgramId として利用可能な範囲は以下を参照
        // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=111809427
        return g_BeginAppletProgramId <= id.value && id.value <= g_EndAppletProgramId;
    }

    bool IsPermittedApplicationProgramId(ncm::ProgramId id) NN_NOEXCEPT
    {
        // 下記のページのサンプル以降の全ての ID を許可する
        // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=109326482
        return g_BeginAppicationProgramId <= id.value && id.value <= g_EndAppicationProgramId;
    }

    void UsageForSetProgramPathCommand() NN_NOEXCEPT
    {
        NN_LOG("usage: " DEVMENUCOMMAND_NAME " applet redirect-program --id <applet_program_id> --path <absolute_nca_path>\n");
    }

    #define HOST_PREFIX "@Host:/"
    Result RedirectProgramCommand(bool* outValue, const Option& option)
    {
        bool isAppletRedirect = true;
        auto idString    = option.GetValue("--id");
        auto idStringLen = idString ? std::strlen(idString) : 0;
        if (idStringLen == 0)
        {
            UsageForSetProgramPathCommand();
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ncm::ProgramId programId;
        programId.value = STR_TO_ULL(idString, NULL, 16);

        if (!IsPermittedAppletProgramId(programId))
        {
            if ( IsPermittedApplicationProgramId(programId) )
            {
                isAppletRedirect = false;
            }
            else
            {
                NN_LOG("Invalid programId '0x%016llx' to redirect.\n", programId.value);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        auto ncaPath    = option.GetValue("--path");
        auto ncaPathLen = ncaPath ? std::strlen(ncaPath) : 0;
        if (ncaPathLen == 0 || ncaPathLen > sizeof(nn::lr::Path) - sizeof(HOST_PREFIX))
        {
            UsageForSetProgramPathCommand();
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::lr::Path hostPath;
        std::strcpy(hostPath.string, HOST_PREFIX);
        std::strcat(hostPath.string, ncaPath);

        if (!devmenuUtil::IsAbsolutePath(ncaPath))
        {
            // Cygwin の full-path 指定でもなければエラー
            if (!(ncaPathLen > 3 && ncaPath[0] == '/' && std::isalpha(ncaPath[1]) && ncaPath[2] == '/'))
            {
                NN_LOG("'%s' is not an absolute path.\n", ncaPath);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            // Cygwin の full-path 指定を通常の full-path 指定に書き換える
            hostPath.string[sizeof(HOST_PREFIX) - 1] = ncaPath[1];
            hostPath.string[sizeof(HOST_PREFIX) + 0] = ':';
        }

        nn::lr::LocationResolver resolver;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::lr::OpenLocationResolver(&resolver,
            isAppletRedirect ? nn::ncm::StorageId::BuildInSystem : nn::ncm::StorageId::BuildInUser ));

        //NN_LOG("0x%016llx: redirected to '%s'.\n", programId.value, hostPath);

        resolver.RedirectProgramPath(programId, hostPath);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ListRedirectPrograms(bool* outValue, const Option& option)
    {
        NN_UNUSED(outValue);
        NN_UNUSED(option);

        auto storageId   = ncm::StorageId::BuildInSystem;
        auto idString    = option.GetValue("--id");
        auto idStringLen = idString ? std::strlen(idString) : 0;

        NN_LOG("ProgramId           Path\n");
        //      0x0123456789abcdef  ""

        int successCount = 0;
        uint64_t beginId = g_BeginAppletProgramId;
        uint64_t endId = g_EndAppletProgramId;
        if ( idStringLen != 0 )
        {
            uint64_t id = STR_TO_ULL(idString, NULL, 16);
            beginId = id;
            endId = id;
            if ( id < g_BeginAppletProgramId || g_EndAppletProgramId < id )
            {
                storageId = ncm::StorageId::BuildInUser;
            }
        }
        for (auto id=beginId; id<=endId; ++id)
        {
            ncm::ProgramId programId = { id };

            nn::lr::LocationResolver resolver;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::lr::OpenLocationResolver(&resolver, storageId));

            nn::lr::Path path;
            auto result = resolver.ResolveProgramPath(&path, programId);
            if (result.IsSuccess())
            {
                NN_LOG("0x%016llx  %s\n", programId.value, path);
                ++successCount;
            }
        }

        if (successCount == 0)
        {
            NN_LOG("[INFO] No redirection setting found.\n");
        }
        else
        {
            NN_LOG("[INFO] %d redirection setting(s) found.\n", successCount);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetShopAppletProgram(bool* outValue, const Option& option)
    {
        auto idString = option.GetTarget();
        if (std::strlen(idString) == 0)
        {
            NN_LOG("usage: " DEVMENUCOMMAND_NAME " applet set-shop-applet-program <applet_program_id>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char* endPtr;
        auto id = std::strtoull(idString, &endPtr, 16);
        if (*endPtr != '\0')
        {
            NN_LOG("Invalid program id format\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char setId[32];
        util::SNPrintf(setId, sizeof(setId), "0x%016llx", id);
        settings::fwdbg::SetSettingsItemValue("ns.applet", "shop_applet_id", setId, std::strlen(setId) + 1);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetShopAppletProgram(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        char idStr[32];
        settings::fwdbg::GetSettingsItemValue(idStr, sizeof(idStr), "ns.applet", "shop_applet_id");

        NN_LOG("%s\n", idStr);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    const SubCommand g_SubCommands[] =
    {
        { "list",                   AppletListCommand       },
        { "install",                AppletInstallCommand    },
        { "uninstall",              AppletUninstallCommand  },
        { "digest",                 AppletDigestCommand     },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "redirect-program",       RedirectProgramCommand  },
        { "list-redirect-programs", ListRedirectPrograms    },
        { "set-shop-applet-program", SetShopAppletProgram   },
        { "get-shop-applet-program", GetShopAppletProgram   }
#endif
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result AppletCommand(bool* outValue, const Option& option)
{
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    {
        ProgramIdNameMap map;
        NN_RESULT_DO(map.Initialize());

        auto beginId = map.GetId("qlaunch");
        auto endId   = map.GetId("EndOceanProgramId");
        if (beginId && endId)
        {
            g_BeginAppletProgramId = *beginId;
            g_EndAppletProgramId   = *endId;
        }
        else
        {
            // TORIAEZU: TODO:
            // ProgramId を取得できなかった場合の保険でデフォルト値を入れておく
            g_BeginAppletProgramId = 0x0100000000001000;
            g_EndAppletProgramId   = 0x0100000000001fff;
        }
#if 0
        NN_LOG("BeginAppletProgramId: 0x%016llx\n", g_BeginAppletProgramId);
        NN_LOG("EndAppletProgramId:   0x%016llx\n", g_EndAppletProgramId);
#endif
    }
#endif

    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // TORIAEZU : ns::InitializeForSystemUpdate() を呼び出しただけでは接続に成功して待たないため、
    // 動作上無意味な API を読んで初期化が終わるまで待つようにする
    ns::GetBackgroundNetworkUpdateState();

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu applet command. See '" DEVMENUCOMMAND_NAME " applet --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
