﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_Windows.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/result/result_HandlingUtility.h>

#include "CalWriter_FileSystem.h"

namespace {

//!< パス文字列から末尾のファイル名を削除します。
::std::string RemoveFileName(const ::std::string& path) NN_NOEXCEPT;

//!< 絶対パスを相対パスに変換します。
::std::string GetRelativePath(const ::std::string& src,
                              const ::std::string& dst) NN_NOEXCEPT;

} // namespace

const char* GetRawPartitionMountName() NN_NOEXCEPT
{
    if (ExistsPartition())
    {
        return ::nn::fs::GetBisMountName(::nn::fs::BisPartitionId::CalibrationBinary);
    }
    else
    {
        return ::nn::fs::GetBisMountName(::nn::fs::BisPartitionId::CalibrationFile);
    }
}

const char* GetFatPartitionMountName() NN_NOEXCEPT
{
    return ::nn::fs::GetBisMountName(::nn::fs::BisPartitionId::CalibrationFile);
}

void InitializeFileSystem() NN_NOEXCEPT
{
    // ファイルシステム用のアロケータを設定
    ::nn::fs::SetAllocator(AllocateForFileSystem, DeallocateForFileSystem);

    // カレントディレクトリのパスを取得
    ::std::unique_ptr<char[]> path(new char[MAX_PATH]);
    NN_ABORT_UNLESS_NOT_EQUAL(
        0u, ::GetCurrentDirectoryA(MAX_PATH, path.get()));

    // 実行ファイル自身のパスを取得
    ::std::unique_ptr<char[]> moduleFilePath(new char[MAX_PATH]);
    NN_ABORT_UNLESS_NOT_EQUAL(
        0u, ::GetModuleFileNameA(NULL, moduleFilePath.get(), MAX_PATH));

    // エミュレーション BIS のパスを設定
    ::nn::fs::SetBisRootForHost(
        ::nn::fs::BisPartitionId::CalibrationFile,
        GetRelativePath(path.get(),
                        RemoveFileName(moduleFilePath.get()).c_str()).c_str());
}

RawPartition::RawPartition() NN_NOEXCEPT
    : m_IsMounted(false)
{
    // 何もしない
}

RawPartition::~RawPartition() NN_NOEXCEPT
{
    if (m_IsMounted)
    {
        ::nn::fs::Unmount(GetRawPartitionMountName());
    }
}

::nn::Result RawPartition::Mount() NN_NOEXCEPT
{
    if (!m_IsMounted)
    {
        NN_RESULT_DO(
            ::nn::fs::MountBis(::nn::fs::BisPartitionId::CalibrationFile, nullptr));

        m_IsMounted = true;
    }

    NN_RESULT_SUCCESS;
}

FatPartition::FatPartition() NN_NOEXCEPT
    : m_IsMounted(false)
{
    // 何もしない
}

FatPartition::~FatPartition() NN_NOEXCEPT
{
    if (m_IsMounted)
    {
        ::nn::fs::Unmount(GetFatPartitionMountName());
    }
}

::nn::Result FatPartition::Mount() NN_NOEXCEPT
{
    if (!m_IsMounted)
    {
        NN_RESULT_DO(
            ::nn::fs::MountBis(::nn::fs::BisPartitionId::CalibrationFile, nullptr));

        m_IsMounted = true;
    }

    NN_RESULT_SUCCESS;
}

namespace {

::std::string RemoveFileName(const ::std::string& path) NN_NOEXCEPT
{
    const size_t pos = path.find_last_of('\\');
    return (pos == ::std::string::npos) ? path : path.substr(0, pos + 1);
}

::std::string GetRelativePath(const ::std::string& src,
                              const ::std::string& dst) NN_NOEXCEPT
{
    size_t head = 0;

    while (head < src.size() && head < dst.size() && src[head] == dst[head])
    {
        ++head;
    }

    ::std::string path;

    if (head == src.size())
    {
        path += '.';
    }
    else if (0 < head && head < src.size())
    {
        path += "..\\";

        for (size_t i = head; i < src.size(); ++i)
        {
            if (src[i] == '\\')
            {
                path += "..\\";
            }
        }
    }

    if (head < dst.size())
    {
        path += dst.substr(head);
    }

    return path;
}

} // namespace
