﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/wlan/wlan_Types.h>
#include <horizon_main.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include "wlan_WpaSupplicant.h"
#include "wlan_CommonType.h"

namespace nn { namespace wlan {

// initialize static handle
void * WpaSupplicant::m_WpaSuppHandle(NULL);
StateMachine* WpaSupplicant::m_pStateMachine(NULL);
bool WpaSupplicant::m_IsSleepState;

WpaSupplicant::WpaSupplicant() NN_NOEXCEPT
{
    m_WpaSuppHandle = NULL;
    m_pStateMachine = NULL;
    m_IsInitialized = false;
}

WpaSupplicant::~WpaSupplicant() NN_NOEXCEPT
{
    m_WpaSuppHandle = NULL;
    m_pStateMachine = NULL;
}

bool WpaSupplicant::InitializeSupplicant() NN_NOEXCEPT
{
    bool ret = true;
    if( m_IsInitialized == false )
    {
        WLAN_LOG_INFO("<WPA_SUPP>>>>> %s::wpa_supplicant_main <<<<\n", __FUNCTION__);
        int32_t wpaRet;
        void *pHandle;
        const char* argv[4] = {"wpa_supplicant_argument", "-iwlan0", "-Dbcm80211", "-C/data/misc/wifi/wlan0"};
        const int argc = sizeof(argv) / sizeof(char*);

        wpaRet = wpa_supplicant_main(argc, const_cast<char**>(argv), &pHandle);
        ret = (wpaRet < 0) ? false : true;

        m_WpaSuppHandle = pHandle;

        NN_ABORT_UNLESS(wpaRet == BWL_ERR_SUCCESS, "<<<<<< WPA Supplicant failed to initialize >>>>>>");

        RegisterCallbacks();

        m_IsInitialized = true;
        m_IsSleepState = false;
    }
    else
    {
        WLAN_LOG_INFO("<WPA_SUPP> %s::wpa_supplicant is already initialized <<<<\n", __FUNCTION__);
    }

    return ret;
}

void WpaSupplicant::FinalizeSupplicant() NN_NOEXCEPT
{
    if( m_IsInitialized == true )
    {
        wpa_supplicant_exit(m_WpaSuppHandle);
        WLAN_LOG_INFO("<WPA_SUPP> Supplicant finalized \n");
        m_IsInitialized = false;
    }
    else
    {
        WLAN_LOG_INFO("<WPA_SUPP> Supplicant is not initialized @ %s\n", __FUNCTION__);
    }
}

bool WpaSupplicant::ScanRequest(WlanScanParameters* pScanParams) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("<WPA_SUPP> Supplicant scan request issued\n");

    int32_t wpaRet;

    struct ether_addr* pBssid;
    struct ether_addr bssid;
    Bit8 broadcastMac[] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

    if( std::memcmp(&pScanParams->bssid[0], &broadcastMac[0], nn::wlan::MacAddress::MacAddressSize) == 0 )
    {
        pBssid = NULL;
    }
    else
    {
        std::memcpy(&bssid.octet[0], &pScanParams->bssid[0], nn::wlan::MacAddress::MacAddressSize);
        pBssid = &bssid;
    }


    wpaRet = wpa_supplicant_ctrl_scan(
            m_WpaSuppHandle,                     // handle
            pScanParams->ssidList,               // ssid list
            pScanParams->ssidCount,              // ssid count
            reinterpret_cast<uint8_t*>(pBssid),  // bssid
            pScanParams->scanType,               // scan type
            pScanParams->channelScanTime,        // dwell time
            pScanParams->channelCount,           // channel count
            &pScanParams->channelList[0]);       // channel list

    return (wpaRet < 0) ? false : true;
}

bool WpaSupplicant::CancelScan() NN_NOEXCEPT
{
    int32_t wpaRet = wpa_supplicant_ctrl_scan_abort(m_WpaSuppHandle);

    return (wpaRet < 0) ? false : true;
}

bool WpaSupplicant::CancelJoinNetwork() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("<WPA_SUPP> Cancel Join requested!!!\n");
    int32_t wpaRet = wpa_supplicant_ctrl_disconnect(m_WpaSuppHandle, 1);

    return (wpaRet < 0) ? false : true;
}


bool WpaSupplicant::JoinNetwork(WlanConnectinoParameters *pConnectParams) NN_NOEXCEPT
{
    int32_t wpaRet = false;
    uint16_t channel = 0;
    uint16_t num_channels = 0;
    char ssidStr[Ssid::SsidLengthMax + 1];

    WLAN_LOG_DEBUG("<WPA_SUPP> -----------------Association command started ----------------\n");

    std::memset(ssidStr, '\0', sizeof(ssidStr));
    std::memcpy(ssidStr, pConnectParams->ssid.GetSsidData(), pConnectParams->ssid.GetLength());

    WLAN_LOG_INFO("<WPA_SUPP> Joining: SSID = %s\n", ssidStr);
    WLAN_LOG_INFO("<WPA_SUPP> Key = %s\n", pConnectParams->security.key);
    WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = %d\n", pConnectParams->security.privacyMode);

    wpaRet = wpa_supplicant_ctrl_add_network(m_WpaSuppHandle);
    NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");

    switch(pConnectParams->security.privacyMode)
    {
        case SecurityMode_Open:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = NONE\n");

            wpaRet = wpa_supplicant_ctrl_set_wpa_auth(m_WpaSuppHandle, WPA_SUPPLICANT_KEY_MGMT_NONE);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec(m_WpaSuppHandle, WPA_SUPPLICANT_WSEC_NONE);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        case SecurityMode_Wep64Open:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = WEP64-OPEN\n");
            wpaRet = wpa_supplicant_ctrl_set_wpa_auth(m_WpaSuppHandle, WPA_SUPPLICANT_KEY_MGMT_NONE);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec(m_WpaSuppHandle, WPA_SUPPLICANT_WSEC_WEP);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_auth_type(m_WpaSuppHandle, WPA_SUPPLICANT_AUTH_OPEN);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wep_key(m_WpaSuppHandle, pConnectParams->security.keyIdx,(char*)&pConnectParams->security.key);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        case SecurityMode_Wep128Open:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = WEP128-OPEN\n");
            wpaRet = wpa_supplicant_ctrl_set_wpa_auth(m_WpaSuppHandle, WPA_SUPPLICANT_KEY_MGMT_NONE);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec(m_WpaSuppHandle, WPA_SUPPLICANT_WSEC_WEP);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_auth_type(m_WpaSuppHandle, WPA_SUPPLICANT_AUTH_OPEN);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wep_key(m_WpaSuppHandle, pConnectParams->security.keyIdx,(char*)&pConnectParams->security.key);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        case SecurityMode_Wep64Shared:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = WEP64-SHARED\n");
            wpaRet = wpa_supplicant_ctrl_set_wpa_auth(m_WpaSuppHandle, WPA_SUPPLICANT_KEY_MGMT_NONE);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec(m_WpaSuppHandle, WPA_SUPPLICANT_WSEC_WEP);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_auth_type(m_WpaSuppHandle, WPA_SUPPLICANT_AUTH_SHARED);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wep_key(m_WpaSuppHandle, pConnectParams->security.keyIdx,(char*)&pConnectParams->security.key);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        case SecurityMode_Wep128Shared:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = WEP128-SHARED\n");
            wpaRet = wpa_supplicant_ctrl_set_wpa_auth(m_WpaSuppHandle, WPA_SUPPLICANT_KEY_MGMT_NONE);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec(m_WpaSuppHandle, WPA_SUPPLICANT_WSEC_WEP);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_auth_type(m_WpaSuppHandle, WPA_SUPPLICANT_AUTH_SHARED);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wep_key(m_WpaSuppHandle, pConnectParams->security.keyIdx,(char*)&pConnectParams->security.key);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        case SecurityMode_WpaTkip:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = WPA-PSK-TKIP\n");
            wpaRet = wpa_supplicant_ctrl_set_wpa_auth(m_WpaSuppHandle, WPA_SUPPLICANT_KEY_MGMT_WPA_PSK);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec(m_WpaSuppHandle, WPA_SUPPLICANT_WSEC_TKIP);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec_key(m_WpaSuppHandle, (char*)&pConnectParams->security.key);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        case SecurityMode_Wpa2Tkip:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = WPA2-PSK-TKIP\n");
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wpa_auth(m_WpaSuppHandle, WPA_SUPPLICANT_KEY_MGMT_WPA2_PSK);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec(m_WpaSuppHandle, WPA_SUPPLICANT_WSEC_TKIP);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec_key(m_WpaSuppHandle, (char*)&pConnectParams->security.key);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        case SecurityMode_WpaAes:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = WPA-PSK-AES\n");
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wpa_auth(m_WpaSuppHandle, WPA_SUPPLICANT_KEY_MGMT_WPA_PSK);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec(m_WpaSuppHandle, WPA_SUPPLICANT_WSEC_CCMP);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec_key(m_WpaSuppHandle, (char*)&pConnectParams->security.key);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        case SecurityMode_Wpa2Aes:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = WPA2-PSK-AES\n");
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wpa_auth(m_WpaSuppHandle, WPA_SUPPLICANT_KEY_MGMT_WPA2_PSK);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec(m_WpaSuppHandle, WPA_SUPPLICANT_WSEC_CCMP);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            wpaRet = wpa_supplicant_ctrl_set_wsec_key(m_WpaSuppHandle, (char*)&pConnectParams->security.key);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        default:
        {
            NN_SDK_ASSERT(false, "<WPA_SUPP> joining: unknown privacy mode %d\n", pConnectParams->security.privacyMode);
        }
    }


    if( pConnectParams->channel != -1 )
    {
        NN_SDK_REQUIRES_RANGE(pConnectParams->channel, WirelessChannel_1ch, WirelessChannel_165ch + 1);
        channel = pConnectParams->channel;
        num_channels = 1;
    }
    else
    {
        channel = 0;
        num_channels = 0;
    }

    WLAN_LOG_INFO("<WPA_SUPP> Channel = %d, num_channels = %d\n", channel, num_channels);
    wpaRet = wpa_supplicant_ctrl_set_ssid(m_WpaSuppHandle, ssidStr, num_channels, &channel);

    NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
    wpaRet = wpa_supplicant_ctrl_enable_network(m_WpaSuppHandle);
    NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");

    WLAN_LOG_DEBUG("<WPA_SUPP> [---Association command ended -----]\n");

    return (wpaRet < 0) ? false : true;
}  //NOLINT(impl/function_size)

bool WpaSupplicant::JoinNetworkWithWps(WlanWpsParameters *pWpsParams) NN_NOEXCEPT
{
    int32_t wpaRet = false;

    WLAN_LOG_DEBUG("<WPA_SUPP> -----------------Association with WPS command started ----------------\n");

    WLAN_LOG_DEBUG("<WPA_SUPP> WPS method = %s\n", (pWpsParams->method == WpsMethod_Pbc) ? "PBC" : "PIN");

    wpaRet = wpa_supplicant_ctrl_add_network(m_WpaSuppHandle);
    NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");

    switch(pWpsParams->method)
    {
        case WpsMethod_Pbc:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: WPS type = PBC\n");
            wpaRet = wpa_supplicant_ctrl_connect_wsc_pbc(m_WpaSuppHandle);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        case WpsMethod_Pin:
        {
            WLAN_LOG_INFO("<WPA_SUPP> Joining: Security type = PIN\n");
            WLAN_LOG_INFO("<WPA_SUPP> Joining: PIN code : %s\n", pWpsParams->pin);
            wpaRet = wpa_supplicant_ctrl_connect_wsc_pin(m_WpaSuppHandle, &pWpsParams->pin[0]);
            NN_ABORT_UNLESS(wpaRet >= 0, "supplicant command failure");
            break;
        }
        default:
        {
            NN_SDK_ASSERT(false, "<WPA_SUPP> joining: unknown WPS method %d\n", pWpsParams->method);
        }
    }

    WLAN_LOG_DEBUG("<WPA_SUPP> [---Association command ended -----]\n");

    return (wpaRet < 0) ? false : true;
}

bool WpaSupplicant::Disassociate() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("<WPA_SUPP> Disassociation requested!!!\n");
    int32_t wpaRet = wpa_supplicant_ctrl_disconnect(m_WpaSuppHandle, 0);

    return (wpaRet < 0) ? false : true;
}

bool WpaSupplicant::SetGtkKeyInfo() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("<WPA_SUPP> %s requested!!!\n", __FUNCTION__);
    int32_t wpaRet = wpa_supplicant_ctrl_suspend(m_WpaSuppHandle);

    return (wpaRet < 0) ? false : true;
}

bool WpaSupplicant::ClearGtkKeyInfo() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("<WPA_SUPP> %s requested!!!\n", __FUNCTION__);
    int32_t wpaRet = wpa_supplicant_ctrl_resume(m_WpaSuppHandle);

    return (wpaRet < 0) ? false : true;
}

// pass-through API
void WpaSupplicant::ForwardEvent(void* pEvent, void* pBuf) NN_NOEXCEPT
{
    void*            pDrvPriv;
    void*            pPktData;
    void*            pEvData;
    uint32_t*        pType;
    wl_event_msg_t*  pMsg;

    if( m_IsInitialized == false || m_IsSleepState == true )
    {
        return;
    }

    pType = reinterpret_cast<uint32_t *>(pEvent);
    pMsg  = reinterpret_cast<wl_event_msg_t *>(pEvent);

    WLAN_LOG_DEBUG("<WPA_SUPP> [Event forwarding] %s ver:0x%x event:0x%x len:%d status 0x%x\n",__FUNCTION__,
            pMsg->version, pMsg->event_type, pMsg->datalen, pMsg->status);

    nn::mbuf::Mbuf* pMbuf = reinterpret_cast<nn::mbuf::Mbuf *>(pBuf);
    pPktData = nn::mbuf::MbufTod(pMbuf);
    pEvData = reinterpret_cast<void *>( reinterpret_cast<char *>(pPktData) + sizeof(bcm_event_t));

    // Debug code just to monitor key events being forwarded to WPA supplicant
    switch(pMsg->event_type)
    {
        case WLC_E_JOIN:
            WLAN_LOG_DEBUG("<WPA_SUPP> [%s(%d) WLC_E_JOIN]\n",__FUNCTION__, __LINE__);
            break;

        case WLC_E_ASSOC:
            WLAN_LOG_DEBUG("<WPA_SUPP> [%s(%d) WLC_E_ASSOC]\n",__FUNCTION__, __LINE__);
            break;

        case WLC_E_DISASSOC:
            WLAN_LOG_DEBUG("<WPA_SUPP> [%s(%d) WLC_E_DISASSOC]\n",__FUNCTION__, __LINE__);
            break;

        case WLC_E_DISASSOC_IND:
            WLAN_LOG_DEBUG("<WPA_SUPP> [%s(%d) WLC_E_DISASSOC_IND]\n",__FUNCTION__, __LINE__);
            break;

        case WLC_E_DEAUTH:
            WLAN_LOG_DEBUG("<WPA_SUPP> [%s(%d) WLC_E_DEAUTH]\n",__FUNCTION__, __LINE__);
            break;

        case WLC_E_DEAUTH_IND:
            WLAN_LOG_DEBUG("<WPA_SUPP> [%s(%d) WLC_E_DEAUTH_IND]\n",__FUNCTION__, __LINE__);
            break;

        case WLC_E_ESCAN_RESULT:
            WLAN_LOG_DEBUG("<WPA_SUPP> [ESCAN_RESULT: %s(%d) ver:%d event:%d len:%d ev data 0x%x]\n",__FUNCTION__, __LINE__,
               pMsg->version, pMsg->event_type, pMsg->datalen, pEvData);
            break;

        default:
            WLAN_LOG_DEBUG("<WPA_SUPP> [UNKNOWN supplicant event: %s(%d) ver:%d event:%d len:%d evdata 0x%x\n",__FUNCTION__, __LINE__,
                pMsg->version, pMsg->event_type, pMsg->datalen, pEvData);
            break;
    }

    if (m_WpaSuppHandle)
    {
        //WLAN_LOG_DEBUG("<WPA_SUPP> [Will call WPA supplicant APIs to forward event]\n");
        // this gets a sort of handle to do the read
        pDrvPriv = wpa_supplicant_get_drv_priv(m_WpaSuppHandle);
        _bcm_event_handle_read(pDrvPriv, NULL, reinterpret_cast<char *>(pPktData) + sizeof(ether_header), 0);
    }
    else
    {
        WLAN_LOG_DEBUG("<WPA_SUPP> Nothing to do, WPA supplicant not yet initialized\n");
    }

}

// pass-through API
void WpaSupplicant::ForwardRxEapol(void* pBuf) NN_NOEXCEPT
{
    if( m_IsSleepState == true )
    {
        return;
    }
    nn::mbuf::Mbuf *pMbuf = reinterpret_cast<nn::mbuf::Mbuf *>(pBuf);
    uint8_t *pRxData = reinterpret_cast<uint8_t*>(nn::mbuf::MbufTod(pMbuf));

    WLAN_LOG_DEBUG("<WPA_SUPP> Forwarding RX_EAPOL to WPA supplicant\n");
    wpa_supplicant_rx_eapol_queue_pkt(m_WpaSuppHandle, pRxData + 6, pRxData + sizeof(ether_header), pMbuf->_len - sizeof(ether_header));
}

void WpaSupplicant::RegisterCallbacks() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("<WPA_SUPP> Supplicant will WDM register callbacks");
    // register WPA supplicant event callback
    wpa_supplicant_register_notify_cb(m_WpaSuppHandle, EventCallBackFromSupplicant);

    //register EAPOL Tx data callback
    wpa_supplicant_register_l2_send_eapol(m_WpaSuppHandle, TxEapolCallBackFromSupplicant);

    // register IOCTL callback
    wpa_supplicant_register_ioctl_cb(m_WpaSuppHandle, IoctlCallBackFromSupplicant);
}

// TODO: we need to queue up this message on WDM event queue
int WpaSupplicant::EventCallBackFromSupplicant(void *ctx, int level, int type, const char *buf, int len) NN_NOEXCEPT
{
    wl_escan_result_t *eresult = NULL;
    wl_event_msg_t *event = (wl_event_msg_t *)(buf);
    uint32_t event_type   = event->event_type;

    WLAN_LOG_DEBUG("<WPA_SUPP> Event callback from Supplicant occurred\n");

    // TODO: eventually this needs to go. It needs to be substituted by
    // queueing of event on event queue for WDM
    switch (type)
    {
        case WPA_NOTIFY_SCAN_RESULTS:
            WLAN_LOG_DEBUG("<WPA_SUPP> WPA_NOTIFY_SCAN_RESULTS RECEIVED \n");
            if( len == 0 )
            {
                WLAN_LOG_DEBUG("No BSS info!\n");
            }
            else
            {
                eresult = (wl_escan_result_t *)&event[1];
                WLAN_LOG_DEBUG("***** SCAN RESULT *****BSS count : %d\n", eresult->bss_count);
                WLAN_LOG_DEBUG("Scan Result Total Length : %d\n", len);
            }

            // queue up this event to StateMachine
            {
                WlanEvent* pevnbuff = NULL;

                pevnbuff = m_pStateMachine->WlanGetEventBuff(sizeof(bool));
                NN_SDK_ASSERT_NOT_NULL(pevnbuff);
                bool* result = static_cast<bool*>(pevnbuff->Args);
                pevnbuff->id = WLAN_EVENT_SCAN_COMPLETE;
                *result = true;

                m_pStateMachine->PostEventMessage( pevnbuff );
            }

            break;
        case WPA_NOTIFY_CONNECTED:
            WLAN_LOG_DEBUG("<WPA_SUPP> WPA_NOTIFY_CONNECTED %s\n", buf);
            if (event_type == WLC_E_ASSOC)
            {
                WLAN_LOG_INFO("<WPA_SUPP> WLC_E_ASSOC received\n");
                WLAN_LOG_INFO("<WPA_SUPP> STATUS = %d. ASSOC_STATUS = %d.\n", event->status, event->reason);
            }
            else
            {
                WLAN_LOG_INFO("<WPA_SUPP> Error, incorrect event type = %d.\n", event_type);
                NN_SDK_ASSERT(false, "Unknown event comes");
            }

            {
                // queue up this event to StateMachine
                WlanEvent* pevnbuff = NULL;

                pevnbuff = m_pStateMachine->WlanGetEventBuff(0);
                NN_SDK_ASSERT_NOT_NULL(pevnbuff);
                pevnbuff->id = WLAN_WPA_NOTIFY_SUCCESS;

                m_pStateMachine->PostEventMessage( pevnbuff );
            }

            break;
        case WPA_NOTIFY_DISCONNECTED:
            WLAN_LOG_INFO("<WPA_SUPP> WPA_NOTIFY_DISCONNECTED %s\n", buf);
            if (event_type == WLC_E_DISASSOC)
            {
                WLAN_LOG_FORCE("<WPA_SUPP> WLC_E_DISASSOC received\n");
                WLAN_LOG_FORCE("<WPA_SUPP> STATUS = %d. DISASSOC_REASON = %d.\n", event->status, event->reason);
            }
            else if (event_type == WLC_E_DISASSOC_IND)
            {
                WLAN_LOG_FORCE("<WPA_SUPP> WLC_E_DISASSOC_IND received\n");
                WLAN_LOG_FORCE("<WPA_SUPP> STATUS = %d. DISASSOC_IND_REASON = %d.\n", event->status, event->reason);
            }
            else if (event_type == WLC_E_DEAUTH)
            {
                WLAN_LOG_FORCE("<WPA_SUPP> WLC_E_DEAUTH received\n");
                WLAN_LOG_FORCE("<WPA_SUPP> STATUS = %d. DEAUTH_REASON = %d.\n", event->status, event->reason);
            }
            else if (event_type == WLC_E_DEAUTH_IND)
            {
                WLAN_LOG_FORCE("<WPA_SUPP> WLC_E_DEAUTH_IND received\n");
                WLAN_LOG_FORCE("<WPA_SUPP> STATUS = %d. DEAUTH_IND_REASON = %d.\n", event->status, event->reason);
            }
            else
            {
                WLAN_LOG_FORCE("<WPA_SUPP> Error, incorrect event type = %d.\n", event_type);
                NN_SDK_ASSERT(false, "Unknown event comes");
            }

            {
                // queue up this event to StateMachine
                WlanEvent* pevnbuff = NULL;

                pevnbuff = m_pStateMachine->WlanGetEventBuff(sizeof(WlanDisconnectInfo));
                NN_SDK_ASSERT_NOT_NULL(pevnbuff);
                WlanDisconnectInfo* pinfo = static_cast<WlanDisconnectInfo*>(pevnbuff->Args);
                pevnbuff->id = WLAN_WPA_NOTIFY_FAIURE;
                pinfo->reason = static_cast<uint16_t>(event->reason);

                m_pStateMachine->PostEventMessage( pevnbuff );
            }

            break;
        case WPS_NOTIFY_SUCCESS:
            WLAN_LOG_INFO("<WPA_SUPP> WPS_NOTIFY_SUCCESS %s\n", buf);
            WLAN_LOG_INFO("<WPA_SUPP> TYPE = %d. STATUS = %d. ASSOC_STATUS = %d.\n", event_type, event->status, event->reason);
            {
                // queue up this event to StateMachine
                WlanEvent* pevnbuff = NULL;

                pevnbuff = m_pStateMachine->WlanGetEventBuff(0);
                NN_SDK_ASSERT_NOT_NULL(pevnbuff);
                pevnbuff->id = WLAN_WPA_NOTIFY_SUCCESS;

                m_pStateMachine->PostEventMessage( pevnbuff );
            }
            break;
        case WPS_NOTIFY_FAILURE:
            WLAN_LOG_INFO("<WPA_SUPP> WPS_NOTIFY_FAILURE %s\n", buf);
            WLAN_LOG_INFO("<WPA_SUPP> TYPE = %d. STATUS = %d. ASSOC_STATUS = %d.\n", event_type, event->status, event->reason);
            {
                // queue up this event to StateMachine
                WlanEvent* pevnbuff = NULL;

                pevnbuff = m_pStateMachine->WlanGetEventBuff(sizeof(WlanDisconnectInfo));
                NN_SDK_ASSERT_NOT_NULL(pevnbuff);
                WlanDisconnectInfo* pinfo = static_cast<WlanDisconnectInfo*>(pevnbuff->Args);
                pevnbuff->id = WLAN_WPS_NOTIFY_FAIURE;
                pinfo->reason = Dot11ReasonCode_UnspecifiedFailure;

                m_pStateMachine->PostEventMessage( pevnbuff );
            }
            break;
        case WPA_NOTIFY_GTK_KEY_SET:
            WLAN_LOG_INFO("<WPA_SUPP> WPA_NOTIFY_GTK_KEY_SET %s\n", buf);
            WLAN_LOG_INFO("<WPA_SUPP> STATUS = %d.\n", event->status);
            {
                // queue up this event to StateMachine
                WlanEvent* pevnbuff = NULL;

                pevnbuff = m_pStateMachine->WlanGetEventBuff(0);
                NN_SDK_ASSERT_NOT_NULL(pevnbuff);
                pevnbuff->id = WLAN_WPA_NOTIFY_GTK_SET;
                if( event->status == 0 )
                {
                    pevnbuff->Result = true;
                }
                else
                {
                    pevnbuff->Result = false;
                }
                m_pStateMachine->PostEventMessage( pevnbuff );
            }
            break;
        case WPA_NOTIFY_GTK_KEY_CLEAR:
            WLAN_LOG_INFO("<WPA_SUPP> WPA_NOTIFY_GTK_KEY_CLEAR %s\n", buf);
            WLAN_LOG_INFO("<WPA_SUPP> STATUS = %d.\n", event->status);
            {
                // queue up this event to StateMachine
                WlanEvent* pevnbuff = NULL;

                pevnbuff = m_pStateMachine->WlanGetEventBuff(0);
                NN_SDK_ASSERT_NOT_NULL(pevnbuff);
                pevnbuff->id = WLAN_WPA_NOTIFY_GTK_CLEAR;
                if( event->status == 0 )
                {
                    pevnbuff->Result = true;
                }
                else
                {
                    pevnbuff->Result = false;
                }
                m_pStateMachine->PostEventMessage( pevnbuff );
            }
            break;
        case WPA_NOTIFY_CONNECT_START:
            WLAN_LOG_INFO("<WPA_SUPP> WPA_NOTIFY_CONNECT_START %s\n", buf);
            WLAN_LOG_INFO("<WPA_SUPP> STATUS = %d.\n", event->status);
            if( event->status == 0 )
            {
                // queue up this event to StateMachine
                WlanEvent* pevnbuff = NULL;

                pevnbuff = m_pStateMachine->WlanGetEventBuff(0);
                NN_SDK_ASSERT_NOT_NULL(pevnbuff);
                pevnbuff->id = WLAN_WPA_NOTIFY_CONNECT_START;
                pevnbuff->Result = true;
                m_pStateMachine->PostEventMessage( pevnbuff );
            }
            else
            {
                // 想定外の挙動
                NN_ABORT("already associated or unknown status\n");
            }
            break;
        case WPA_NOTIFY_CONNECT_ABORT:
            WLAN_LOG_INFO("<WPA_SUPP> WPA_NOTIFY_CONNECT_ABORT %s\n", buf);
            WLAN_LOG_INFO("<WPA_SUPP> STATUS = %d.\n", event->status);
            {
                // queue up this event to StateMachine
                WlanEvent* pevnbuff = NULL;

                pevnbuff = m_pStateMachine->WlanGetEventBuff(0);
                NN_SDK_ASSERT_NOT_NULL(pevnbuff);
                pevnbuff->id = WLAN_WPA_NOTIFY_CONNECT_ABORT;
                pevnbuff->Result = true;
                m_pStateMachine->PostEventMessage( pevnbuff );
            }
            break;
        default:
            WLAN_LOG_DEBUG("<WPA_SUPP> Error, unrecognized event type = %d.\n", event_type);
    }

    return 0;
}  //NOLINT(impl/function_size)

// pass-through API
int WpaSupplicant::TxEapolCallBackFromSupplicant(void *buf) NN_NOEXCEPT
{
    WLAN_LOG_INFO("<WPA_SUPP> Forwarding Tx EAPOLs from supplicant\n");
    NN_SDK_ASSERT_NOT_NULL(buf);
    nn::mbuf::Mbuf* pMbuf = reinterpret_cast<nn::mbuf::Mbuf*>(buf);
    nn::Result result = m_pStateMachine->PutFrameForce(&pMbuf);
    if( ResultTxQueueIsFull().Includes(result) == true )
    {
        WLAN_LOG_ERROR("%s: Tx queue is full\n", __FUNCTION__);
        if( pMbuf != NULL )
        {
            nn::mbuf::MbufFreem(pMbuf);
        }
    }
    return 0;
}


// pass-through API
int WpaSupplicant::IoctlCallBackFromSupplicant(void *ifh, int cmd, void *buf, int len, bool set) NN_NOEXCEPT
{
    int ret = 0;
    if( m_IsSleepState == false )
    {
        // Here we call Broadcom-API to pass IOCTls
        dhd_ioctl_entry(ifh, cmd, buf, len, set);
    }
    else
    {
        ret = -1;
    }
    return ret;
}

}} // end of 'namespace nn { namespace wlan {'



