﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/TargetConfigs/build_Base.h>

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#endif

#include "wlan_DebugLog.h"
#include "wlan_LockableMutexType.h"
#include "wlan_Settings.h"

namespace nn { namespace wlan {

namespace {

//!< ファームウェアデバッグ設定の設定値を表す構造体です。
struct FirmwareDebugSettingsValue
{
    //!< wlan プロセスのブートが制限されているかどうか
    bool isSkipBoot;
};

//!< ファームウェアデバッグ設定の設定値が初期化されているか否かを表す値
::std::atomic<bool> g_IsFirmwareDebugSettingsValueInitialized(false);

//!< ファームウェアデバッグ設定の設定値に対する操作の排他用ミューテックス
LockableMutexType g_FirmwareDebugSettingsValueMutex =
{
    NN_OS_MUTEX_INITIALIZER(false)
};

//!< ファームウェアデバッグ設定の設定値
FirmwareDebugSettingsValue g_FirmwareDebugSettingsValue;

//!< ファームウェアデバッグ設定の設定値を初期化します。
void InitializeFirmwareDebugSettingsValue() NN_NOEXCEPT;

} // namespace

void FirmwareDebugSettings::Initialize() NN_NOEXCEPT
{
    if (g_IsFirmwareDebugSettingsValueInitialized)
    {
        // 初期化済みならば何もしない
        return;
    }

    ::std::lock_guard<decltype(g_FirmwareDebugSettingsValueMutex)
                      > locker(g_FirmwareDebugSettingsValueMutex);

    if (g_IsFirmwareDebugSettingsValueInitialized)
    {
        // 排他された状態で初期化済みか否かを再確認
        return;
    }

    InitializeFirmwareDebugSettingsValue();

    g_IsFirmwareDebugSettingsValueInitialized = true;
}

bool FirmwareDebugSettings::IsSkipBoot() NN_NOEXCEPT
{
    Initialize();
    return g_FirmwareDebugSettingsValue.isSkipBoot;
}

namespace {

void InitializeFirmwareDebugSettingsValue() NN_NOEXCEPT
{
    const char* const WlanDebugSettingsName = "wlan_debug";

    // 初期設定ではwlanのブートはスキップしないとしておく
    g_FirmwareDebugSettingsValue.isSkipBoot = false;

    ::nn::settings::fwdbg::GetSettingsItemValue(
        &g_FirmwareDebugSettingsValue.isSkipBoot,
        sizeof(g_FirmwareDebugSettingsValue.isSkipBoot),
        WlanDebugSettingsName, "skip_wlan_boot");

    WLAN_LOG_INFO("wlan skip : %s\n", (g_FirmwareDebugSettingsValue.isSkipBoot == true) ? "skip" : "not skip");
}

} // namespace

}} // namespace nn::wlan
