﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/mbuf/mbuf_MbufInit.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_MacAddress.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_StateDefinition.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/wlan/driver/wlan_DriverTypes.h>

#include "wlan_LocalApiImpl.h"
#include "../wlan_MsgBuffer.h"
#include "../wlan_IeParser.h"
#include "../wlan_SignalStrength.h"
#include "../driver/wlan_Driver.h"
#include "../wlan_MemoryInit.h"

namespace nn { namespace wlan { namespace server
{
    nn::Result LocalApiImpl::OpenMasterMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalMasterIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }
        return ChangeMode(WLAN_CHANGE_STATE_LOCAL_AP);
    }

    nn::Result LocalApiImpl::CloseMasterMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalMasterIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }
        return ChangeMode(WLAN_CHANGE_STATE_READY);
    }

    nn::Result LocalApiImpl::OpenClientMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalClientIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }
        return ChangeMode(WLAN_CHANGE_STATE_LOCAL_STA);
    }

    nn::Result LocalApiImpl::CloseClientMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalClientIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }
        return ChangeMode(WLAN_CHANGE_STATE_READY);
    }

    nn::Result LocalApiImpl::OpenSpectatorMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalSpectatorIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }
        return ChangeMode(WLAN_CHANGE_STATE_LOCAL_SPECTATOR);
    }

    nn::Result LocalApiImpl::CloseSpectatorMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalSpectatorIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }
        return ChangeMode(WLAN_CHANGE_STATE_READY);
    }

    nn::Result LocalApiImpl::CreateBss(const nn::wlan::detail::SfdlMasterBssParameters& bssParameters) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( !(bssParameters.channel == -1 ||
            (bssParameters.channel >= WirelessChannel_1ch && bssParameters.channel <= WirelessChannel_11ch) ||
            bssParameters.channel == WirelessChannel_36ch ||
            bssParameters.channel == WirelessChannel_40ch ||
            bssParameters.channel == WirelessChannel_44ch ||
            bssParameters.channel == WirelessChannel_48ch)
        )
        {
            NN_SDK_REQUIRES(false, "channel is invalid");
            return ResultInvalidArgument();
        }
        if( bssParameters.inactivePeriod > 180 )
        {
            NN_SDK_REQUIRES(false, "inactivePeriod must be less than 180");
            return ResultInvalidArgument();
        }
        if( !(bssParameters.security.privacyMode == SecurityMode_Open || bssParameters.security.privacyMode == SecurityMode_StaticAes) )
        {
            NN_SDK_REQUIRES(false, "privacyMode must be Open or StaticAES");
            return ResultInvalidArgument();
        }
        if( bssParameters.ssid.length == 0 )
        {
            NN_SDK_REQUIRES(false, "SSID must not be empty");
            return ResultInvalidArgument();
        }

        // ステートチェック
        WlanStates currentState;
        GetWlanState(&currentState);
        if( currentState.mainState == WdrvMainState_OutOfService && currentState.subState == WdrvSubState_None )
        {
            return ResultWlanDeviceAbnormal();
        }
        if( !((currentState.mainState == WdrvMainState_LocalMasterIdle || currentState.mainState == WdrvMainState_LocalLcsMasterIdle) &&
            (currentState.subState == WdrvSubState_None || currentState.subState == WdrvSubState_Idle))
        )
        {
            return ResultInvalidState();
        }

        nn::Result result;
        WlanCommand* pcmdbuff;

        /*
         * APの設定を行う
         * 内部でSet channelをするので無線チップはIDLEステート(I/Fがdownの状態)の必要あり
         */

        // APの設定をする -------------------------------------------------------------------------
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanLocalBssConfiguration));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        WlanLocalBssConfiguration* pbssinfo = static_cast<WlanLocalBssConfiguration*>(pcmdbuff->Args);

        // 設定情報のコピー
        pbssinfo->channel = bssParameters.channel;
        pbssinfo->hiddenSsid = bssParameters.hiddenSsid;

        pcmdbuff->id = WLAN_SET_APCONFIGRATION;

        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        if( result.IsFailure() )
        {
            return result;
        }
        // --------------------------------------------------------------------------------------

        // AESモードの設定 --------------------------------------------------------------------------
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(bool));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        bool* pMode = static_cast<bool*>(pcmdbuff->Args);
        pcmdbuff->id = WLAN_SET_STATICAES_MODE;
        if( bssParameters.security.privacyMode == SecurityMode_StaticAes )
        {
            // StaticAES keyをWDMに覚えさせておく
            m_pStateMachine->SetStaticAesKey(bssParameters.security.key);
            *pMode = true;
        }
        else
        {
            *pMode = false;
        }
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        if( result.IsFailure() )
        {
            return result;
        }
        // --------------------------------------------------------------------------------------

        // interfaceのUPを要求する ------------------------------------------------------------------
        result = RequestIfUpDown(1);
        if( result.IsFailure() )
        {
            return result;
        }
        // --------------------------------------------------------------------------------------

        // Request to execute ACSD if channel is -1 ----------------------------------------------
        if( bssParameters.channel == -1 )
        {
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
            NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_EXECUTE_ACSD;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            result = pcmdbuff->nnResult;
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
            if( result.IsFailure() )
            {
                return result;
            }
        }
        // --------------------------------------------------------------------------------------

        // AP開始要求 ------------------------------------------------------------------------------
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanSsidInfo));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        WlanSsidInfo* pSsid = static_cast<WlanSsidInfo*>(pcmdbuff->Args);

        pcmdbuff->id = WLAN_CREATE_AP;
        std::memcpy(pSsid->ssid, bssParameters.ssid.data, static_cast<size_t>(bssParameters.ssid.length));
        pSsid->length = static_cast<int>(bssParameters.ssid.length);
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        if( result.IsFailure() )
        {
            return result;
        }
        // --------------------------------------------------------------------------------------

        // Inactive period 設定 ------------------------------------------------------------------
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(uint32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        uint32_t* pTime = static_cast<uint32_t*>(pcmdbuff->Args);

        pcmdbuff->id = WLAN_SET_CLIENT_TIMEOUT;
        *pTime = bssParameters.inactivePeriod;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        if( result.IsFailure() )
        {
            // 作成したBSSを破棄しておく
            DestroyBss();
        }
        // --------------------------------------------------------------------------------------

        return result;
    } //NOLINT(impl/function_size)

    nn::Result LocalApiImpl::DestroyBss() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WlanStates currentState;
        GetWlanState(&currentState);
        switch( currentState.mainState )
        {
        case WdrvMainState_LocalMasterIdle:
        case WdrvMainState_LocalLcsMasterIdle:
            // do nothing
            return ResultSuccess();
        case WdrvMainState_LocalMasterBss:
        case WdrvMainState_LocalLcsMasterBss:
            switch( currentState.subState )
            {
            case WdrvSubState_Connected:
                break;
            default:
                return ResultInvalidState();
            }
            break;
        default:
            return ResultInvalidState();
        }

        WlanCommand* pcmdbuff;
        nn::Result result;

        // まずは接続済みのClientにDeauthを投げる
        nn::wlan::detail::SfdlDisconnectClient client = {
                {{ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF }},
                Dot11ReasonCode_DisassocUnableToHandle
        };
        result = Disconnect(LocalCommunicationMode_Master, client);
        if( ResultInvalidState().Includes(result) == true )
        {
            NN_SDK_ASSERT( false, "Disconnect is called in invalid state\n" );
            return result;
        }

        // BeaconActionFrameを送信していたら停止する
        result = CancelActionFrameWithBeacon();
        if( result.IsFailure() )
        {
            return result;
        }

        // AP終了要求
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_DESTROY_AP;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result LocalApiImpl::StartScan(const nn::wlan::detail::SfdlScanParameters& scanParam) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // パラメータチェック
        if( !(scanParam.scanType == static_cast<uint32_t>(ScanType_Active) ||
                scanParam.scanType == static_cast<uint32_t>(ScanType_Passive)) )
        {
            NN_SDK_REQUIRES(false, "scanType is invalid");
            return ResultInvalidArgument();
        }
        if( !(scanParam.channelScanTime == -1 || (scanParam.channelScanTime >= 10 && scanParam.channelScanTime <= 3000)) )
        {
            NN_SDK_REQUIRES(false, "channelScanTime is invalid");
            return ResultInvalidArgument();
        }
        if( !(scanParam.homeChannelTime == -1 || (scanParam.homeChannelTime >= 0 && scanParam.homeChannelTime <= 500)) )
        {
            NN_SDK_REQUIRES(false, "homeChannelTime is invalid");
            return ResultInvalidArgument();
        }
        if( std::memcmp(&scanParam.bssid.data[0], MacAddress::CreateZeroMacAddress().GetMacAddressData(),
                MacAddress::MacAddressSize) == 0 )
        {
            NN_SDK_REQUIRES(false, "bssid must not be ZeroMacAddress");
            return ResultInvalidArgument();
        }
        // スキャンチャンネルチェック
        NN_RESULT_DO(m_pStateMachine->CheckChannels(scanParam.channelList.channel, scanParam.channelCount));

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;

        // I/FのUP要求
        // 既にUPされていたら、スルーされる
        result = RequestIfUpDown(1);
        if( result.IsFailure() )
        {
            return result;
        }

        // scanリクエスト送る
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanScanParameters));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

        pcmdbuff->id = WLAN_SCAN_REQUEST;
        WlanScanParameters* params = static_cast<WlanScanParameters*>(pcmdbuff->Args);

        // scanパラメータのコピー
        if( scanParam.channelCount > 0 )  // channelCountのコピー
        {
            for(int i = 0; i < scanParam.channelCount; i++)
            {
                params->channelList[i] = scanParam.channelList.channel[i];
            }
        }
        if( scanParam.ssidCount > 0 )  // SSIDリストのコピー
        {
            for(int i = 0; i < scanParam.ssidCount; i++)
            {
                params->ssidList[i].length = scanParam.ssidList[i].length;
                std::memcpy(&params->ssidList[i].ssid[0], &scanParam.ssidList[i].data[0], scanParam.ssidList[i].length);
            }
        }
        params->ssidCount = scanParam.ssidCount;
        std::memcpy(&params->bssid[0], &scanParam.bssid.data[0], nn::wlan::MacAddress::MacAddressSize); // ZeroMacはShimではじく
        params->scanType = scanParam.scanType;
        params->channelScanTime = scanParam.channelScanTime;
        params->homeChannelTime = scanParam.homeChannelTime;
        params->channelCount = scanParam.channelCount;

        WLAN_LOG_DEBUG("[%s] scan request post.\n", __FUNCTION__);
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff); // コマンドバッファの削除

        return result;
    } //NOLINT(impl/function_size)

    nn::Result LocalApiImpl::GetScanResult(const nn::sf::OutBuffer& pOutScanBuffer, const nn::wlan::ScanIeMatchInfo& ieInfo) NN_NOEXCEPT
    {
        NN_UNUSED(ieInfo);
        // パラメータチェック
        if( pOutScanBuffer.GetSize() <= sizeof(ScanResultHeader) )
        {
            NN_SDK_REQUIRES(false, "%s: scan buffer size must be larger than %d",
                    __FUNCTION__, sizeof(ScanResultHeader));
            return ResultInvalidArgument();
        }

        WlanCommand* pcmdbuff;

        /*
         * スキャン結果の取得
         */
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanBssInfo*));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_SCANLIST_REQUEST;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff->Args);

        WlanBssInfo** pbsslist = static_cast<WlanBssInfo**>(pcmdbuff->Args);
        WlanBssInfo* pBssInfo = *pbsslist;

        uint8_t* pScanBuf = reinterpret_cast<uint8_t*>(pOutScanBuffer.GetPointerUnsafe());
        // バッファの先頭にスキャン結果全体に関する情報を入れるためのヘッダを配置
        ScanResultHeader* resultHeader = reinterpret_cast<ScanResultHeader*>(pScanBuf);
        resultHeader->bufferLength = pOutScanBuffer.GetSize();
        resultHeader->bssCount = 0;
        resultHeader->resultLength = sizeof(ScanResultHeader);

        // ポインタを進める
        pScanBuf += sizeof(ScanResultHeader);

        if( pBssInfo == NULL )
        {
            // 1個もBSSが見つからなかったということ
            WLAN_LOG_DEBUG("Can't find any BSS\n");
        }
        else
        {
            size_t capacity = pOutScanBuffer.GetSize();  // スキャンバッファの空き容量を記録する変数

            WLAN_LOG_DEBUG("Scan buff capacity is %d\n", capacity);

            capacity -= sizeof(ScanResultHeader);  // ヘッダ分を削っておく

            // スキャン結果を掘っていき、BSSがスキャンバッファに収まるようであれば、収めていく。
            // スキャンバッファが足りなくなったらBSSの格納を止める。収まりきらなかったBSSは上位層には渡らない。
            while(pBssInfo != NULL)
            {
                // BSSのサイズを取得
                IeParser* pParser = NULL;
                size_t bssSize = sizeof(nn::wlan::BssInfo); // IE以外の基本情報を格納する構造体のサイズ

                // IEのサイズを取得
                // VIEに限定せず全てのタグIEを含める
                uint32_t ieCount = 0;
                if( pBssInfo->IeData != NULL )
                {
                    pParser = new IeParser(pBssInfo->IeData, pBssInfo->IeLength);
                    NN_SDK_ASSERT_NOT_NULL(pParser);

                    uint8_t* rp;
                    uint8_t id;
                    uint8_t length;
                    rp = pParser->SearchIe(-1);
                    while(rp != NULL)
                    {
                        pParser->ReadIdAndLen(&id, &length, rp);
                        bssSize += sizeof(uint8_t); // IE ID サイズ
                        bssSize += sizeof(uint8_t); // IE length サイズ
                        bssSize += length;         // IE body の長さ
                        ieCount++;
                        rp = pParser->SearchIeContinue(-1);
                    }
                    delete pParser;
                    WLAN_LOG_DEBUG("BSS size is %d\n", bssSize);
                }

                // 上位層から与えられたScanBufferの空き容量に収まれば、BSS情報を格納する
                if( capacity >= bssSize )
                {
                    nn::wlan::BssInfo* pInfo = reinterpret_cast<nn::wlan::BssInfo*>(pScanBuf);
                    pInfo->beaconPeriod = pBssInfo->BeaconPeriod;
                    pInfo->bssSize = bssSize;
                    std::memcpy(&pInfo->bssid[0], &pBssInfo->Bssid[0], nn::wlan::MacAddress::MacAddressSize);
                    pInfo->ssidLength = pBssInfo->SsidInfo.length;
                    std::memcpy(&pInfo->ssid[0], &pBssInfo->SsidInfo.ssid[0], pBssInfo->SsidInfo.length);
                    pInfo->capability = pBssInfo->Capability;
                    pInfo->channel = pBssInfo->Channel;
                    pInfo->ieCount = ieCount;
                    pInfo->rateCount = pBssInfo->RateCount;
                    std::memset(&pInfo->rates[0], 0, sizeof(pInfo->rates));  // 先に0埋めしておく
                    std::memcpy(&pInfo->rates[0], &pBssInfo->Rates[0], pBssInfo->RateCount);
                    pInfo->rssi = pBssInfo->Rssi;
                    pInfo->linkLevel = static_cast<uint32_t>(SignalStrength(pBssInfo->Rssi).GetLinkLevelLocal());

                    // ポインタを進める
                    pScanBuf += sizeof(nn::wlan::BssInfo);

                    // IEのコピー
                    if( pBssInfo->IeData != NULL )
                    {
                        pParser = new IeParser(pBssInfo->IeData, pBssInfo->IeLength);
                        NN_SDK_ASSERT_NOT_NULL(pParser);

                        uint8_t* rp;
                        uint8_t id;
                        uint8_t length;
                        rp = pParser->SearchIe(-1);
                        while(rp != NULL)
                        {
                            pParser->ReadIdAndLen(&id, &length, rp);

                            // IE ID のコピー
                            *reinterpret_cast<uint8_t*>(pScanBuf) = id;
                            // ポインタを進める
                            pScanBuf += sizeof(uint8_t);

                            // IE length のコピー
                            *reinterpret_cast<uint8_t*>(pScanBuf) = length;
                            // ポインタを進める
                            pScanBuf += sizeof(uint8_t);

                            // VIE 本体のコピー（OUI以降）
                            std::memcpy(pScanBuf, pParser->GetPayloadAddress(rp), length);
                            // ポインタを進める
                            pScanBuf += length;

                            rp = pParser->SearchIeContinue(-1);
                        }
                        delete pParser;
                    }

                    capacity -= bssSize;
                    resultHeader->bssCount += 1;
                    resultHeader->resultLength += bssSize;
                }
                else
                {
                    // Scan buffer サイズが足りない
                    WLAN_LOG_DEBUG("Scan buffer does not have an empty space to store a BSS info\n");
                    break;
                }

                pBssInfo = pBssInfo->NextItem;
            }
            WLAN_LOG_DEBUG("Scan buff capacity is %d left\n", capacity);
            WLAN_LOG_DEBUG("BSS info count : %d\n", resultHeader->bssCount);
        }

        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff); // コマンドバッファの削除

        NN_RESULT_SUCCESS;

    } //NOLINT(impl/function_size)

    nn::Result LocalApiImpl::StopScan() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_CANCEL_SCAN;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);

        return result;
    }

    nn::Result LocalApiImpl::Connect(const nn::wlan::detail::SfdlSsid& ssid, nn::wlan::detail::SfdlMacAddress bssid,
            int16_t channel, const nn::wlan::detail::SfdlSecurity& security, bool autoKeepAlive, uint32_t beaconInd, int32_t beaconLostTimeout) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        NN_UNUSED(autoKeepAlive);

        // パラメータチェック
        if( ssid.length > Ssid::SsidLengthMax )
        {
            NN_SDK_REQUIRES(false, "SSID size must be less than %d", Ssid::SsidLengthMax + 1);
            return ResultInvalidArgument();
        }
        if( std::memcmp(&bssid.data[0], MacAddress::CreateZeroMacAddress().GetMacAddressData(),
                MacAddress::MacAddressSize) == 0 )
        {
            NN_SDK_REQUIRES(false, "bssid must not be ZeroMacAddress");
            return ResultInvalidArgument();
        }
        else if( std::memcmp(&bssid.data[0], MacAddress::CreateBroadcastMacAddress().GetMacAddressData(),
                MacAddress::MacAddressSize) == 0 )
        {
            // SSIDチェック
            if( ssid.length == 0 )
            {
                NN_SDK_REQUIRES(false, "SSID must not be empty");
                return ResultInvalidArgument();
            }
        }
        if( !(channel == -1 || (channel >= WirelessChannel_1ch && channel <= WirelessChannel_165ch)) )
        {
            NN_SDK_REQUIRES(false, "channel is invalid");
            return ResultInvalidArgument();
        }
        if( !(security.privacyMode == SecurityMode_Open || security.privacyMode == SecurityMode_StaticAes) )
        {
            NN_SDK_REQUIRES(false, "privacyMode is invalid");
            return ResultInvalidArgument();
        }
        if( !(beaconLostTimeout >= 1 && beaconLostTimeout <= 30) )
        {
            NN_SDK_REQUIRES(false, "beaconLostTimeout is invalid");
            return ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalLcsClientIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;

        // I/FのUP要求
        // 既にUPされていたら、スルーされる
        result = RequestIfUpDown(1);
        if( result.IsFailure() )
        {
            return result;
        }

        // Set beacon lost timeout
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(int32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_SET_BEACONLOST_TIMEOUT;
        int32_t* pTimeout = reinterpret_cast<int32_t*>(pcmdbuff->Args);
        NN_SDK_REQUIRES_RANGE(beaconLostTimeout, 1, 31);
        *pTimeout = beaconLostTimeout;
        m_pStateMachine->PostCommandMessage( pcmdbuff );
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        if( result.IsFailure() )
        {
            return result;
        }

        // AESモードの設定 --------------------------------------------------------------------------
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(bool));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        bool* pMode = static_cast<bool*>(pcmdbuff->Args);
        pcmdbuff->id = WLAN_SET_STATICAES_MODE;
        if( security.groupPrivacyMode == SecurityMode_StaticAes )
        {
            // StaticAES keyをWDMに覚えさせておく
            nn::Bit8 staticAesKey[16];
            std::memcpy(&staticAesKey[0], &security.key[0], 16);
            m_pStateMachine->SetStaticAesKey(staticAesKey);

            *pMode = true;
        }
        else
        {
            *pMode = false;
        }
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        if( result.IsFailure() )
        {
            return result;
        }
        // --------------------------------------------------------------------------------------

        // 接続試行
        WLAN_LOG_DEBUG("[%s] start to connect\n", __FUNCTION__);
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanConnectinoParameters));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_JOIN_NETWORK_STA;
        WlanConnectinoParameters* pConnectParam = reinterpret_cast<WlanConnectinoParameters*>(pcmdbuff->Args);

        NN_SDK_REQUIRES(ssid.length <= nn::wlan::Ssid::SsidLengthMax);

        // 接続情報のコピー
        pConnectParam->ssid.Set(ssid.data, ssid.length);
        pConnectParam->bssid.Set(bssid.data);
        pConnectParam->channel = channel;
        pConnectParam->beaconInd = static_cast<BeaconIndication>(beaconInd);

        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);

        return result;
    } //NOLINT(impl/function_size)

    nn::Result LocalApiImpl::CancelConnect() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalLcsClientIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;

        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_CANCEL_JOIN_NETWORK;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result LocalApiImpl::Join(const nn::wlan::detail::SfdlSsid& ssid, nn::wlan::detail::SfdlMacAddress bssid, int16_t channel,
            const nn::wlan::detail::SfdlSecurity& security, uint32_t beaconInd, int32_t beaconLostTimeout) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // パラメータチェック
        if( ssid.length == 0 )
        {
            NN_SDK_REQUIRES(false, "SSID must not be empty");
            return ResultInvalidArgument();
        }
        if( ssid.length > Ssid::SsidLengthMax )
        {
            NN_SDK_REQUIRES(false, "SSID size must be less than %d", Ssid::SsidLengthMax + 1);
            return ResultInvalidArgument();
        }
        if( std::memcmp(&bssid.data[0], MacAddress::CreateZeroMacAddress().GetMacAddressData(),
                MacAddress::MacAddressSize) == 0 )
        {
            NN_SDK_REQUIRES(false, "bssid must not be ZeroMacAddress");
            return ResultInvalidArgument();
        }
        if( !(channel == -1 || (channel >= WirelessChannel_1ch && channel <= WirelessChannel_165ch)) )
        {
            NN_SDK_REQUIRES(false, "channel is invalid");
            return ResultInvalidArgument();
        }
        if( !(security.privacyMode == SecurityMode_Open || security.privacyMode == SecurityMode_StaticAes) )
        {
            NN_SDK_REQUIRES(false, "privacyMode is invalid");
            return ResultInvalidArgument();
        }
        if( !(beaconLostTimeout >= 1 && beaconLostTimeout <= 30) )
        {
            NN_SDK_REQUIRES(false, "beaconLostTimeout is invalid");
            return ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalSpectatorIdle
        };
        nn::Result result = CheckState(states, 1);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;

        // I/FのUP要求
        // 既にUPされていたら、スルーされる
        result = RequestIfUpDown(1);
        if( result.IsFailure() )
        {
            return result;
        }

        // Set beacon lost timeout
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(int32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_SET_BEACONLOST_TIMEOUT;
        int32_t* pTimeout = reinterpret_cast<int32_t*>(pcmdbuff->Args);
        NN_SDK_REQUIRES_RANGE(beaconLostTimeout, 1, 31);
        *pTimeout = beaconLostTimeout;
        m_pStateMachine->PostCommandMessage( pcmdbuff );
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        if( result.IsFailure() )
        {
            return result;
        }

        // AESモードの設定 --------------------------------------------------------------------------
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(bool));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        bool* pMode = static_cast<bool*>(pcmdbuff->Args);
        pcmdbuff->id = WLAN_SET_STATICAES_MODE;
        if( security.groupPrivacyMode == SecurityMode_StaticAes )
        {
            // StaticAES keyをWDMに覚えさせておく
            nn::Bit8 staticAesKey[16];
            std::memcpy(&staticAesKey[0], &security.key[0], 16);
            m_pStateMachine->SetStaticAesKey(staticAesKey);

            *pMode = true;
        }
        else
        {
            *pMode = false;
        }
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        if( result.IsFailure() )
        {
            return result;
        }
        // --------------------------------------------------------------------------------------

        // 接続試行
        WLAN_LOG_DEBUG("[%s] start to connect\n", __FUNCTION__);
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanConnectinoParameters));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_JOIN_NETWORK_SPECTATOR;
        WlanConnectinoParameters* pConnectParam = reinterpret_cast<WlanConnectinoParameters*>(pcmdbuff->Args);

        NN_SDK_REQUIRES(ssid.length <= nn::wlan::Ssid::SsidLengthMax);

        // 接続情報のコピー
        pConnectParam->ssid.Set(ssid.data, ssid.length);
        pConnectParam->bssid.Set(bssid.data);
        pConnectParam->channel = channel;
        pConnectParam->beaconInd = static_cast<BeaconIndication>(beaconInd);

        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);

        return result;
    }

    nn::Result LocalApiImpl::CancelJoin() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalSpectatorIdle
        };
        nn::Result result = CheckState(states, 1);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;

        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_CANCEL_JOIN_NETWORK;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result LocalApiImpl::Disconnect(uint32_t mode, const nn::wlan::detail::SfdlDisconnectClient& client) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        nn::Result result;
        WlanCommand* pcmdbuff;

        if( mode == LocalCommunicationMode_Master )
        {
            // パラメータチェック
            if( std::memcmp(client.clientMacAddress.data,
                            MacAddress::CreateZeroMacAddress().GetMacAddressData(),
                            MacAddress::MacAddressSize) == 0 )
            {
                NN_SDK_REQUIRES(false, "clientMacAddress must not be ZeroMac");
                return ResultInvalidArgument();
            }

            // ステートチェック
            WdrvMainState states[] = {
                    WdrvMainState_LocalMasterIdle,
                    WdrvMainState_LocalMasterBss,
                    WdrvMainState_LocalLcsMasterIdle,
                    WdrvMainState_LocalLcsMasterBss
            };
            result = CheckState(states, 4);
            if( result.IsFailure() )
            {
                return result;
            }
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanDisconnectInfo));
            NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_DEAUTHENTICATE;
            WlanDisconnectInfo* deauthParam = static_cast<WlanDisconnectInfo*>(pcmdbuff->Args);
            deauthParam->reason = client.reasonCode;
            std::memcpy(&deauthParam->PeerAddr[0], &client.clientMacAddress.data[0], MacAddress::MacAddressSize);
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            result = pcmdbuff->nnResult;
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        }
        else if( mode == LocalCommunicationMode_ClientSpectator )
        {
            // ステートチェック
            WdrvMainState states[] = {
                    WdrvMainState_LocalClientIdle,
                    WdrvMainState_LocalClient,
                    WdrvMainState_LocalSpectatorIdle,
                    WdrvMainState_LocalSpectator,
                    WdrvMainState_LocalLcsClientIdle,
                    WdrvMainState_LocalLcsClient
            };
            result = CheckState(states, 6);
            if( result.IsFailure() )
            {
                return result;
            }

            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
            NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_DISASSOCIATE;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            result = pcmdbuff->nnResult;
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        }
        else
        {
            // ありえないのでABORTにする
            NN_ABORT("Invalid argument.\n");
        }

        return result;
    }

    nn::Result LocalApiImpl::SetBeaconLostCount(uint32_t count) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        NN_UNUSED(count);
        NN_ABORT("Sorry. This function is not implemented yet.");
        NN_RESULT_SUCCESS;
    }

    nn::Result LocalApiImpl::GetSystemEvent(nn::sf::Out<nn::sf::NativeHandle> pOutSystemEventReadableHandle, std::uint32_t type) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        if( type < detail::SystemEventType_ConnectionEvent || type > detail::SystemEventType_ConnectionComplete )
        {
            NN_SDK_ASSERT(false, "%s: type is invalid", __FUNCTION__);
            return ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 11);
        if( result.IsFailure() )
        {
            return result;
        }

        pOutSystemEventReadableHandle.Set(
                nn::sf::NativeHandle(m_pStateMachine->GetSystemEventReadableHandle(static_cast<detail::SystemEventType>(type)),
                        false)
        );

        NN_RESULT_SUCCESS;
    }

    nn::Result LocalApiImpl::GetConnectionStatus(nn::sf::Out<nn::wlan::detail::SfdlConnectionStatus> pOutStatus) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle,
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        // 出力用
        ConnectionStatus status;
        m_pStateMachine->GetConnectionStatus(&status);

        // コピー
        pOutStatus->state = static_cast<uint32_t>(status.state);
        pOutStatus->cause = static_cast<uint32_t>(status.cause);
        pOutStatus->channel = status.channel;
        std::memcpy(pOutStatus->bssid.data, status.bssid.GetMacAddressData(), MacAddress::MacAddressSize);
        pOutStatus->ssid.length = static_cast<uint8_t>(status.ssid.GetLength());
        std::memcpy(pOutStatus->ssid.data, status.ssid.GetSsidData(), status.ssid.GetLength());
        pOutStatus->aid = status.aid;
        pOutStatus->statusReasonCode = status.statusReasonCode;
        pOutStatus->capabilityInfo = status.capabilityInfo;
        pOutStatus->beaconInterval = status.beaconInterval;

        NN_RESULT_SUCCESS;
    }

    nn::Result LocalApiImpl::GetClientStatus(nn::sf::Out<nn::wlan::detail::SfdlClientStatusList> pOutStatusList) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 4);
        if( result.IsFailure() )
        {
            return result;
        }

        // 出力用
        WlanClientStatusInfo statusInfo;
        m_pStateMachine->GetClientStatus(&statusInfo);

        // コピー
        pOutStatusList->updatedIndexBitMap = statusInfo.updatedIndexBitMap;

        for(uint8_t i = 0; i < ConnectableClientsCountMax; i++)
        {
            pOutStatusList->statusList[i].state = static_cast<uint32_t>(statusInfo.status[i].state);
            pOutStatusList->statusList[i].cause = static_cast<uint32_t>(statusInfo.status[i].cause);
            std::memcpy(pOutStatusList->statusList[i].clientMacAddress.data,
                    statusInfo.status[i].clientMacAddress.GetMacAddressData(),
                    MacAddress::MacAddressSize);
            pOutStatusList->statusList[i].statusReasonCode = statusInfo.status[i].statusReasonCode;
            pOutStatusList->statusList[i].capabilityInfo = statusInfo.status[i].capabilityInfo;
            pOutStatusList->statusList[i].rssi = statusInfo.status[i].rssi;
            pOutStatusList->statusList[i].updateTick = statusInfo.status[i].updateTick.GetInt64Value();
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result LocalApiImpl::GetBssIndicationEvent(nn::sf::Out<nn::sf::NativeHandle> pOutSystemEventReadableHandle) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        NN_UNUSED(pOutSystemEventReadableHandle);
        NN_ABORT("Sorry. This function is not implemented yet.");
        NN_RESULT_SUCCESS;
    }

    nn::Result LocalApiImpl::GetBssIndicationInfo(nn::sf::Out<nn::wlan::detail::SfdlBssInfo> pOutInfo) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        NN_UNUSED(pOutInfo);
        NN_ABORT("Sorry. This function is not implemented yet.");
        NN_RESULT_SUCCESS;
    }

    nn::Result LocalApiImpl::GetAllowedChannels(const nn::sf::OutArray<std::int16_t>& outArray, nn::sf::Out<std::uint32_t> pOutCount) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        uint32_t count;
        m_pStateMachine->GetAllowedChannels(outArray.GetData(), outArray.GetLength(), &count);
        pOutCount.Set(count);

        NN_RESULT_SUCCESS;
    }

    nn::Result LocalApiImpl::AddIe(nn::sf::Out<uint32_t> pOutIeIndex, uint32_t managementFrameType, const nn::sf::InBuffer& pIeBody) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( (managementFrameType & (nn::wlan::ManagementFrameType_Beacon |
                nn::wlan::ManagementFrameType_ProbeRes |
                nn::wlan::ManagementFrameType_AssocRes |
                nn::wlan::ManagementFrameType_AuthRes |
                nn::wlan::ManagementFrameType_ProbeReq |
                nn::wlan::ManagementFrameType_AssocReq)) == 0 )
        {
            NN_SDK_REQUIRES(false, "ManagementFrameType(0x%2X) is out of range\n", managementFrameType);
            return nn::wlan::ResultInvalidArgument();
        }
        if( pIeBody.GetSize() < 1 || pIeBody.GetSize() > 255  )
        {
            NN_SDK_REQUIRES(false, "ieBodySize(%d) is invalid\n", pIeBody.GetSize());
            return nn::wlan::ResultInvalidArgument();
        }
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanIeContainer));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        WlanIeContainer* pContainer = static_cast<WlanIeContainer*>(pcmdbuff->Args);

        pcmdbuff->id = WLAN_ADD_IE;
        pContainer->flag = managementFrameType;
        pContainer->length = pIeBody.GetSize();

        std::memcpy(&pContainer->oui[0], pIeBody.GetPointerUnsafe(), 3);
        std::memcpy(&pContainer->body[0], pIeBody.GetPointerUnsafe() + 3, pIeBody.GetSize() - 3);

        m_pStateMachine->PostCommandMessage(pcmdbuff);

        result = pcmdbuff->nnResult;
        if( result.IsSuccess() )
        {
            pOutIeIndex.Set(pContainer->index);
        }

        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result LocalApiImpl::DeleteIe(uint32_t ieIndex) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(uint32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

        pcmdbuff->id = WLAN_DELETE_IE;
        *(reinterpret_cast<uint32_t*>(pcmdbuff->Args)) = ieIndex;

        m_pStateMachine->PostCommandMessage(pcmdbuff);

        result = pcmdbuff->nnResult;

        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result LocalApiImpl::PutFrameRaw(const nn::sf::InBuffer& pTxData) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[Local::%s]\n", __FUNCTION__);
        // パラメータチェック
        if( pTxData.GetSize() >= 2048 )
        {
            NN_SDK_REQUIRES(false, "Tx size(%d) must be less than 2048\n", pTxData.GetSize());
            return ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsMasterBss
        };
        nn::Result result = CheckState(states, 4);
        if( result.IsFailure() )
        {
            return result;
        }

        // pTxDataをmbufに詰め込む
        nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(NULL, pTxData.GetSize(), nn::mbuf::MbufAllocationMode_Wait,
                nn::wlan::driver::MbufType_TxData);
        NN_SDK_ASSERT_NOT_NULL(pMbuf);
        void* pData = nn::mbuf::MbufTod(pMbuf);
        std::memcpy(pData, pTxData.GetPointerUnsafe(), pTxData.GetSize());
        nn::mbuf::MbufExpand(pMbuf, pTxData.GetSize());

        result = m_pStateMachine->PutFrame(&pMbuf);
        if( ResultTxQueueIsFull().Includes(result) == true )
        {
            // 400us待ってリトライして無理なら諦める
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(400));
            result = m_pStateMachine->PutFrame(&pMbuf);
            if( ResultTxQueueIsFull().Includes(result) == true )
            {
                if( pMbuf != NULL )
                {
                    nn::mbuf::MbufFreem(pMbuf);
                }
            }
        }
        // ResultTxQueueIsFull以外の場合は、mbufの解放は内部で適切に行われるので、ここで行う必要はない

        return result;
    }

    nn::Result LocalApiImpl::CancelGetFrame(std::uint32_t rxId) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        return m_pStateMachine->CancelRx(rxId);
    }

    nn::Result LocalApiImpl::CreateRxEntry(nn::sf::Out<std::uint32_t> pOutRxId, const nn::sf::InArray<std::uint16_t>& pEthertypeArray, std::uint32_t capacity) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( pEthertypeArray.GetLength() > EthertypeCountMax )
        {
            NN_SDK_REQUIRES(false, "The number of ethertypes must be less than %d\n", EthertypeCountMax + 1);
            return nn::wlan::ResultInvalidArgument();
        }
        if( capacity > DataFrameRxCapacityMax )
        {
            NN_SDK_REQUIRES(false, "Capacity of RxEntry must be less than %d\n", DataFrameRxCapacityMax + 1);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        uint32_t rxId; // 出力用変数
        result = m_pStateMachine->AddRxEntry(&rxId, capacity, pEthertypeArray.GetData(), pEthertypeArray.GetLength(), false);  // RingBufferの上書き禁止にしておく。つまり満杯の場合、新しいものは捨てられる。
        if( result.IsSuccess() )
        {
            pOutRxId.Set(rxId);
        }

        return result;
    }

    nn::Result LocalApiImpl::DeleteRxEntry(std::uint32_t rxId) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        result = m_pStateMachine->RemoveRxEntry(rxId);

        return result;
    }

    nn::Result LocalApiImpl::AddEthertypeToRxEntry(std::uint32_t rxId, std::uint16_t ethertype) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        result = m_pStateMachine->AddEthertypeToRxEntry(rxId, ethertype);

        return result;
    }

    nn::Result LocalApiImpl::DeleteEthertypeFromRxEntry(nn::sf::Out<std::uint32_t> pOutRxId, std::uint16_t ethertype) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        uint32_t rxId; // 出力用変数
        nn::Result result = m_pStateMachine->RemoveEthertypeFromRxEntry(&rxId, ethertype);
        pOutRxId.Set(rxId);

        return result;
    }

    nn::Result LocalApiImpl::AddMatchingDataToRxEntry(std::uint32_t rxId, const nn::wlan::ReceivedDataMatchInfo& pMatchInfo) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( pMatchInfo.matchLength > ReceivedDataMatchLengthMax )
        {
            NN_SDK_REQUIRES(false, "ReceiveDataMatch length must be less than %d\n", ReceivedDataMatchLengthMax + 1);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanMatchingDataInfo));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

        WlanMatchingDataInfo* pWlanMatchInfo = static_cast<WlanMatchingDataInfo*>(pcmdbuff->Args);
        pWlanMatchInfo->rxId = rxId;
        pWlanMatchInfo->pInfo = &(const_cast<ReceivedDataMatchInfo&>(pMatchInfo)); // ポインタ型に変換して渡す

        pcmdbuff->id = WLAN_ADD_MATCHING_DATA;

        m_pStateMachine->PostCommandMessage(pcmdbuff);

        result = pcmdbuff->nnResult;

        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );


        return result;
    }

    nn::Result LocalApiImpl::RemoveMatchingDataFromRxEntry(std::uint32_t rxId, const nn::wlan::ReceivedDataMatchInfo& pMatchInfo) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( pMatchInfo.matchLength > ReceivedDataMatchLengthMax )
        {
            NN_SDK_REQUIRES(false, "ReceiveDataMatch length must be less than %d\n", ReceivedDataMatchLengthMax + 1);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanMatchingDataInfo));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

        WlanMatchingDataInfo* pWlanMatchInfo = static_cast<WlanMatchingDataInfo*>(pcmdbuff->Args);
        pWlanMatchInfo->rxId = rxId;
        pWlanMatchInfo->pInfo = &(const_cast<ReceivedDataMatchInfo&>(pMatchInfo)); // ポインタ型に変換して渡す

        pcmdbuff->id = WLAN_REMOVE_MATCHING_DATA;

        m_pStateMachine->PostCommandMessage(pcmdbuff);

        result = pcmdbuff->nnResult;

        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result LocalApiImpl::PutActionFrameOneShot(nn::wlan::detail::SfdlMacAddress macAddr,
            const nn::sf::InBuffer& pTxData, std::uint32_t channel, std::uint32_t dwellTime) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( std::memcmp(macAddr.data, MacAddress::CreateZeroMacAddress().GetMacAddressData(), MacAddress::MacAddressSize) == 0 )
        {
            NN_SDK_REQUIRES(false, "%s: macAddr must not be Zero mac address\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( pTxData.GetSize() < 5 || pTxData.GetSize() > OneShotActionFrameSizeMax )
        {
            NN_SDK_REQUIRES(false, "%s: pTxData.size is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( channel < WirelessChannel_1ch || channel > WirelessChannel_165ch )
        {
            NN_SDK_REQUIRES(false, "%s: channel is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( dwellTime > 300 )
        {
            NN_SDK_REQUIRES(false, "%s: dwellTime must be less than 301\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        // データの正当性評価
        if( pTxData.GetPointerUnsafe()[0] != 0x7F )
        {
            NN_SDK_REQUIRES(false, "%s: pData[0] must be 0x7F\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( pTxData.GetPointerUnsafe()[4] < ActionFrameType_Beacon || pTxData.GetPointerUnsafe()[4] >= ActionFrameType_End )
        {
            NN_SDK_REQUIRES(false, "%s: pData[4] is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss
        };
        nn::Result result = CheckState(states, 8);
        if( result.IsFailure() )
        {
            return result;
        }

        result = RequestIfUpDown(1);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        bool isRetried = false;
        while( 1 )
        {
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanOneShotActionFrameInfo));
            NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
            WlanOneShotActionFrameInfo* pInfo = static_cast<WlanOneShotActionFrameInfo*>(pcmdbuff->Args);
            pInfo->pData = const_cast<char*>(pTxData.GetPointerUnsafe());
            pInfo->size = pTxData.GetSize();
            std::memcpy(&pInfo->dstMac[0], &macAddr.data[0], MacAddress::MacAddressSize);
            std::memcpy(&pInfo->bssid[0], &MacAddress::CreateBroadcastMacAddress().GetMacAddressData()[0], MacAddress::MacAddressSize);
            pInfo->channel = channel;
            pInfo->dwellTime = dwellTime;
            pcmdbuff->id = WLAN_PUT_ONE_SHOT_ACTION_FRAME;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            result = pcmdbuff->nnResult;
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

            if( ResultNoMemory().Includes(result) == true && isRetried == false )
            {
                // 400us待って1度だけリトライ
                nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(400));
                isRetried = true;
            }
            else
            {
                break;
            }
        }

        return result;
    }

    nn::Result LocalApiImpl::SetActionFrameWithBeacon(const nn::sf::InBuffer& pTxData) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( pTxData.GetSize() < 5 || pTxData.GetSize() > BeaconActionFrameSizeMax )
        {
            NN_SDK_REQUIRES(false, "%s: pTxData.size is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        // データの正当性評価
        if( pTxData.GetPointerUnsafe()[0] != 0x7F )
        {
            NN_SDK_REQUIRES(false, "%s: pData[0] must be 0x7F\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( pTxData.GetPointerUnsafe()[4] != ActionFrameType_Beacon )
        {
            NN_SDK_REQUIRES(false, "%s: pData[4] must be ActionFrameType_Beacon\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 4);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanPeriodicActionFrameInfo));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

        WlanPeriodicActionFrameInfo* pInfo = static_cast<WlanPeriodicActionFrameInfo*>(pcmdbuff->Args);
        pInfo->pData = const_cast<char*>(pTxData.GetPointerUnsafe());
        pInfo->size = pTxData.GetSize();
        // BSSIDはBroadcastアドレスにしておく。不特定の端末にも受信可能なようにするため。  TODO BSSIDは上位層からも指定可能なようにする
        std::memcpy(&pInfo->bssid[0],
                MacAddress::CreateBroadcastMacAddress().GetMacAddressData(), MacAddress::MacAddressSize);

        pcmdbuff->id = WLAN_SET_BEACON_ACTION_FRAME;

        m_pStateMachine->PostCommandMessage(pcmdbuff);

        result = pcmdbuff->nnResult;

        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result LocalApiImpl::CancelActionFrameWithBeacon() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 4);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

        pcmdbuff->id = WLAN_CANCEL_BEACON_ACTION_FRAME;

        m_pStateMachine->PostCommandMessage(pcmdbuff);

        result = pcmdbuff->nnResult;

        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result LocalApiImpl::CreateRxEntryForActionFrame(nn::sf::Out<std::uint32_t> pOutRxId,
            const nn::sf::InArray<std::uint16_t>& pSubtypeArray, std::uint32_t capacity) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( pSubtypeArray.GetLength() > ActionFrameTypeCountMax )
        {
            NN_SDK_REQUIRES(false, "The number of ActionFrameType must be less than %d\n", ActionFrameTypeCountMax + 1);
            return nn::wlan::ResultInvalidArgument();
        }
        if( capacity > ActionFrameRxCapacityMax )
        {
            NN_SDK_REQUIRES(false, "Capacity of RxEntry must be less than %d\n", ActionFrameRxCapacityMax + 1);
            return nn::wlan::ResultInvalidArgument();
        }
        for( int i = 0; i < pSubtypeArray.GetLength(); i++ )
        {
            if( pSubtypeArray.GetData()[i] < ActionFrameType_Beacon || pSubtypeArray.GetData()[i] >= ActionFrameType_End )
            {
                return nn::wlan::ResultInvalidArgument();
            }
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        uint32_t rxId; // 出力用変数
        result = m_pStateMachine->AddRxEntryForActionFrame(&rxId, capacity, pSubtypeArray.GetData(), pSubtypeArray.GetLength(), true);  // RingBufferの上書き可能にしておく。つまり満杯の場合、古いものから上書きされる。
        if( result.IsSuccess() )
        {
            pOutRxId.Set(rxId);
        }

        return result;
    }

    nn::Result LocalApiImpl::DeleteRxEntryForActionFrame(std::uint32_t rxId) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        result = m_pStateMachine->RemoveRxEntryForActionFrame(rxId);

        return result;
    }

    nn::Result LocalApiImpl::AddSubtypeToRxEntryForActionFrame(std::uint32_t rxId, std::uint32_t type) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        if( type < ActionFrameType_Beacon || type >= ActionFrameType_End )
        {
            NN_SDK_REQUIRES(false, "%s: Action frame type is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        result = m_pStateMachine->AddSubtypeToRxEntryForActionFrame(rxId, static_cast<uint16_t>(type));

        return result;
    }

    nn::Result LocalApiImpl::DeleteSubtypeFromRxEntryForActionFrame(nn::sf::Out<std::uint32_t> pOutRxId, std::uint32_t type) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        if( type < ActionFrameType_Beacon || type >= ActionFrameType_End )
        {
            NN_SDK_REQUIRES(false, "%s: Action frame type is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        uint32_t rxId; // 出力用変数
        result = m_pStateMachine->RemoveSubtypeFromRxEntryForActionFrame(&rxId, static_cast<uint16_t>(type));
        if( result.IsSuccess() )
        {
            pOutRxId.Set(rxId);
        }

        return result;
    }

    nn::Result LocalApiImpl::CancelGetActionFrame(std::uint32_t rxId) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        result = m_pStateMachine->CancelRxActionFrame(rxId);
        return result;
    }

    nn::Result LocalApiImpl::GetRssi(nn::sf::Out<std::int32_t> pOutRssi) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalClient,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalLcsClient,
        };
        nn::Result result = CheckState(states, 3);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(int32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_RSSI;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        int32_t* pRssi = reinterpret_cast<int32_t*>(pcmdbuff->Args);
        NN_SDK_ASSERT_NOT_NULL(pRssi);
        pOutRssi.Set(*pRssi);
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result LocalApiImpl::SetMaxAssociationNumber(std::uint32_t maxNum) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        if( maxNum < 1 || maxNum > ConnectableClientsCountMax )
        {
            NN_SDK_REQUIRES(false, "Max association number must be in 1 ~ %d", ConnectableClientsCountMax);
            return ResultInvalidArgument();
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(int));
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        int* pNum = reinterpret_cast<int*>(pcmdbuff->Args);
        *pNum = static_cast<int>(maxNum);
        pcmdbuff->id = WLAN_SET_MAX_ASSOC_NUM;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result LocalApiImpl::OpenLcsMasterMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }
        return ChangeMode(WLAN_CHANGE_STATE_LOCAL_LCS_AP);
    }

    nn::Result LocalApiImpl::CloseLcsMasterMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalLcsMasterIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }
        return ChangeMode(WLAN_CHANGE_STATE_READY_FROM_LCS_AP);
    }

    nn::Result LocalApiImpl::OpenLcsClientMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalLcsClientIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }
        return ChangeMode(WLAN_CHANGE_STATE_LOCAL_LCS_STA);
    }

    nn::Result LocalApiImpl::CloseLcsClientMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_LocalLcsClientIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }
        return ChangeMode(WLAN_CHANGE_STATE_READY_FROM_LCS_STA);
    }

    /*
     * LocalGetFrameApiImpl
     */

    nn::Result LocalGetFrameApiImpl::GetFrameRaw(const nn::sf::OutBuffer& pOutRxData, nn::sf::Out<std::uint32_t> pOutSize, std::uint32_t rxId) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[Local::%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalClient,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsMasterBss
        };
        nn::Result result = CheckState(states, 5);
        if( result.IsFailure() )
        {
            return result;
        }

        nn::mbuf::Mbuf* pOutMbuf;
        result = m_pStateMachine->PullRxBuffer(&pOutMbuf, rxId);
        size_t dataSize = 0;

        if( result.IsSuccess() )
        {
            NN_SDK_ASSERT_NOT_NULL(pOutMbuf);
            // データコピー
            // 出力先バッファの容量が足りているかチェック
            dataSize = nn::mbuf::MbufLength(pOutMbuf, NULL);
            if( pOutRxData.GetSize() >=  dataSize )
            {
                std::memcpy(pOutRxData.GetPointerUnsafe(), nn::mbuf::MbufTod(pOutMbuf), dataSize);
            }
            else
            {
                WLAN_LOG_ERROR("Failed to copy rx data due to capacity shortage of out buffer\n");
                result = ResultBufferTooShort();
            }

            // コピー版なので、mbufはここで破棄
            nn::mbuf::MbufFreem(pOutMbuf);
        }

        pOutSize.Set(dataSize);

        return result;
    }

    nn::Result LocalGetActionFrameApiImpl::GetActionFrameCore(nn::sf::Out<nn::wlan::detail::SfdlMacAddress> pOutMacAddr,
            const nn::sf::OutBuffer& pOutRxData, nn::sf::Out<std::uint32_t> pOutSize, std::uint32_t rxId,
            nn::sf::Out<std::uint16_t> pOutChannel, nn::sf::Out<std::int16_t> pOutRssi, bool isEx) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_LocalMasterBss,
                WdrvMainState_LocalMasterIdle,
                WdrvMainState_LocalClient,
                WdrvMainState_LocalClientIdle,
                WdrvMainState_LocalSpectator,
                WdrvMainState_LocalSpectatorIdle,
                WdrvMainState_LocalLcsClient,
                WdrvMainState_LocalLcsClientIdle,
                WdrvMainState_LocalLcsMasterBss,
                WdrvMainState_LocalLcsMasterIdle,
        };
        nn::Result result = CheckState(states, 10);
        if( result.IsFailure() )
        {
            return result;
        }

        nn::mbuf::Mbuf* pOutMbuf;
        result = m_pStateMachine->PullRxActionFrame(&pOutMbuf, rxId);
        size_t dataSize = 0;

        if( result.IsSuccess() )
        {
            NN_SDK_ASSERT_NOT_NULL(pOutMbuf);

            // データコピー
            // mbufから上位層に渡すべきデータの抽出
            uint8_t* pData = reinterpret_cast<uint8_t*>(nn::mbuf::MbufTod(pOutMbuf));

            // 送信元MACアドレスの抽出
            driver::GetPeerMacAddress(pOutMacAddr->data, nn::mbuf::MbufTod(pOutMbuf));

            // 出力先バッファの容量が足りているかチェック
            dataSize = driver::GetActionFrameDataSizeEx(nn::mbuf::MbufTod(pOutMbuf));
            WLAN_LOG_DEBUG("[%s] action frame size : %d\n", __FUNCTION__, dataSize);

            // 本体コピー
            if( pOutRxData.GetSize() >=  dataSize )
            {
                std::memcpy(pOutRxData.GetPointerUnsafe(), &pData[driver::offsetToDataEx], dataSize);
                if( isEx == true )
                {
                    WlanActionFrameRecvInfo* pInfo = reinterpret_cast<WlanActionFrameRecvInfo*>(pOutMbuf->_pGeneral);
                    if( pInfo != NULL )
                    {
                        pOutChannel.Set(pInfo->channel);
                        pOutRssi.Set(static_cast<int16_t>(pInfo->rssi));
                    }
                    else
                    {
                        pOutChannel.Set(0);
                        pOutRssi.Set(-999);
                    }
                }
            }
            else
            {
                WLAN_LOG_DEBUG("Failed to copy rx data due to capacity shortage of out buffer\n");
                result = ResultBufferTooShort();
            }

            // コピー版なので、mbufはここで破棄
            if( pOutMbuf->_pGeneral != NULL )
            {
                nnwlanFreeNormal(pOutMbuf->_pGeneral);
            }
            nn::mbuf::MbufFreem(pOutMbuf);
        }

        pOutSize.Set(dataSize);
        return result;
    }

    nn::Result LocalApiImpl::GetChannelStats(const nn::sf::OutArray<nn::wlan::ChannelStats>& outStats, nn::sf::Out<std::uint32_t> pOutCount) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanTotalChannelStats));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_CHANNEL_STATS;
        WlanTotalChannelStats* pTotalStats = reinterpret_cast<WlanTotalChannelStats*>(pcmdbuff->Args);
        NN_ABORT_UNLESS_NOT_NULL(pTotalStats);
        pTotalStats->stats = outStats.GetData();
        pTotalStats->length = outStats.GetLength();
        pTotalStats->count = pOutCount.GetPointer();
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    } //NOLINT(impl/function_size)

}}} // end of namespace nn::wlan::server

