﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cctype>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_MacAddress.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_StateDefinition.h>
#include <nn/wlan/wlan_Result.h>

#include "wlan_InfraApiImpl.h"
#include "../wlan_MsgBuffer.h"
#include "../wlan_IeParser.h"
#include "../wlan_SignalStrength.h"

namespace nn { namespace wlan { namespace server
{
nn::Result ApiImplBase::ChangeMode(WlanCommandId id) NN_NOEXCEPT
{
    nn::Result result;
    WlanCommand* pcmdbuff;
    pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
    pcmdbuff->id = id;
    m_pStateMachine->PostCommandMessage(pcmdbuff);
    result = pcmdbuff->nnResult;
    m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
    return result;
}

nn::Result ApiImplBase::RequestIfUpDown(uint32_t value) NN_NOEXCEPT
{
    WlanCommand* pcmdbuff;
    pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(uint32_t));
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
    pcmdbuff->id = WLAN_SET_IFUPDOWN;
    *(reinterpret_cast<uint32_t*>(pcmdbuff->Args)) = value;
    m_pStateMachine->PostCommandMessage(pcmdbuff);
    nn::Result result = pcmdbuff->nnResult;
    m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
    return result;
}

nn::Result ApiImplBase::CheckState(WdrvMainState* allowedStates, uint8_t length) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
    NN_SDK_ASSERT_NOT_NULL(allowedStates);

    WlanCommand* pcmdbuff;
    pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
    pcmdbuff->id = WLAN_GET_STATE;
    m_pStateMachine->PostCommandMessage(pcmdbuff);
    WlanStates* state = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
    WdrvMainState main = state->mainState;
    WdrvSubState sub = state->subState;
    m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);

    if( main == WdrvMainState_OutOfService && sub == WdrvSubState_None )
    {
        // 無線機能無効状態
        return ResultWlanDeviceAbnormal();
    }

    for( uint8_t i = 0; i < length; i++ )
    {
        if( main == allowedStates[i] )
        {
            return ResultSuccess();
        }
    }
    return ResultInvalidState();
}

void ApiImplBase::GetWlanState(WlanStates* pOutStates) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutStates);
    WlanCommand* pcmdbuff;
    pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
    pcmdbuff->id = WLAN_GET_STATE;
    m_pStateMachine->PostCommandMessage(pcmdbuff);

    WlanStates* pStates = reinterpret_cast<WlanStates*>(pcmdbuff->Args);

    pOutStates->mainState = pStates->mainState;
    pOutStates->subState  = pStates->subState;

    m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);
}

nn::Result ApiImplBase::GetMacAddress(nn::sf::Out<nn::wlan::detail::SfdlMacAddress> pOutMacAddr) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

    WlanCommand* pcmdbuff;
    pcmdbuff = m_pStateMachine->WlanGetCommandBuff(nn::wlan::MacAddress::MacAddressSize);
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    pcmdbuff->id = WLAN_GET_MACADDRESS;
    m_pStateMachine->PostCommandMessage(pcmdbuff);

    nn::wlan::detail::SfdlMacAddress ret;
    std::memcpy(&ret.data[0], pcmdbuff->Args, nn::wlan::MacAddress::MacAddressSize);
    pOutMacAddr.Set(ret);

    m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

    NN_RESULT_SUCCESS;
}

nn::Result ApiImplBase::GetState(nn::sf::Out<uint32_t> pOutState) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

    WlanCommand* pcmdbuff;
    pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
    pcmdbuff->id = WLAN_GET_STATE;
    m_pStateMachine->PostCommandMessage(pcmdbuff);

    WlanStates* state = reinterpret_cast<WlanStates*>(pcmdbuff->Args);

    WdrvMainState main = state->mainState;
    WdrvSubState  sub  = state->subState;

    // メインとサブを総合したステートを返す
    switch( main )
    {
    case WdrvMainState_Stop:  // Stop
        pOutState.Set(static_cast<uint32_t>(WlanState_Stop));
        break;
    case WdrvMainState_Ready: // Ready
        pOutState.Set(static_cast<uint32_t>(WlanState_Ready));
        break;
    case WdrvMainState_InfraIdle:  // Infra Idle
        switch( sub )
        {
        case WdrvSubState_None:
        case WdrvSubState_Idle:
        case WdrvSubState_Connecting:
            pOutState.Set(static_cast<uint32_t>(WlanState_InfraIdle));  // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_InfraIdleScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_InfraSta:  // Infra connected
        switch( sub )
        {
        case WdrvSubState_Disconnecting:
        case WdrvSubState_Connected:
            pOutState.Set(static_cast<uint32_t>(WlanState_InfraSta));  // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_InfraStaScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalMasterIdle:  // Master idle
        switch( sub )
        {
        case WdrvSubState_None:
        case WdrvSubState_Idle:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalMasterIdle));  // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalMasterIdleScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalMasterBss:  // Master Bss
        switch( sub )
        {
        case WdrvSubState_Connected:
        case WdrvSubState_Disconnecting:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalMasterBss));   // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalMasterBssScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalClientIdle:  // Local Client Idle
        switch( sub )
        {
        case WdrvSubState_None:
        case WdrvSubState_Idle:
        case WdrvSubState_Connecting:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalClientIdle));  // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalClientIdleScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalClient:  // Local Client connected
        switch( sub )
        {
        case WdrvSubState_Connected:
        case WdrvSubState_Disconnecting:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalClient));  // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalClientScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalSpectatorIdle:  // Local Spectator Idle
        switch( sub )
        {
        case WdrvSubState_None:
        case WdrvSubState_Idle:
        case WdrvSubState_Connecting:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalSpectatorIdle));  // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalSpectatorIdleScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalSpectator:  // Local Spectator joined
        switch( sub )
        {
        case WdrvSubState_Connected:
        case WdrvSubState_Disconnecting:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalSpectator));  // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalSpectatorScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalLcsMasterIdle:  // LCS Master idle
        switch( sub )
        {
        case WdrvSubState_None:
        case WdrvSubState_Idle:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalLcsMasterIdle));  // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalLcsMasterIdleScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalLcsMasterBss:  // LCS Master Bss
        switch( sub )
        {
        case WdrvSubState_Connected:
        case WdrvSubState_Disconnecting:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalLcsMasterBss));   // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalLcsMasterBssScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalLcsClientIdle:  // Local LCS Client Idle
        switch( sub )
        {
        case WdrvSubState_None:
        case WdrvSubState_Idle:
        case WdrvSubState_Connecting:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalLcsClientIdle));  // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalLcsClientIdleScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalLcsClient:  // Local LCS Client connected
        switch( sub )
        {
        case WdrvSubState_Connected:
        case WdrvSubState_Disconnecting:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalLcsClient));  // 非scan状態
            break;
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            pOutState.Set(static_cast<uint32_t>(WlanState_LocalLcsClientScan));  // scan状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_DetectIdle:
        switch( sub )
        {
        case WdrvSubState_None:
            pOutState.Set(static_cast<uint32_t>(WlanState_DetectIdle));  // I/F down状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_Detect:
        switch( sub )
        {
        case WdrvSubState_Idle:
            pOutState.Set(static_cast<uint32_t>(WlanState_Detect));  // I/F up状態
            break;
        default:
            NN_ABORT("[WLAN][ERR]Unexpected sub state in %s\n", __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_Sleep: // Sleep state
        pOutState.Set(static_cast<uint32_t>(WlanState_Sleep));
        break;
    case WdrvMainState_Exit:  // Wlan exit
        pOutState.Set(static_cast<uint32_t>(WlanState_Exit));
        break;
    case WdrvMainState_OutOfService:
        pOutState.Set(static_cast<uint32_t>(WlanState_Stop));
        break;
    default:
        NN_ABORT("[WLAN][ERR]Unknown state in %s\n", __FUNCTION__);
    }

    m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

    NN_RESULT_SUCCESS;
} //NOLINT(impl/function_size)

}}} // end of namespace nn::wlan::server

