﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/usb/pd/driver/usb_Pd.h>
#include <nn/usb/pd/server/usb_PdHipcServer.h>

#include "usb_PdServer.h"

namespace nn { namespace usb {

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

namespace {

NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_ServerStack[16384];
nn::os::ThreadType g_ServerThread;
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_CradleServerStack[16384];
nn::os::ThreadType g_CradleServerThread;

void PdServerFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    // USB::PD 用サーバーのループ処理
    NN_SDK_LOG("[usb] Start USB::PD Server\n");
    nn::usb::pd::server::LoopPdServer();
}

void PdCradleServerFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    // USB::PD:C 用サーバーのループ処理
    NN_SDK_LOG("[usb] Start USB::PD:C Server\n");
    nn::usb::pd::server::LoopPdCradleServer();
}

} // namespace

void InitializePdServer() NN_NOEXCEPT
{
    nn::Result result;

    // USB::PD 用サーバの初期化
    // 準備ができていないサービスに対して接続中に、サービスの登録を行おうとするとデッドロックする場合があるので
    // ポート登録は何よりも先に行うのが安全
    nn::usb::pd::server::InitializePdServer();

    // Type-C USB Power Delivery ドライバの初期化
    nn::usb::pd::driver::Initialize();

    // USB::PD 用サーバ処理スレッドの作成
    result = nn::os::CreateThread(&g_ServerThread, &PdServerFunction, nullptr,
                                  g_ServerStack, sizeof(g_ServerStack), NN_SYSTEM_THREAD_PRIORITY(usb, PdServer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::SetThreadNamePointer(&g_ServerThread, NN_SYSTEM_THREAD_NAME(usb, PdServer)) ;
    result = nn::os::CreateThread(&g_CradleServerThread, &PdCradleServerFunction, nullptr,
                                  g_CradleServerStack, sizeof(g_CradleServerStack), NN_SYSTEM_THREAD_PRIORITY(usb, PdCradleServer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::SetThreadNamePointer(&g_CradleServerThread, NN_SYSTEM_THREAD_NAME(usb, PdCradleServer)) ;
    nn::os::StartThread(&g_ServerThread);
    nn::os::StartThread(&g_CradleServerThread);
}

void FinalizePdServer() NN_NOEXCEPT
{
    // USB::PD サーバー停止要求
    nn::usb::pd::server::RequestStopPdServer();

    // USB::PD 用サーバ処理スレッド終了待機と破棄
    nn::os::WaitThread(&g_CradleServerThread);
    nn::os::DestroyThread(&g_CradleServerThread);
    nn::os::WaitThread(&g_ServerThread);
    nn::os::DestroyThread(&g_ServerThread);

    // USB::PD サーバの終了処理
    nn::usb::pd::server::FinalizePdServer();

    // Type-C USB Power Delivery ドライバの終了処理
    nn::usb::pd::driver::Finalize();
}

#endif // NN_BUILD_CONFIG_HARDWARE_NX

}} // namespace nn::usb
