﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "shell_ProcessManagement.h"
#include <nn/util/util_IntrusiveList.h>
#include <nn/os.h>
#include <nn/osdbg.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <mutex>
#include <alloca.h>
#include <algorithm>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Synchronization.h>

#include <nn/ldr/ldr_ShellApi.h>
#include <nn/ldr/ldr_Result.h>

#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/ns/ns_DevelopApi.h>

namespace nn { namespace shell {

    namespace
    {
        class ProcessEntry : public nn::util::IntrusiveListBaseNode<ProcessEntry>
        {
        private:
            Bit64       m_ProcessId;
            svc::Handle m_Handle;
            bool        m_IsStopped;

        public:
            void Set(Bit64 processId, svc::Handle handle) NN_NOEXCEPT
            {
                m_ProcessId = processId;
                m_Handle = handle;
                m_IsStopped = false;
            }
            svc::Handle GetHandle() const NN_NOEXCEPT
            {
                return m_Handle;
            }
            Bit64 GetProcessId() const NN_NOEXCEPT
            {
                return m_ProcessId;
            }
            void SetStopped() NN_NOEXCEPT
            {
                m_IsStopped = true;
            }
            bool IsStopped() const NN_NOEXCEPT
            {
                return m_IsStopped;
            }
        };
        typedef nn::util::IntrusiveList<ProcessEntry, nn::util::IntrusiveListBaseNodeTraits<ProcessEntry>> PrcocessEntries;

        ProcessEntry    g_PrcocessEntries[0x100];
        PrcocessEntries g_FreeEntries;
        PrcocessEntries g_UsedEntries;


        class Mutex
        {
        public:
            void Initialize() NN_NOEXCEPT
            {
                nn::os::InitializeMutex(&m_Mutex, false, 0);
            }
            void Finalize() NN_NOEXCEPT
            {
                nn::os::FinalizeMutex(&m_Mutex);
            }
            void lock() NN_NOEXCEPT
            {
                nn::os::LockMutex(&m_Mutex);
            }
            void unlock() NN_NOEXCEPT
            {
                nn::os::UnlockMutex(&m_Mutex);
            }
        private:
            nn::os::MutexType m_Mutex;
        };

        Mutex g_Lock;

        nn::os::ThreadType      g_Thread;
        const size_t            ThreadStackSize = 16384; // 0x4000
        NN_ALIGNAS(4096) char   g_ThreadStack[ ThreadStackSize ];

        void ThreadFunction(void* arg) NN_NOEXCEPT
        {
            nn::svc::Handle* pHandles = static_cast<nn::svc::Handle *>(alloca(
                        sizeof(nn::svc::Handle) * sizeof(g_PrcocessEntries) / sizeof(g_PrcocessEntries[0])));
            for (;;)
            {
                int i = 0;
                {
                    std::lock_guard<Mutex> lock(g_Lock);

                    for (PrcocessEntries::iterator it = g_UsedEntries.begin(); it != g_UsedEntries.end(); it++)
                    {
                        pHandles[i++] = it->GetHandle();
                    }
                }
                int32_t index;
                Result result = nn::svc::WaitSynchronization(&index, pHandles, i, nn::svc::WAIT_INFINITE);
                if (result.IsSuccess())
                {
                    std::lock_guard<Mutex> lock(g_Lock);
                    nn::svc::Handle handle = pHandles[index];
                    PrcocessEntries::iterator it = std::find_if(
                            g_UsedEntries.begin(),
                            g_UsedEntries.end(),
                            [&handle](const ProcessEntry& x)
                            {
                                return x.GetHandle() == handle;
                            }
                            );
                    if (it != g_UsedEntries.end())
                    {
                        int64_t processState;
                        result = nn::svc::GetProcessInfo(&processState, it->GetHandle(), nn::svc::ProcessInfoType_State);
                        NN_ABORT_UNLESS(result.IsSuccess());
                        if (processState != nn::svc::ProcessState_WaitAttach)
                        {
                            result = nn::svc::ResetSignal(it->GetHandle());
                            NN_ABORT_UNLESS(result.IsSuccess());
                            it->SetStopped();
                        }
                        else
                        {
                            nn::svc::CloseHandle(it->GetHandle());
                            g_UsedEntries.erase(it);
                            g_FreeEntries.push_back(*it);
                        }
                    }
                }
            }
        }


        lr::LocationResolver    g_LocationResolver;
    }

    void InitializeProcessManager() NN_NOEXCEPT
    {
        g_Lock.Initialize();
        NN_ABORT_UNLESS(nn::lr::OpenLocationResolver(&g_LocationResolver, nn::ncm::StorageId::Host).IsSuccess());
        for (size_t i = 0; i < sizeof(g_PrcocessEntries) / sizeof(g_PrcocessEntries[0]); i++)
        {
            g_FreeEntries.push_back(g_PrcocessEntries[i]);
        }
        NN_ABORT_UNLESS(nn::os::CreateThread(&g_Thread, ThreadFunction, NULL, g_ThreadStack, ThreadStackSize, nn::os::HighestThreadPriority - 1).IsSuccess());
        nn::os::StartThread(&g_Thread);
    }

    void OnCreated(Bit64 processId, nn::svc::Handle handle) NN_NOEXCEPT
    {

        std::lock_guard<Mutex> lock(g_Lock);
        if (!g_FreeEntries.empty())
        {
            ProcessEntry* pEntry = &g_FreeEntries.front();
            g_FreeEntries.pop_front();
            pEntry->Set(processId, handle);
            g_UsedEntries.push_back(*pEntry);

            nn::svc::CancelSynchronization(static_cast<nn::svc::Handle>(g_Thread._handle));
        }

    }
    bool TerminateProcess(Bit64 processId) NN_NOEXCEPT
    {
        os::ProcessId pid = { processId };
        return nn::ns::TerminateProcess(pid).IsSuccess();
    }
    void PrintProcessList(const CommandExecuteContext* pContext) NN_NOEXCEPT
    {
        std::lock_guard<Mutex> lock(g_Lock);

        static const int ArraySize = 64;
        Bit64 list[ArraySize];
        int32_t num;
        Result result = nn::svc::GetProcessList(&num, list, ArraySize);

        pContext->pConsole->TPrintf("Process List\n");
        if( result.IsSuccess() )
        {
            for( int i = 0; i < num; ++i )
            {
                pContext->pConsole->TPrintf("   PID= %lld\n", list[i]);
            }
        }
    }

    void InfoThreadList(const CommandExecuteContext* pContext) NN_NOEXCEPT
    {
        std::lock_guard<Mutex> lock(g_Lock);

        pContext->pConsole->TPrintf("Thread List\n");
        for (PrcocessEntries::iterator it = g_UsedEntries.begin(); it != g_UsedEntries.end(); it++)
        {
            pContext->pConsole->TPrintf("   TID= %lld %s\n", it->GetProcessId(), it->IsStopped()?"Stop":"Run");
        }
    }

    Result PrepareToLaunchProgram(ns::ProgramLaunchProperty* pOutProperty, const char* pPath, size_t length, const void* pArgument, size_t argumentSize) NN_NOEXCEPT
    {
        lr::Path path;
        std::memset(path.string, 0, sizeof(path.string));

        const char SystemMountNameChar = '@';
        const char HostMountName[] = "@Host:/";
        const char Fs0HostArchiveName[] = "host:";

        if (pPath[0] == SystemMountNameChar)
        {
            std::strcat(path.string, pPath);
            return ResultSuccess();
        }
        else
        {
            if (std::strncmp(pPath, Fs0HostArchiveName, std::strlen(Fs0HostArchiveName)) != 0)
            {
                std::strcat(path.string, HostMountName);
            }
            std::strcat(path.string, pPath);
        }

        ns::ProgramLaunchProperty launchProperty;
        Result result = nn::ns::PrepareLaunchProgramFromHost(&launchProperty, path.string);
        if ( result.IsFailure() )
        {
            return result;
        }

        result = nn::ldr::SetProgramArgument(launchProperty.programId, pArgument, argumentSize);
        if( result <= nn::ldr::ResultArgumentCountOverflow() )
        {
            result = nn::ldr::FlushArguments();
            if( result.IsSuccess() )
            {
                result = nn::ldr::SetProgramArgument(launchProperty.programId, pArgument, argumentSize);
            }
        }
        if( result.IsFailure() )
        {
            return result;
        }

        *pOutProperty = launchProperty;

        return ResultSuccess();
    }

    void FlushProgramArgument(ncm::ProgramId programId) NN_NOEXCEPT
    {
        // 直前の引数を残さないように argv[0] に既定値の空文字列を上書きしておく
        nn::ldr::SetProgramArgument(programId, "", 1);
    }

}}
