﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/htcs.h>
#include <nn/htcs/htcs_LibraryPrivate.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_Mutex.h>

#include "scs_Htcs.h"
#include "scs_LockableMutexType.h"

namespace nn { namespace scs {

namespace {

//!< htcs 用ヒープのハンドル
std::atomic<nn::lmem::HeapHandle> g_HeapHandle;

//!< htcs 用ヒープのメモリ領域
char g_HeapBuffer[2 * 1024];

//!< cs プロセスが使用するソケット数
const int SocketCountForCs = 4;
const int SocketCountForHid = 2;

//!< htcs 用ヒープ操作の排他用ミューテックス
LockableMutexType g_HeapMutex = { NN_OS_MUTEX_INITIALIZER(false) };

void* Allocate(size_t size) NN_NOEXCEPT
{
    std::lock_guard<decltype(g_HeapMutex)> locker(g_HeapMutex);

    return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
}

} // namespace

void InitializeHtcs(int additionalSocketCount) NN_NOEXCEPT
{
    g_HeapHandle = nn::lmem::CreateExpHeap(g_HeapBuffer,
                                           sizeof(g_HeapBuffer),
                                           nn::lmem::CreationOption_NoOption);

    NN_SDK_ASSERT_NOT_NULL(g_HeapHandle.load());

    size_t bufferSize = nn::htcs::GetWorkingMemorySize(SocketCountForCs + SocketCountForHid + additionalSocketCount);
    void* buffer = Allocate(bufferSize);
    NN_SDK_ASSERT_NOT_NULL(buffer);

    nn::htcs::InitializeForDisableDisconnectionEmulation(buffer, bufferSize, 4);
}

}} // namespace nn::scs
