﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <new>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/os.h>
#include <nn/psc.h>

#include <nn/tc/impl/tc.h>
#include <nn/tc/impl/tc_PowerMode.h>
#include <nn/tc/detail/tc_Log.h>
#include <nn/tc/tc.h>
#include <nn/tc/tc_IManager.sfdl.h>
#include <nn/tc/tc_ManagerImpl.h>
#include <nn/tc/tc_ServiceName.h>
#include <nn/tc/tc_ShimInternal.h>

namespace nn { namespace ptm {

namespace {

const auto MaxSessionForPort = 3; // am, application + 1(margin)
const auto SessionCountMax = MaxSessionForPort;
const auto PortCountMax = 1; // tc

// HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
class MyServerManager : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax>
{};
MyServerManager* g_pMyServerManager;

// 繰り返しのサーバの起動と終了が可能となるように placement new で初期化を行う。
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
nn::sf::UnmanagedServiceObject<nn::tc::IManager, nn::tc::ManagerImpl> g_Manager;

const size_t TcStackSize = 0x4000;
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_TcServerStack[TcStackSize];
nn::os::ThreadType g_TcServerThread;

// システム電源状態の遷移通知を受け取るオブジェクト。
nn::psc::PmModule g_PmModule;

void InitializeServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pMyServerManager);

    nn::tc::InitializeWith(g_Manager.GetShared());

    // MyServerManager オブジェクトのコンストラクト。
    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;

    // サービス名の登録とポートの初期化。
    g_pMyServerManager->RegisterObjectForPort(nn::tc::GetInternalManager(), MaxSessionForPort, nn::tc::TcServiceName);

    // PSC へのモジュール登録。
    const nn::psc::PmModuleId tcDependencies[] =
    {
        nn::psc::PmModuleId_Bpc,
        nn::psc::PmModuleId_Fan,
        nn::psc::PmModuleId_I2c,
        nn::psc::PmModuleId_PcvClock,
        nn::psc::PmModuleId_PcvVoltage,
    };
    auto result = g_PmModule.Initialize(nn::psc::PmModuleId_Tc,
        tcDependencies,
        sizeof(tcDependencies) / sizeof(tcDependencies[0]),
        nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // サーバマネージャの開始。
    g_pMyServerManager->Start();
}

void TcServerFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    NN_SDK_ASSERT(g_pMyServerManager);

    const uintptr_t UpdateThermalCoordinatorTag = 0x80000001;
    const uintptr_t PmEventTag = 0x80000002;

    nn::os::TimerEventType timerEvent;
    nn::os::MultiWaitHolderType timerEventHolder;

    nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&timerEventHolder, &timerEvent);
    nn::os::SetMultiWaitHolderUserData(&timerEventHolder, UpdateThermalCoordinatorTag);

    g_pMyServerManager->AddUserWaitHolder(&timerEventHolder);

    nn::os::MultiWaitHolderType pmEventHolder;

    nn::os::InitializeMultiWaitHolder(&pmEventHolder, g_PmModule.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(&pmEventHolder, PmEventTag); // PmModuleId とローカル定義のタグが衝突しない保証がないので PmModuleId は使わない。

    g_pMyServerManager->AddUserWaitHolder(&pmEventHolder);

    // 1000ms 毎にサーマルコーディネータの設定を更新する。
    const int64_t ControlSpanUsec = 1000000LL;

    nn::os::StartPeriodicTimerEvent(&timerEvent,
        nn::TimeSpan::FromMicroSeconds(ControlSpanUsec),
        nn::TimeSpan::FromMicroSeconds(ControlSpanUsec));

    // timerEvent が有効な状態であるか否かを表す。
    bool timerWorking = true;

    while ( auto p = g_pMyServerManager->Wait() )
    {
        uintptr_t userData =  nn::os::GetMultiWaitHolderUserData(p);
        switch ( userData )
        {
        case MyServerManager::InvokeTag:
        case MyServerManager::AcceptTag:
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_pMyServerManager->ProcessAuto(p));
            break;
        case UpdateThermalCoordinatorTag:
            if ( nn::os::TryWaitTimerEvent(&timerEvent) )
            {
                nn::os::ClearTimerEvent(&timerEvent);
                g_pMyServerManager->AddUserWaitHolder(p);

                if ( timerWorking )
                {
                    nn::tc::impl::Update();
                }
            }
            break;
        case PmEventTag:
            if ( g_PmModule.GetEventPointer()->TryWait() )
            {
                g_PmModule.GetEventPointer()->Clear();
                g_pMyServerManager->AddUserWaitHolder(p);

                nn::psc::PmState state;
                nn::psc::PmFlagSet flags;
                NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&state, &flags));

                switch ( state )
                {
                case nn::psc::PmState_FullAwake:
                    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_FullAwake);
                    break;
                case nn::psc::PmState_MinimumAwake:
                    nn::os::StartPeriodicTimerEvent(&timerEvent,
                        nn::TimeSpan::FromMicroSeconds(ControlSpanUsec),
                        nn::TimeSpan::FromMicroSeconds(ControlSpanUsec));
                    timerWorking = true;
                    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_MinimumAwake);
                    break;
                case nn::psc::PmState_SleepReady:
                case nn::psc::PmState_ShutdownReady:
                    nn::os::StopTimerEvent(&timerEvent);
                    timerWorking = false;
                    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_SleepReady);
                    break;
                default:
                    break;
                }
                g_PmModule.Acknowledge(state, ResultSuccess());
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    nn::os::StopTimerEvent(&timerEvent);
    nn::os::FinalizeMultiWaitHolder(&timerEventHolder);
    nn::os::FinalizeTimerEvent(&timerEvent);
}

void RequestStopServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    g_pMyServerManager->RequestStop();
}

void FinalizeServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.Finalize());

    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;
}

} // namespace

// Manager Object と impl library の初期化。
void InitializeTcServer() NN_NOEXCEPT
{
    // 準備ができていないサービスに対して接続中に、サービスの登録を行おうとするとデッドロックする場合があるので
    // ポート登録は何よりも先に行うのが安全
    InitializeServer();

    nn::tc::impl::Initialize();
}

// Thread の初期化と開始。
void StartTcServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_TcServerThread, &TcServerFunction, nullptr, g_TcServerStack, sizeof(g_TcServerStack), NN_SYSTEM_THREAD_PRIORITY(tc, IpcServer)));

    nn::os::SetThreadNamePointer(&g_TcServerThread, NN_SYSTEM_THREAD_NAME(tc, IpcServer));

    nn::os::StartThread(&g_TcServerThread);

    NN_DETAIL_TC_INFO("Start Thermal Coordinator.\n");
}

// Thread の終了と破棄。
void RequestStopTcServer() NN_NOEXCEPT
{
    RequestStopServer();

    nn::os::WaitThread(&g_TcServerThread);
    nn::os::DestroyThread(&g_TcServerThread);
}

// Manager Object と impl library の破棄。
void FinalizeTcServer() NN_NOEXCEPT
{
    FinalizeServer();

    nn::tc::impl::Finalize();
}

}}  // namespace nn::ptm
