﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <new>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/os.h>
#include <nn/psc.h>
#include <nn/pinmux/pinmux.h>

#include <nn/fan/detail/fan_IManager.h>
#include <nn/fan/detail/fan_Log.h>
#include <nn/fan/fan.h>
#include <nn/fan/fan_ControlDev.h>
#include <nn/fan/fan_ManagerImpl.h>
#include <nn/fan/fan_ServiceName.h>
#include <nn/fan/fan_ShimInternal.h>
#include <nn/fan/impl/fan.h>

namespace nn { namespace ptm {

namespace {

const auto MaxSessionForPort = 2; // application + 1(margin)
const auto SessionCountMax = MaxSessionForPort;
const auto PortCountMax = 1; // fan

nn::fan::Controller g_FanController;

// HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
class MyServerManager : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax>
{};
MyServerManager* g_pMyServerManager;

// 繰り返しのサーバの起動と終了が可能となるように placement new で初期化を行う。
// 繰り返しの起動と終了が必要ない場合には MyServerManager は直接配置してもよい。
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
nn::sf::UnmanagedServiceObject<nn::fan::detail::IManager, nn::fan::ManagerImpl> g_Manager;

const size_t FanStackSize = 0x4000;
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_FanServerStack[FanStackSize];
nn::os::ThreadType g_FanServerThread;

// システム電源状態の遷移通知を受け取るオブジェクト。
nn::psc::PmModule g_PmModule;

// スリープ時の Pinmux 操作用の session
nn::pinmux::PinmuxSession g_PinmuxSession;

void InitializeServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pMyServerManager);

    nn::fan::InitializeWith(g_Manager.GetShared());

    // MyServerManager オブジェクトのコンストラクト
    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;

    // サービス名の登録とポートの初期化
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    g_pMyServerManager->RegisterObjectForPort(nn::fan::GetInternalManager(), MaxSessionForPort, nn::fan::ServiceName);

    // PSC へのモジュール登録。
    const nn::psc::PmModuleId fanDependencies[] =
    {
        nn::psc::PmModuleId_Pinmux,
        nn::psc::PmModuleId_Pwm,
    };
    auto result = g_PmModule.Initialize(nn::psc::PmModuleId_Fan,
        fanDependencies,
        sizeof(fanDependencies) / sizeof(fanDependencies[0]),
        nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要がある。
    g_pMyServerManager->Start();
}

void FanHipcServerFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    NN_SDK_ASSERT(g_pMyServerManager);

    const uintptr_t UpdateFanRotationTag = 0x80000001;
    const uintptr_t PmEventTag = 0x80000002;

    nn::os::TimerEventType timerEvent;
    nn::os::MultiWaitHolderType timerEventHolder;

    nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&timerEventHolder, &timerEvent);
    nn::os::SetMultiWaitHolderUserData(&timerEventHolder, UpdateFanRotationTag);

    g_pMyServerManager->AddUserWaitHolder(&timerEventHolder);

    nn::os::MultiWaitHolderType pmEventHolder;

    nn::os::InitializeMultiWaitHolder(&pmEventHolder, g_PmModule.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(&pmEventHolder, PmEventTag); // PmModuleId とローカル定義のタグが衝突しない保証がないので PmModuleId は使わない。

    g_pMyServerManager->AddUserWaitHolder(&pmEventHolder);

    // 100ms 毎にファンの速度を更新する。
    const int64_t ControlSpanUsec = 100000LL;

    nn::os::StartPeriodicTimerEvent(&timerEvent,
        nn::TimeSpan::FromMicroSeconds(ControlSpanUsec),
        nn::TimeSpan::FromMicroSeconds(ControlSpanUsec));

    // timerEvent が有効な状態であるか否かを表す。
    bool timerWorking = true;

    while ( auto p = g_pMyServerManager->Wait() )
    {
        uintptr_t userData =  nn::os::GetMultiWaitHolderUserData(p);
        switch ( userData )
        {
        case MyServerManager::InvokeTag:
        case MyServerManager::AcceptTag:
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_pMyServerManager->ProcessAuto(p));
            break;
        case UpdateFanRotationTag:
            if ( nn::os::TryWaitTimerEvent(&timerEvent) )
            {
                nn::os::ClearTimerEvent(&timerEvent);
                g_pMyServerManager->AddUserWaitHolder(p);

                if ( timerWorking )
                {
                    nn::fan::impl::NotifyElapsedTime(ControlSpanUsec);
                }
            }
            break;
        case PmEventTag:
            if ( g_PmModule.GetEventPointer()->TryWait() )
            {
                g_PmModule.GetEventPointer()->Clear();
                g_pMyServerManager->AddUserWaitHolder(p);

                nn::psc::PmState state;
                nn::psc::PmFlagSet flags;
                NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&state, &flags));

                switch ( state )
                {
                case nn::psc::PmState_MinimumAwake:
                    nn::os::StartPeriodicTimerEvent(&timerEvent,
                        nn::TimeSpan::FromMicroSeconds(ControlSpanUsec),
                        nn::TimeSpan::FromMicroSeconds(ControlSpanUsec));
                    timerWorking = true;
                    nn::pinmux::SetPinAssignment(&g_PinmuxSession, nn::pinmux::PinAssignment_PwmFanPwm);
                    nn::fan::impl::EnableFanFromServer();
                    break;
                case nn::psc::PmState_SleepReady:
                case nn::psc::PmState_ShutdownReady:
                    nn::os::StopTimerEvent(&timerEvent);
                    timerWorking = false;
                    nn::fan::impl::DisableFanFromServer();
                    nn::pinmux::SetPinAssignment(&g_PinmuxSession, nn::pinmux::PinAssignment_PwmFanGpio);
                    break;
                default:
                    break;
                }
                g_PmModule.Acknowledge(state, ResultSuccess());
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    nn::os::StopTimerEvent(&timerEvent);
    nn::os::FinalizeMultiWaitHolder(&timerEventHolder);
    nn::os::FinalizeTimerEvent(&timerEvent);
}

void RequestStopServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.Finalize());

    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    g_pMyServerManager->RequestStop();
}

void FinalizeServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // MyServerManager のデストラクト
    // 登録したサービスなどはここで登録が解除される。
    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;

    nn::fan::Finalize();
}

} // namespace

void InitializeFanControlServer() NN_NOEXCEPT
{
    // FAN 用サーバの初期化
    // 準備ができていないサービスに対して接続中に、サービスの登録を行おうとするとデッドロックする場合があるので
    // ポート登録は何よりも先に行うのが安全
    InitializeServer();

    nn::pinmux::Initialize();
    nn::pinmux::OpenSession(&g_PinmuxSession, nn::pinmux::AssignablePinGroupName_PwmFan);

    // FAN の実装レイヤの初期化
    nn::fan::impl::Initialize();

    // FAN に依存している PSM が VDD50A を立ち上げる前に FAN PWM を初期化する。
    nn::fan::OpenController(&g_FanController, nn::fan::FanName_Cpu);
}

void StartFanControlServer() NN_NOEXCEPT
{
    auto result = nn::os::CreateThread(&g_FanServerThread, &FanHipcServerFunction, nullptr,
                                       g_FanServerStack, sizeof(g_FanServerStack),
                                       NN_SYSTEM_THREAD_PRIORITY(fan, IpcServer));
    NN_UNUSED(result);
    NN_SDK_ASSERT(result.IsSuccess());

    nn::os::SetThreadNamePointer(&g_FanServerThread, NN_SYSTEM_THREAD_NAME(fan, IpcServer));

    nn::os::StartThread(&g_FanServerThread);

    NN_DETAIL_FAN_INFO("Start FAN Control Server.\n");
}

void RequestStopFanControlServer() NN_NOEXCEPT
{
    // FAN サーバ終了をリクエスト
    RequestStopServer();
}

void FinalizeFanControlServer() NN_NOEXCEPT
{
    nn::fan::CloseController(&g_FanController);

    // FAN 用サーバ処理スレッド終了待機と破棄
    nn::os::WaitThread(&g_FanServerThread);
    nn::os::DestroyThread(&g_FanServerThread);

    // FAN サーバの終了処理
    FinalizeServer();

    // FAN サーバ自体の Finalize
    nn::fan::impl::Finalize();

    // Pinmux ライブラリの Finalize
    nn::pinmux::CloseSession(&g_PinmuxSession);
    nn::pinmux::Finalize();
}

}}  // namespace nn::ptm
