﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkLog.h>
#include <nn/diag/detail/diag_DetailStructuredSdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

namespace nn { namespace profiler {
    extern void SendPcDebugOutput(const char* message, ...);
}}

#define LOG_AS_FATAL    4 // NOLINT(preprocessor/const)
#define LOG_AS_ERROR    3 // NOLINT(preprocessor/const)
#define LOG_AS_WARNING  2 // NOLINT(preprocessor/const)
#define LOG_AS_INFO     1 // NOLINT(preprocessor/const)
#define LOG_AS_DEBUG    0 // NOLINT(preprocessor/const)
#define LOG_AS_VERBOSE  -1 // NOLINT(preprocessor/const)

NN_STATIC_ASSERT(LOG_AS_FATAL == ::nn::diag::LogSeverity_Fatal);
NN_STATIC_ASSERT(LOG_AS_ERROR == ::nn::diag::LogSeverity_Error);
NN_STATIC_ASSERT(LOG_AS_WARNING == ::nn::diag::LogSeverity_Warn);
NN_STATIC_ASSERT(LOG_AS_INFO == ::nn::diag::LogSeverity_Info);
NN_STATIC_ASSERT(LOG_AS_DEBUG == ::nn::diag::LogSeverity_Trace);

#ifndef MINIMUM_LOG_TIER
#define MINIMUM_LOG_TIER LOG_AS_WARNING
#endif

// Ensure that we always log warning, error, and fatal messages
NN_STATIC_ASSERT(MINIMUM_LOG_TIER <= LOG_AS_WARNING);

//#define TIERED_LOG(tier, ...) \
//    do { \
//        if (NN_STATIC_CONDITION((tier) >= MINIMUM_LOG_TIER)) { \
//            if      (NN_STATIC_CONDITION((tier) == LOG_AS_FATAL))   { NN_DETAIL_STRUCTURED_SDK_LOG(profiler, Fatal, 0, __VA_ARGS__); } \
//            else if (NN_STATIC_CONDITION((tier) == LOG_AS_ERROR))   { NN_DETAIL_STRUCTURED_SDK_LOG(profiler, Error, 0, __VA_ARGS__); } \
//            else if (NN_STATIC_CONDITION((tier) == LOG_AS_WARNING)) { NN_DETAIL_STRUCTURED_SDK_LOG(profiler, Warn, 0, __VA_ARGS__); } \
//            else if (NN_STATIC_CONDITION((tier) == LOG_AS_INFO))    { NN_DETAIL_STRUCTURED_SDK_LOG(profiler, Info, 0, __VA_ARGS__); } \
//            else if (NN_STATIC_CONDITION((tier) == LOG_AS_DEBUG))   { NN_DETAIL_STRUCTURED_SDK_LOG(profiler, Trace, 0, __VA_ARGS__); } \
//            else if (NN_STATIC_CONDITION((tier) == LOG_AS_VERBOSE)) { NN_DETAIL_STRUCTURED_SDK_LOG(profiler, Trace, 0, __VA_ARGS__); } \
//        } \
//    } while (NN_STATIC_CONDITION(0))

#if !defined(NN_PROFILER_SEND_LOGS_TO_PC)

#if defined(NN_DETAIL_ENABLE_SDK_STRUCTURED_LOG)
#define TIERED_LOG(tier, ...) \
        do \
        { \
            if (NN_STATIC_CONDITION((tier) >= MINIMUM_LOG_TIER)) \
            { \
                ::nn::diag::LogMetaData logMetaData;                            \
                logMetaData.sourceInfo.lineNumber = __LINE__;                   \
                logMetaData.sourceInfo.fileName = __FILE__;                     \
                logMetaData.sourceInfo.functionName = NN_CURRENT_FUNCTION_NAME; \
                logMetaData.moduleName = "$profiler";                           \
                logMetaData.severity = ::nn::diag::LogSeverity(tier);           \
                logMetaData.verbosity = 0;                                      \
                logMetaData.useDefaultLocaleCharset = false;                    \
                logMetaData.pAdditionalData = nullptr;                          \
                logMetaData.additionalDataBytes = 0;                            \
                if (logMetaData.severity < 0) { logMetaData.severity = ::nn::diag::LogSeverity_Trace; } \
                ::nn::diag::detail::LogImpl(logMetaData, __VA_ARGS__);  \
            } \
        } while (NN_STATIC_CONDITION(0))
#else // !defined(NN_DETAIL_ENABLE_SDK_STRUCTURED_LOG)
#define TIERED_LOG(tier, ...) ((void)0)
#endif

#else // defined(NN_PROFILER_SEND_LOGS_TO_PC)
#define TIERED_LOG(tier, ...) \
        do \
        { \
            if (NN_STATIC_CONDITION((tier) >= MINIMUM_LOG_TIER)) \
            { \
                ::nn::profiler::SendPcDebugOutput(__VA_ARGS__); \
            } \
        } while (NN_STATIC_CONDITION(0))
#endif

#define FORCE_LOG(...) TIERED_LOG(MINIMUM_LOG_TIER, __VA_ARGS__)
#define FATAL_LOG(...) TIERED_LOG(LOG_AS_FATAL, __VA_ARGS__)
#define ERROR_LOG(...) TIERED_LOG(LOG_AS_ERROR, __VA_ARGS__)
#define WARNING_LOG(...) TIERED_LOG(LOG_AS_WARNING, __VA_ARGS__)
#define INFO_LOG(...) TIERED_LOG(LOG_AS_INFO, __VA_ARGS__)
#define DEBUG_LOG(...) TIERED_LOG(LOG_AS_DEBUG, __VA_ARGS__)
#define VERBOSE_LOG(...) TIERED_LOG(LOG_AS_VERBOSE, __VA_ARGS__)

#define DumpResultInformation(tier, result) \
    TIERED_LOG((tier), "Result: %s\n  InnerValue: 0x%08x\n  Module: %d\n  Description: %d\n  File: %s\n  Line: %d\n", \
        NN_MACRO_STRINGIZE(result), \
        result.GetInnerValueForDebug(), \
        result.GetModule(), \
        result.GetDescription(), \
        __FILE__, \
        __LINE__)

#define DUMP_CURRENT_LINE() TIERED_LOG(LOG_AS_DEBUG, "File: %s; Line: %d\n", __FILE__, __LINE__)

#if LOG_AS_DEBUG >= MINIMUM_LOG_TIER
#include <nn/os/os_Thread.h>
#define DumpThreadInformation() \
        do \
        { \
            ::nn::os::ThreadType *currentThread = ::nn::os::GetCurrentThread(); \
            DEBUG_LOG("-- %s --\n", ::nn::os::GetThreadNamePointer(currentThread)); \
            DEBUG_LOG(" Thread ID: %llu\n", ::nn::os::GetThreadId(currentThread)); \
            DEBUG_LOG(" Processor: %d\n", ::nn::os::GetCurrentProcessorNumber()); \
            DEBUG_LOG(" Priority : %d\n", ::nn::os::GetThreadCurrentPriority(currentThread)); \
            DEBUG_LOG("--\n\n"); \
        } while (NN_STATIC_CONDITION(false))
#else
#define DumpThreadInformation() (void(0))
#endif

#define ProfilerPanic(...) NN_ABORT(__VA_ARGS__)
