﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>

#include <nn/nn_Common.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma GCC diagnostic ignored "-Wsign-conversion"
#include <nn/os/os_Thread.h>
#include <nn/os/os_Event.h>
#include <nn/ro/detail/ro_RoModule.h>
#include <nn/util/util_BitUtil.h>
NN_PRAGMA_POP_WARNINGS

#include <nn/profiler/profiler_Result.h>
#include <nn/profiler/profiler_Api.h>
#include <nn/profiler/profiler_Control.private.h>

#include "profiler_LibPrivate.h"

#include "pmu/profiler_PerfCounterGroups.h"
#include "profiler_Comms.h"
#include "profiler_CommsIpc.h"
#include "profiler_Core.h"
#include "profiler_DataStream.h"
#include "profiler_Defines.h"
#include "profiler_LibModule.h"
#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_RecordMethods.h"
#include "profiler_ResultPrivate.h"
#include "profiler_SampleRates.h"
#include "profiler_SamplingThread.h"
#include "profiler_StringTable.h"
#include "profiler_TargetApplication.h"
#include "profiler_Workarea.h"



namespace nn { namespace profiler {

namespace /*anonymous*/
{
    std::atomic<ProfilerStatus> sProfilerStatus;

    StringTable* gMarkerNames;


    WorkArea* GetFirstActiveCore(uint32_t* pOutCore)
    {
        // Profiling not active
        if (GetProfilerStatus() != ProfilerStatus_Profiling) { return nullptr; }

        WorkArea *ws = GetWorkAreaForCore(SampleBufferIndex_Instrumentation);
        if (!CheckAndExpandBuffersIfNeeded(SampleBufferIndex_Instrumentation))
        {
            StopProfilingSamplingThreads();
            return nullptr;
        }

        for (uint32_t i = 0; i < SupportedCoreCount; ++i)
        {
            if (ws->record_cores & (1 << i))
            {
                *pOutCore = i;
                break;
            }
        }

        return ws;
    }
}


void InitializeLib()
{
    nn::Result result;

    gMarkerNames = reinterpret_cast<StringTable*>(Memory::GetInstance()->Allocate(sizeof(StringTable)));
    new (gMarkerNames) StringTable;

    ModuleInitialize();
}



void SetProfilerStatus(ProfilerStatus status)
{
    sProfilerStatus = status;

    IpcEventInfo event;
    event.event = IpcEvent_Status;
    event.info.status.status = status;

    if (TargetApplication::GetCurrent()->IsLibraryInitialized())
    {
        auto eventQueue = GetIpcEventQueue();
        eventQueue->Push(&event);
    }
}



ProfilerStatus GetProfilerStatus()
{
    return sProfilerStatus.load();
}



StringTable* GetMarkerNamesStringTable()
{
    return gMarkerNames;
}


void RecordVsync()
{
    uint32_t core;
    WorkArea* ws = GetFirstActiveCore(&core);
    if (ws != nullptr)
    {
        RecordHeartbeat(ws, nn::profiler::Heartbeats_Vsync, core);
    }
}


nn::Result StartProfiling()
{
    ProfilerStatus status = GetProfilerStatus();
    if (status == ProfilerStatus_Offline)
    {
        return nn::profiler::ResultNotInitialized();
    }
    else if (status != ProfilerStatus_Active)
    {
        return nn::profiler::ResultAlreadyDone();
    }

    auto pSettings = GetProfilerSettingsPointer();
    SignalBeginEvent();

    if (pSettings->IsOutOfProcess())
    {
        nn::profiler::WaitProfilingStarted();
    }

    return nn::ResultSuccess();
}



nn::Result StopProfiling()
{
    ProfilerStatus status = GetProfilerStatus();
    if (status == ProfilerStatus_Offline)
    {
        return nn::profiler::ResultNotInitialized();
    }
    else if (status != ProfilerStatus_Profiling)
    {
        return nn::profiler::ResultAlreadyDone();
    }

    StopProfilingSamplingThreads();

    if (GetProfilerSettingsPointer()->IsOutOfProcess())
    {
        nn::profiler::WaitCoresClosed();
    }

    return nn::ResultSuccess();
}



nn::Result SetProfileSettings(
    uint32_t affinityMask,
    uint32_t flags,
    PerformanceCounterGroup performanceCounterGroup,
    SampleRate sampleRate)
{
    ProfilerStatus status = GetProfilerStatus();
    if (status == ProfilerStatus_Offline)
    {
        return nn::profiler::ResultNotInitialized();
    }
    else if (status != ProfilerStatus_Active)
    {
        return nn::profiler::ResultInvalidProfilerStatus();
    }

    if ((affinityMask == 0) ||
        ((affinityMask & (~nn::os::GetThreadAvailableCoreMask())) != 0))
    {
        ERROR_LOG("Invalid affinty mask passed in.\n");
        return nn::profiler::ResultInvalidArgument();
    }
    affinityMask &= SupportedCoreMask;

    if ((flags & ~static_cast<uint32_t>(SettingsFromThePcGui::FlagBit_MASK)) != 0)
    {
        ERROR_LOG("Invalid flags!\n");
        return nn::profiler::ResultInvalidArgument();
    }

    auto profilingModes = TargetApplication::GetCurrent()->GetAllAvailableProfilingModes();
    if (((flags & Flags_OutOfProcessProfiling) && !(profilingModes & ProfilingMode_OutOfProcess)) ||
        (!(flags & Flags_OutOfProcessProfiling) && !(profilingModes & ProfilingMode_InProcess)))
    {
        ERROR_LOG("Invalid profiling mode selected.\n");
        return nn::profiler::ResultInvalidArgument();
    }

    if (performanceCounterGroup < PerformanceCounterGroup_Disabled ||
        performanceCounterGroup > PerformanceCounterGroup_UnalignedDataAccess)
    {
        ERROR_LOG("Invalid performance counter group passed in.\n");
        return nn::profiler::ResultInvalidArgument();
    }

    if (!(SampleRate_ByTimeMin <= sampleRate && sampleRate < SampleRate_ByTimeMax))
    {
        ERROR_LOG("Invalid sample rate passed in.\n");
        return nn::profiler::ResultInvalidArgument();
    }

    auto pSettings = GetProfilerSettingsPointer();

    pSettings->version = ProfilerRuntimeVersion;

    pSettings->coreMask = affinityMask;

    pSettings->flags = flags & SettingsFromThePcGui::FlagBit_MASK;

    auto perfGroup = pmu::GetPerformanceCounterGroup(performanceCounterGroup);
    for (int i = 0; i < pmu::PerformanceCounterCount; ++i)
    {
        pSettings->perf_counters[i] = (*perfGroup)[i];
    }
    pSettings->perf_counter_cycle = (performanceCounterGroup > 0) ? 1u : 0u;

    uint32_t actualSampleRate = 0;
    if (sampleRate < SampleRate_ByTimeMax)
    {
        actualSampleRate = gSampleByTimeRates[sampleRate];
    }
    pSettings->requested_time_between_samples_in_nanoseconds = actualSampleRate;

    return nn::ResultSuccess();
}

} // profiler
} // nn
