﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma GCC diagnostic ignored "-Wsign-conversion"
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
NN_PRAGMA_POP_WARNINGS

#include "profiler_PccomTaskQueue.h"

namespace nn { namespace profiler { namespace pccom
{
    // Constructor sets default values.
    TaskQueue::TaskQueue()
        : m_current(0)
        , m_elements(0)
    {
    }



    void TaskQueue::Initialize()
    {
        nn::os::InitializeEvent(&m_event, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMutex(&m_mutex, false, 0);
    }



    void TaskQueue::Finalize()
    {
        ForceReleaseWait();
        nn::os::YieldThread(); // Give things waiting on the wait a chance to release before finalizing

        nn::os::FinalizeEvent(&m_event);
        nn::os::FinalizeMutex(&m_mutex);
    }



    // Return the object at the front of the queue.  Because this needs
    // to be thread safe, we are returning the structure by value.
    TaskItem const * TaskQueue::GetFront()
    {
        TaskItem *front = nullptr;

        ////////////////////////////////////////////
        // Critical Section
        nn::os::LockMutex(&m_mutex);
        if (m_elements > 0)
        {
            front = &m_queue[m_current];
        }
        nn::os::UnlockMutex(&m_mutex);
        //
        ////////////////////////////////////////////

        return front;
    }



    // Adds a task to the queue.
    bool TaskQueue::Push(TaskItem* task)
    {
        bool pushed = false;

        ////////////////////////////////////////////
        // Critical Section
        nn::os::LockMutex(&m_mutex);

        // Make sure we aren't full.
        if (m_elements < TASKQUEUE_SIZE)
        {
            uint32_t next = m_current + m_elements;
            if (next >= TASKQUEUE_SIZE)
                next -= TASKQUEUE_SIZE;

            // Copy the new task.
            m_queue[next] = *task;

            ++m_elements;

            pushed = true;
        }

        nn::os::UnlockMutex(&m_mutex);
        //
        ////////////////////////////////////////////

        if (m_elements > 0)
        {
            nn::os::SignalEvent(&m_event);
        }

        return pushed;
    }



    // Removes a task from the queue.
    void TaskQueue::Pop()
    {
        ////////////////////////////////////////////
        // Critical Section
        nn::os::LockMutex(&m_mutex);

        // Make sure we aren't empty.
        if (m_elements > 0)
        {
            uint32_t next = m_current + 1;
            if (next >= TASKQUEUE_SIZE)
                next = 0;

            m_current = next;

            --m_elements;
        }

        nn::os::UnlockMutex(&m_mutex);
        //
        ////////////////////////////////////////////

        if (m_elements == 0)
        {
            nn::os::ClearEvent(&m_event);
        }
    }



    bool TaskQueue::WaitForItem()
    {
        nn::os::WaitEvent(&m_event);
        return (m_elements != 0);
    }



    void TaskQueue::ForceReleaseWait()
    {
        nn::os::SignalEvent(&m_event);
        nn::os::YieldThread();
        nn::os::ClearEvent(&m_event);
    }



    uint32_t TaskQueue::CurrentCount()
    {
        return m_elements;
    }

} // pccom
} // profiler
} // nn
