﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <type_traits>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/psc.h>

#include <nn/bpc/bpc_ServiceName.h>
#include <nn/bpc/bpc_IBoardPowerControlManager.sfdl.h>
#include <nn/bpc/bpc_IRtcManager.sfdl.h>
#include <nn/bpc/bpc_ManagerImpl.h>
#include <nn/bpc/bpc_ShimInternal.h>
#include <nn/bpc/detail/bpc_Log.h>
#include <nn/bpc/driver/bpc.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include "pcv_BpcServer.h"

namespace nn { namespace pcv {

namespace {

// サーバー側から発行する全セッションの Max
const int BoardPowerControlSessionCountMax = 13; // Manager 分（利用プロセス数）
const int RtcSessionCountMax = 2; // Manager 分（利用プロセス数）
const int BpcSessionCountMax = BoardPowerControlSessionCountMax + RtcSessionCountMax;

class BpcServerManager : public nn::sf::HipcSimpleAllInOneServerManager<BpcSessionCountMax, 4>
{};

BpcServerManager g_BpcServerManager;

const size_t BpcStackSize = 4096;
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_BpcServerStack[BpcStackSize];
nn::os::ThreadType g_BpcServerThread;

nn::sf::UnmanagedServiceObject<nn::bpc::IBoardPowerControlManager, nn::bpc::BoardPowerControlManagerImpl> g_BoardPowerControlManager;
nn::sf::UnmanagedServiceObject<nn::bpc::IRtcManager, nn::bpc::RtcManagerImpl> g_RtcManager;

// スリープ等の通知を受け取るオブジェクト
nn::psc::PmModule g_PmModule;

}

nn::Result HandleBpcPmEvent(nn::psc::PmState state, nn::psc::PmFlagSet flags)
{
    NN_UNUSED(flags);
    switch (state)
    {
        case nn::psc::PmState_EssentialServicesSleepReady:
        {
            // WakeupTimer の RTC アラームをセット
            bool isEnabled;
            int enabledTimerHandle;
            nn::bpc::WakeupTimerType enabledTimerType;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bpc::driver::EnableWakeupTimerOnDevice(&isEnabled, &enabledTimerHandle, &enabledTimerType));
            NN_UNUSED(enabledTimerHandle);
            NN_UNUSED(enabledTimerType);

            // IRQ ハンドラを止める
            nn::bpc::driver::SuspendIrq();
            break;
        }
        case nn::psc::PmState_EssentialServicesAwake:
        {
            // IRQ ハンドラを再開する
            // この中で起床要因レジスタが読まれ GetWakeupReason ができるようになる
            nn::bpc::driver::ResumeIrq();

            // 半起床向けの WakeupTimer の RTC アラームをすべてクリア
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bpc::driver::CleanAllWakeupTimers(nn::bpc::WakeupTimerType_BackgroundTask));
            break;
        }
        case nn::psc::PmState_FullAwake:
        {
            // 全起床向けの WakeupTimer の RTC アラームをすべてクリア
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bpc::driver::CleanAllWakeupTimers(nn::bpc::WakeupTimerType_FullWakeup));
            break;
        }
        default:
            break;
    }
    NN_RESULT_SUCCESS;
}

void LoopBpcServer(void* arg) NN_NOEXCEPT
{
    nn::os::MultiWaitHolderType pmEventHolder;
    nn::os::InitializeMultiWaitHolder(
                &pmEventHolder,
                g_PmModule.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(
                &pmEventHolder,
                nn::psc::PmModuleId_Bpc); // PmModuleId は SF のタグと衝突しないことを保証している

    g_BpcServerManager.AddUserWaitHolder(&pmEventHolder);

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::MultiWaitHolderType* p = g_BpcServerManager.Wait();

        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
        // SF processing
        case BpcServerManager::HipcSimpleAllInOneServerManager::AcceptTag:
        case BpcServerManager::HipcSimpleAllInOneServerManager::InvokeTag:
            g_BpcServerManager.ProcessAuto(p);
            break;

        case nn::psc::PmModuleId_Bpc:
            {
                bool tryWaitResult = g_PmModule.GetEventPointer()->TryWait();
                NN_SDK_ASSERT(tryWaitResult);
                NN_UNUSED(tryWaitResult);
                g_PmModule.GetEventPointer()->Clear();
                g_BpcServerManager.AddUserWaitHolder(&pmEventHolder);

                nn::psc::PmState    state;
                nn::psc::PmFlagSet  flags;
                auto result = g_PmModule.GetRequest(&state, &flags);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                auto pmEventResult = HandleBpcPmEvent(state, flags);

                g_PmModule.Acknowledge(state, pmEventResult);
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void InitializeBpcServer() NN_NOEXCEPT
{
#if defined(NN_PCV_BUILD_TYPE_SAFE_MODE)
    // セーフモードでは RTC リセットから復帰しない
    nn::bpc::driver::Initialize(false);
#else

    nn::bpc::driver::Initialize(true);

#endif

    nn::bpc::InitializeBoardPowerControlWith(g_BoardPowerControlManager.GetShared());
    nn::bpc::InitializeRtcWith(g_RtcManager.GetShared());
}

void StartBpcServer() NN_NOEXCEPT
{
    // PSC へのモジュール登録
    const nn::psc::PmModuleId   dependencies[] = {nn::psc::PmModuleId_I2cPowerBus}; // max7762x に依存
    auto result = g_PmModule.Initialize(
            nn::psc::PmModuleId_Bpc,
            dependencies,
            sizeof(dependencies) / sizeof(dependencies[0]),
            nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // サービス名の登録とポートの初期化
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    g_BpcServerManager.RegisterObjectForPort(nn::bpc::GetInternalBoardPowerControlManager(),
                                               BoardPowerControlSessionCountMax, nn::bpc::BoardPowerControlServiceName);
    g_BpcServerManager.RegisterObjectForPort(nn::bpc::GetInternalRtcManager(),
                                               RtcSessionCountMax, nn::bpc::RtcServiceName);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要がある。
    g_BpcServerManager.Start();

    // BPC 用サーバ処理スレッドの作成
    result = nn::os::CreateThread(&g_BpcServerThread,
                                       LoopBpcServer,
                                       nullptr,
                                       g_BpcServerStack,
                                       sizeof(g_BpcServerStack),
                                       NN_SYSTEM_THREAD_PRIORITY(bpc, IpcServer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::SetThreadNamePointer(&g_BpcServerThread, NN_SYSTEM_THREAD_NAME(bpc, IpcServer));

    nn::os::StartThread(&g_BpcServerThread);
    NN_DETAIL_BPC_INFO("Start Bpc Server.\n");
}

void RequestStopBpcServer() NN_NOEXCEPT
{
    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    g_BpcServerManager.RequestStop();
}

void WaitAndFinalizeBpcServer() NN_NOEXCEPT
{
    nn::os::WaitThread(&g_BpcServerThread);
    nn::os::DestroyThread(&g_BpcServerThread);

    nn::bpc::driver::Finalize();
}

}} // namespace nn::pcv
