﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/npns/detail/npns_Utility.h>
#include <mutex>
#include <cstdlib>

extern "C" {
#include <src/common.h>
}

#include "npns_XmppStanza.h"

namespace nn{ namespace npns{

#define NN_NPNS_XMPP_HANDLE_ERROR_IF_SUCCESS_PREVIOUS(body)             \
            if (m_resultLast.IsSuccess())                               \
            {                                                           \
                NN_SDK_ASSERT_NOT_NULL(m_pStanza);                      \
                HandleError( body );                                    \
            }

#define NN_NPNS_XMPP_RETURN_VALUE_IF_FAILED_PREVIOUS(v)                 \
            do{                                                         \
            if (!m_pStanza || m_resultLast.IsFailure())                 \
            {                                                           \
                return v;                                               \
            }                                                           \
            }while(NN_STATIC_CONDITION(false))


Xmpp::Stanza::Stanza()
    : m_pStanza(NULL), m_resultLast(ResultSuccess())
{
}

Xmpp::Stanza::Stanza(xmpp_ctx_t* pContext, const char* pName)
    : m_pStanza(NULL), m_resultLast(ResultSuccess())
{
    Create(pContext);
    SetName(pName);
}

Xmpp::Stanza::Stanza(xmpp_ctx_t* pContext, const char* pName, const char* pNs)
    : m_pStanza(NULL), m_resultLast(ResultSuccess())
{
    Create(pContext);
    SetName(pName);
    SetNs(pNs);
}

Xmpp::Stanza::Stanza(xmpp_ctx_t* pContext)
    : m_pStanza(NULL), m_resultLast(ResultSuccess())
{
    Create(pContext);
}

Xmpp::Stanza::Stanza(xmpp_stanza_t* const pRaw)
    : m_resultLast(ResultSuccess())
{
    m_pStanza = xmpp_stanza_clone(pRaw);
}

Xmpp::Stanza::Stanza(const Stanza& stanza)
    : m_pStanza(stanza.m_pStanza ? xmpp_stanza_copy(stanza.m_pStanza) : nullptr), m_resultLast(ResultSuccess())
{
}

Xmpp::Stanza & Xmpp::Stanza::operator=(const Stanza & stanza)
{
    if (m_pStanza)
    {
        xmpp_stanza_release(m_pStanza);
        m_pStanza = NULL;
    }
    if (stanza.m_pStanza)
    {
        m_pStanza = xmpp_stanza_copy(stanza.m_pStanza);
    }
    m_resultLast = stanza.m_resultLast;
    return *this;
}

Xmpp::Stanza::~Stanza()
{
    Destroy();
}

void Xmpp::Stanza::SetName(const char* pName)
{
    NN_NPNS_XMPP_HANDLE_ERROR_IF_SUCCESS_PREVIOUS(xmpp_stanza_set_name(m_pStanza, pName));
}

void Xmpp::Stanza::SetType(const char* pType)
{
    NN_NPNS_XMPP_HANDLE_ERROR_IF_SUCCESS_PREVIOUS(xmpp_stanza_set_type(m_pStanza, pType));
}

void Xmpp::Stanza::SetId(const char* pId)
{
    NN_NPNS_XMPP_HANDLE_ERROR_IF_SUCCESS_PREVIOUS(xmpp_stanza_set_id(m_pStanza, pId));
}

void Xmpp::Stanza::SetAttribute(const char* pKey, const char* pValue)
{
    NN_NPNS_XMPP_HANDLE_ERROR_IF_SUCCESS_PREVIOUS(xmpp_stanza_set_attribute(m_pStanza, pKey, pValue));
}

void Xmpp::Stanza::SetAttribute(const char* pKey, int32_t value)
{
    char buffer[20];
    xmpp_snprintf(buffer, sizeof(buffer), "%ld", value);
    NN_NPNS_XMPP_HANDLE_ERROR_IF_SUCCESS_PREVIOUS(xmpp_stanza_set_attribute(m_pStanza, pKey, buffer));
}

void Xmpp::Stanza::SetNs(const char* pNs)
{
    NN_NPNS_XMPP_HANDLE_ERROR_IF_SUCCESS_PREVIOUS(xmpp_stanza_set_ns(m_pStanza, pNs));
}

void Xmpp::Stanza::SetText(const char* pText)
{
    NN_NPNS_XMPP_HANDLE_ERROR_IF_SUCCESS_PREVIOUS(xmpp_stanza_set_text(m_pStanza, pText));
}

void Xmpp::Stanza::AddChild(Stanza& child)
{
    NN_SDK_REQUIRES(!child.m_pStanza || child.m_pStanza->parent == nullptr, "");
    MergeResult(child.m_resultLast);
    NN_NPNS_XMPP_HANDLE_ERROR_IF_SUCCESS_PREVIOUS(xmpp_stanza_add_child(m_pStanza, child.m_pStanza));
    //child.Dettach();
}

void Xmpp::Stanza::AddChildText(const char* pText)
{
    if (m_resultLast.IsFailure())
    {
        return;
    }
    NN_SDK_ASSERT_NOT_NULL(m_pStanza);
    Stanza stanzaText(m_pStanza->ctx);
    stanzaText.SetText(pText);
    AddChild(stanzaText);
}

const char* Xmpp::Stanza::GetNs() const
{
    return xmpp_stanza_get_ns(m_pStanza);
}

const char* Xmpp::Stanza::GetName() const
{
    return xmpp_stanza_get_name(m_pStanza);
}

bool Xmpp::Stanza::GetChildren(Stanza& stanza) const
{
    xmpp_stanza_t* pRawStanza = xmpp_stanza_get_children(m_pStanza);
    if (!pRawStanza)
    {
        return false;
    }
    stanza.CloneFrom(pRawStanza);
    return true;
}

bool Xmpp::Stanza::GetNext()
{
    xmpp_stanza_t* pRawStanza = xmpp_stanza_get_next(m_pStanza);
    if (!pRawStanza)
    {
        return false;
    }
    this->CloneFrom(pRawStanza);
    return true;
}

bool Xmpp::Stanza::GetChildByName(const char* pName, Stanza& stanza) const
{
    xmpp_stanza_t* pRawStanza = xmpp_stanza_get_child_by_name(m_pStanza, pName);
    if (!pRawStanza)
    {
        return false;
    }
    stanza.CloneFrom(pRawStanza);
    return true;
}

const char* Xmpp::Stanza::GetAttribute(const char* pName) const
{
    NN_NPNS_XMPP_RETURN_VALUE_IF_FAILED_PREVIOUS(nullptr);
    return xmpp_stanza_get_attribute(m_pStanza, pName);
}

const char* Xmpp::Stanza::GetType() const
{
    NN_NPNS_XMPP_RETURN_VALUE_IF_FAILED_PREVIOUS(nullptr);
    return xmpp_stanza_get_type(m_pStanza);
}

bool Xmpp::Stanza::IsTypeError() const
{
    if (!GetType())
    {
        return false;
    }
    return std::strcmp(GetType(), "error") == 0;
}

const char* Xmpp::Stanza::GetId() const
{
    NN_NPNS_XMPP_RETURN_VALUE_IF_FAILED_PREVIOUS(nullptr);
    return xmpp_stanza_get_id(m_pStanza);
}

bool Xmpp::Stanza::GetText(Text& text) const
{
    NN_NPNS_XMPP_RETURN_VALUE_IF_FAILED_PREVIOUS(false);
    char* pText = xmpp_stanza_get_text(m_pStanza);
    if (!pText)
    {
        return false;
    }
    text.Set(m_pStanza->ctx, pText);
    return true;
}

bool Xmpp::Stanza::GetAttributeAsInteger(int64_t* pOut, const char* attribute, int base) const
{
    const char* attributeString = GetAttribute(attribute);
    if (!attributeString)
    {
        return false;
    }

    char* pEnd = nullptr;
    int64_t value = std::strtoll(attributeString, &pEnd, base);

    if (*pEnd != '\0')
    {
        return false;
    }

    *pOut = value;
    return true;
}

Result Xmpp::Stanza::Send(xmpp_conn_t * const pConnection)
{
    if (m_resultLast.IsFailure())
    {
        return m_resultLast;
    }

    xmpp_send(pConnection, m_pStanza);

    return ResultSuccess();
}

void Xmpp::Stanza::HandleError(int error)
{
    switch (error)
    {
    case XMPP_EINVOP:
        m_resultLast = ResultXmppInvalidOperation();
        break;
    case XMPP_EMEM:
        m_resultLast = ResultXmppOutOfMemory();
        break;
    case XMPP_EOK:
        m_resultLast = ResultSuccess();
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void Xmpp::Stanza::MergeResult(Result result)
{
    if (m_resultLast.IsSuccess() && result.IsFailure())
    {
        m_resultLast = result;
    }
}

void Xmpp::Stanza::Create(xmpp_ctx_t* pContext)
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    NN_SDK_ASSERT(m_pStanza == NULL);

    m_pStanza = xmpp_stanza_new(pContext);
    if (!m_pStanza)
    {
        HandleError(XMPP_EMEM);
    }
}

void Xmpp::Stanza::Destroy()
{
    if (m_pStanza)
    {
        xmpp_stanza_release(m_pStanza);
        m_pStanza = NULL;
    }
}

void Xmpp::Stanza::CloneFrom(xmpp_stanza_t* const pRaw)
{
    Destroy();
    m_pStanza = xmpp_stanza_clone(pRaw);
}

void Xmpp::Stanza::Dettach()
{
    if (m_pStanza)
    {
        m_pStanza = NULL;
    }
}

}}

#if 1

class StanzaDebug : public nn::npns::Xmpp::Stanza
{
public:
    StanzaDebug(xmpp_ctx_t* pContext, const char* pName) : Stanza(pContext, pName) {}
    StanzaDebug(xmpp_ctx_t* pContext, const char* pName, const char* pNs) : Stanza(pContext, pName, pNs) {}
    xmpp_stanza_t* GetData() { return m_pStanza; }
    int GetReferenceCount() { return m_pStanza->ref;  }
};

void XmppStanzaLeakTest()
{
    xmpp_ctx_t* pContext = xmpp_ctx_new(NULL, NULL);
    {
        StanzaDebug stz(pContext, "name", "ns");

        NN_SDK_ASSERT(stz.GetReferenceCount() == 1);
        {
            StanzaDebug stz2 = stz;
            NN_SDK_ASSERT(stz.GetReferenceCount() == 1);
            NN_SDK_ASSERT(stz2.GetReferenceCount() == 1);

            {
                StanzaDebug stz3(pContext, "name", "ns");
                stz3.AddChild(stz2);
                NN_SDK_ASSERT(stz3.GetReferenceCount() == 1);
                NN_SDK_ASSERT(stz2.GetReferenceCount() == 2);
                NN_SDK_ASSERT(stz.GetReferenceCount() == 1);
            }
            NN_SDK_ASSERT(stz2.GetReferenceCount() == 1);
        }
        NN_SDK_ASSERT(stz.GetReferenceCount() == 1);

    }
    xmpp_ctx_free(pContext);
}


#endif
