﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_SystemSaveData.h>

#include <memory>

#include "npns_Common.h"
#include "npns_SaveDataManager.h"

namespace nn{ namespace npns{

SaveDataManager::SaveDataManager()
{
}

SaveDataManager::~SaveDataManager()
{
}

Result SaveDataManager::Initialize()
{
    nn::Result result;
    fs::DisableAutoSaveDataCreation();
    result = fs::MountSystemSaveData(NN_NPNS_SAVEDATA_VOLUME, NN_NPNS_SAVEDATA_ID);
    if (result.IsFailure())
    {
        if (result <= fs::ResultTargetNotFound())
        {
            result = fs::CreateSystemSaveData(NN_NPNS_SAVEDATA_ID, NN_NPNS_SAVEDATA_STORAGE_SIZE, NN_NPNS_SAVEDATA_JOURNAL_SIZE, NN_NPNS_SAVEDATA_FLAGS);
            if (result.IsFailure())
            {
                NN_NPNS_ERROR("Failed to create system save data.(0x%08x)\n", result.GetInnerValueForDebug());
                return result;
            }
            result = fs::MountSystemSaveData(NN_NPNS_SAVEDATA_VOLUME, NN_NPNS_SAVEDATA_ID);
        }
        if (result.IsFailure())
        {
            NN_NPNS_ERROR("Failed to mount system save data.(0x%08x)\n", result.GetInnerValueForDebug());
            return result;
        }
    }
    return ResultSuccess();
}

void SaveDataManager::Finalize()
{
    fs::Unmount(NN_NPNS_SAVEDATA_VOLUME);
}

Result SaveDataManager::Read(PersistentData * pData)
{
    Result result;
    fs::FileHandle file;

    std::memset(pData, 0, sizeof(PersistentData));
    pData->version = PersistentData::CurrentVersion;
/*
    pData->bHasValidJid = true;
    pData->creadential = Credential
    {
        "784c978622eeb583@app-jp1.td1.npns.srv.nintendo.net",
        "c18934c712d36e90a4c3f52909347633"
    };
*/
    result = fs::OpenFile(&file, NN_NPNS_SAVEDATA_FULLPATH, fs::OpenMode_Read);
    if (result.IsSuccess())
    {
        result = fs::ReadFile(file, 0, pData, sizeof(PersistentData));
        if (result.IsFailure())
        {
            NN_NPNS_ERROR("Failed to read save data.(0x%08x)\n", result.GetInnerValueForDebug());
        }

        fs::CloseFile(file);
    }
    else if (result <= fs::ResultPathNotFound())
    {
        NN_NPNS_INFO("Save data is initial state.\n");
        return result;
    }

    return ResultSuccess();
}

Result SaveDataManager::Write(const PersistentData & data)
{
    Result result;
    fs::FileHandle file;

    result = fs::OpenFile(&file, NN_NPNS_SAVEDATA_FULLPATH, fs::OpenMode_Write | fs::OpenMode_AllowAppend);
    if (result <= fs::ResultPathNotFound())
    {
        result = fs::CreateFile(NN_NPNS_SAVEDATA_FULLPATH, sizeof(PersistentData));
        if (result.IsFailure())
        {
            NN_NPNS_ERROR("Failed to create save data.(0x%08x)\n", result.GetInnerValueForDebug());
            return result;
        }
        result = fs::OpenFile(&file, NN_NPNS_SAVEDATA_FULLPATH, fs::OpenMode_Write | fs::OpenMode_AllowAppend);
    }

    if (result.IsFailure())
    {
        NN_NPNS_ERROR("Failed to open save data.(0x%08x)\n", result.GetInnerValueForDebug());
        return result;
    }

    result = fs::WriteFile(file, 0, &data, sizeof(PersistentData), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush));

    fs::CloseFile(file);

    if (result.IsFailure())
    {
        NN_NPNS_ERROR("Failed to write save data.(0x%08x)\n", result.GetInnerValueForDebug());
        return result;
    }

    result = fs::CommitSaveData(NN_NPNS_SAVEDATA_VOLUME);
    if (result.IsFailure())
    {
        NN_NPNS_ERROR("Failed to commit save data.(0x%08x)\n", result.GetInnerValueForDebug());
        return result;
    }

    return ResultSuccess();
}


}
}
