﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/util/util_IntrusiveList.h>

#include <list>
#include <memory>

#include "npns_NotificationQueue.h"

namespace nn{ namespace npns{

class Router
{
    NN_DISALLOW_COPY( Router );
    NN_DISALLOW_MOVE( Router );

public:
    class Entry : public nn::util::IntrusiveListBaseNode<Entry>
    {
    public:
        static const size_t ListSize = 32;

        NN_IMPLICIT Entry(NotificationQueue& queue)
            : m_queue(queue)
            , m_toIdCount(0)
        {
        }

        size_t GetTargetCount() const
        {
            return m_toIdCount;
        }

        bool IsTarget(ApplicationId toId) const
        {
            for (int i = 0; i < static_cast<int>(m_toIdCount); ++i)
            {
                if (m_toIdList[i] == toId)
                {
                    return true;
                }
            }
            return false;
        }

        Result Add(ApplicationId toId)
        {
            NN_SDK_REQUIRES_LESS(m_toIdCount, ListSize);

            if (IsTarget(toId))
            {
                return ResultAlreadyRegistered();
            }
            else if (m_toIdCount + 1 >= ListSize)
            {
                return ResultSizeLimitExceeded();
            }

            m_toIdList[m_toIdCount++] = toId;

            return ResultSuccess();
        }

        NotificationQueue& m_queue;

    private:
        ApplicationId m_toIdList[ListSize];
        size_t m_toIdCount;
    };
    typedef nn::util::IntrusiveListBaseNodeTraits<Entry> EntryTraits;
    typedef nn::util::IntrusiveList<Entry, EntryTraits> EntryList;

    Router();

    void RegisterRecipient(Entry& entry);
    void UnregisterRecipient(Entry& entry);
    Result AddRceiveTarget(Entry& entry, ApplicationId toId);

    void DeliverNotification(const NotificationData& nd);
protected:
    bool DeliverNotificationToId(ApplicationId toId, const NotificationData& nd);

private:
    EntryList m_listEntry;
    NotificationQueue m_queueNotificationDefault;

    mutable os::Mutex m_mutex;
};

}}


