﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "npns_NotificationQueue.h"

namespace nn {
namespace npns {

NotificationQueue::NotificationQueue(nn::os::SystemEvent* pEvent)
    : m_count(0)
    , m_pEvent(pEvent)
{

}

NotificationQueue::~NotificationQueue()
{
    while (!m_list.empty())
    {
        NotificationHolder *pHolder = &m_list.front();
        m_list.pop_front();
        delete pHolder;
    }
    m_count = 0;
}

Result NotificationQueue::Send(const NotificationData& nd)
{
    while(m_count >= MaxCount)
    {
        // 通知データを後勝ちにするため先頭から捨てる
        NotificationHolder *pHolder = Pop();
        NN_SDK_ASSERT_NOT_NULL(pHolder);
        delete pHolder;
    }

    NotificationHolder *pHolder = new(std::nothrow) NotificationHolder(nd);
    if (!pHolder)
    {
        return ResultOutOfMemory();
    }

    if (!Push(pHolder))
    {
        delete pHolder;
        return ResultInsufficientBuffer();
    }
    Notify();

    return ResultSuccess();
}

Result NotificationQueue::Receive(char* pBuffer, size_t size, uint16_t version)
{
    if (size < sizeof(NotificationData))
    {
        return ResultInsufficientBuffer();
    }

    if (m_list.empty())
    {
        NN_SDK_ASSERT_EQUAL(m_count, 0);
        return ResultNotReceived();
    }

    Result result;
    NotificationHolder *pHolder = Pop();
    NN_SDK_ASSERT_NOT_NULL(pHolder);

    switch (version)
    {
    case 1:
        // nd = static_cast<NotificationData>(*pHolder);
        std::memcpy(pBuffer, static_cast<NotificationData*>(pHolder), sizeof(NotificationData));
        result = ResultSuccess();
        break;
    default:
        result = ResultUnknownVersion();
        break;
    }

    delete pHolder;

    return result;
}

void NotificationQueue::MoveToAll(NotificationQueue& target)
{
    bool bNeedNotify = false;
    while (!m_list.empty())
    {
        NotificationHolder *pHolder = Pop();
        NN_SDK_ASSERT_NOT_NULL(pHolder);

        if (target.Push(pHolder))
        {
            bNeedNotify = true;
        }
        else
        {
            delete pHolder;
        }
    }
    if (bNeedNotify)
    {
        target.Notify();
    }
}

void NotificationQueue::MoveToPartial(ApplicationId toId, NotificationQueue& target)
{
    bool bNeedNotify = false;
retry:
    for (auto it = m_list.begin(); it != m_list.end(); ++it)
    {
        NotificationHolder& holder = *it;
        if (holder.GetApplicationId() == toId)
        {
            m_list.erase(it);
            if (target.Push(&holder))
            {
                bNeedNotify = true;
            }
            else
            {
                delete &holder;
            }
            goto retry;
        }
    }
    if (bNeedNotify)
    {
        target.Notify();
    }
}

bool NotificationQueue::Push(NotificationHolder* pHolder)
{
    if (m_count >= MaxCount)
    {
        return false;
    }

    m_list.push_back(*pHolder);
    ++m_count;

    return true;
}

NotificationQueue::NotificationHolder * NotificationQueue::Pop()
{
    if (m_list.empty())
    {
        return nullptr;
    }

    NotificationHolder* pHolder = &m_list.front();
    m_list.pop_front();
    --m_count;
    return pHolder;
}

void NotificationQueue::Notify()
{
    if (m_pEvent)
    {
        m_pEvent->Signal();
    }
}


}
}
