﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/msgpack.h>
#include <nn/account/account_ApiBaasAccessToken.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/npns/detail/npns_Utility.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>
#include <nn/bgtc.h>

#include <cstdlib>

#include "npns_Client.h"
#include "npns_Router.h"
#include "npns_Config.h"
#include "npns_Instance.h"
#include "npns_Log.h"
#include "npns_XmppStanza.h"
#include "npns_Common.h"

using nn::msgpack::JsonStreamParser;
using nn::msgpack::JsonStreamParserSettings;
using nn::msgpack::MemoryInputStream;

namespace nn{ namespace npns{

Client::Client()
    : Xmpp(NN_NPNS_CLIENT_NAME, NN_NPNS_CLIENT_VERSION)
    , m_SessionMaintainer(this)
    , m_InitialStatus(Status_Active)
{
}

void Client::Initialize()
{
    Xmpp::Initialize();

    m_InitialStatus = Status_Active;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::account::GetNetworkServiceAccountAvailabilityChangeNotifier(&m_AccountNotifier));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        m_AccountNotifier.GetSystemEvent(&m_AccountChangeEvent));
}

Result Client::SubscribeTopic(const char* pTopicName)
{
    Result result;

    if (!IsConnected())
    {
        return ResultNotConnected();
    }

    result = Xmpp::PubsubSubscribe(pTopicName);
    if (result <= ResultXmppItemNotFound())
    {
        NN_NPNS_WARN("Topic '%s' is not found.\n", pTopicName);
        return ResultTopicNotFound();
    }
    return result;
}

Result Client::UnsubscribeTopic(const char* pTopicName)
{
    Result result;

    if (!IsConnected())
    {
        return ResultNotConnected();
    }

    result = Xmpp::PubsubUnsubscribe(pTopicName);
    if (result <= ResultXmppItemNotFound())
    {
        NN_NPNS_WARN("Topic '%s' is not found.\n", pTopicName);
        return ResultTopicNotFound();
    }
    else if (result <= ResultXmppUnexpectedRequest())
    {
        NN_NPNS_WARN("Topic '%s' is not subscribed yet.\n", pTopicName);
        return ResultTopicNotSubscribed();
    }
    return result;
}

Result Client::QueryIsTopicExist(bool* pIsExist, const char* pTopicName)
{
    Result result;

    if (!IsConnected())
    {
        return ResultNotConnected();
    }

    result = Xmpp::PubsubGet(pTopicName, 1);
    if (result.IsSuccess())
    {
        *pIsExist = true;
    }
    else if (result <= ResultXmppItemNotFound())
    {
        *pIsExist = false;
    }
    else if (result.IsFailure())
    {
        return result;
    }

    return ResultSuccess();
}

TimeSpan Client::GetBackoffWaitFromServer() const
{
    return m_backoffWaitFromServer;
}

void Client::SetStatus(Status status)
{
    Result result;
    m_InitialStatus = status;
    if (IsConnected())
    {
        switch (status)
        {
        case Status_Active:
        {
            SetPresence("chat");

            // 異常切断の cutoff 通知のため chat 遷移でDAリストを送信(これプラス、NA紐づけ変更時にも送信)
            result = SendDeviceAccountListAsync();
            NN_UNUSED(result);

            Xmpp::SetTcpNoDelay(false);
            break;
        }
        case Status_Inactive:
        {
            Xmpp::SetTcpNoDelay(true);

            // dnd/away をトリガーにして cutoff 通知が発行される
#if NN_NPNS_ENABLE_WOWLAN
            SetPresence("dnd");
#else
            SetPresence("away");
#endif
            break;
        }
        default:
            NN_SDK_ASSERT(false);
        }
    }
}

Result Client::EnableWhiteSpacePing(nn::os::Event* pEventInterrupt, bool isPowerModeNormal)
{
    Xmpp::Stanza stanzaIq(Xmpp::GetContext(), "iq");
    stanzaIq.SetType("get");
    stanzaIq.SetAttribute("to", Xmpp::GetConnectedDomain());

    Xmpp::Stanza stanzaIntervals(Xmpp::GetContext(), "intervals", "urn:npns:wsp");
    stanzaIq.AddChild(stanzaIntervals);

    Xmpp::Stanza response;
    Result result = Xmpp::SendAndWait(&response, stanzaIq, pEventInterrupt);
    if (result.IsFailure())
    {
        NN_NPNS_WARN("Failed Client::EnableWhiteSpacePing(). (0x%08x)\n", result.GetInnerValueForDebug());
        return result;
    }

    TimeSpan normal, lazy;
    if (ParseWhiteSpacePingIntervalStanza(&normal, &lazy, response))
    {
        SendWhiteSpacePingModeAsync(isPowerModeNormal ? "normal" : "lazy");
        m_SessionMaintainer.EnableWhiteSpacePing(isPowerModeNormal ? normal : lazy);
    }

    return ResultSuccess();
}

void Client::TriggerCheckConnection()
{
    m_SessionMaintainer.SendDoubleSpaceKeepAliveIfNeeded();
}

Result Client::CheckConnectionAndWait(os::SystemEvent* pCancelEvent, TimeSpan timeout)
{
    return m_SessionMaintainer.PerformPingPong(pCancelEvent, timeout);
}

bool Client::ParseReceiverId(ReceiverId * pOut, const char * pSource)
{
    // 先頭の 0x はスキップ
    if (std::strncmp(pSource, "0x", 2) == 0)
    {
        pSource += 2;
    }

    // 32 桁であることを確認
    if (std::strlen(pSource) != sizeof(account::Uid) * 2)
    {
        return false;
    }

    const size_t Uint64CharCount = sizeof(uint64_t) * 2;
    char buffer[Uint64CharCount + 1] = { 0 };
    char* pFail = nullptr;

    std::strncpy(buffer, pSource, Uint64CharCount);
    pOut->_data[0] = std::strtoull(buffer, &pFail, 16);
    if (pFail[0])
    {
        return false;
    }
    pOut->_data[1] = std::strtoull(&pSource[Uint64CharCount], &pFail, 16);
    if (pFail[0])
    {
        return false;
    }

    return true;
}

bool Client::ParseApplicationId(uint64_t * pOut, const char * pSource)
{
    char* pEnd = nullptr;
    // strtoull は 0 始まりを 8進数扱いするので明示的に判定
    if (std::strncmp(pSource, "0x", 2) == 0)
    {
        *pOut = std::strtoull(pSource, &pEnd, 16);
    }
    else
    {
        *pOut = std::strtoull(pSource, &pEnd, 10);
    }
    if (*pEnd != '\0')
    {
        return false;
    }
    return true;
}

void Client::OnReceiveMessageBody(const Text& text)
{
    NN_NPNS_TRACE("Received notification: %s\n", text.Get());
    m_ndSendBuffer = NotificationData();
    NotificationData::Buffer& data = m_ndSendBuffer.m_Data;

    JsonStreamParser parser;
    JsonStreamParserSettings setting;
    setting.format = JsonStreamParserSettings::kFormatJson;
    setting.max_array_size    = 8;
    setting.max_map_size      = 64;
    setting.max_depth         = 4;
    setting.token_buffer_size = 2048;

    MemoryInputStream istream(text.Get(), text.GetLength());

    if (msgpack::IsError(parser.Init(setting))
        || msgpack::IsError(parser.Open(&istream)))
    {
        NN_NPNS_ERROR("Failed to initialize JSON parser.\n");
        return; // drop
    }

    JsonStreamParser::Event ev;

    if ((ev = parser.Next()) != JsonStreamParser::kEventStartMap)
    {
        NN_NPNS_WARN("Received data is not JSON.\n");
        return; // drop
    }

    while ((ev = parser.Next()) != JsonStreamParser::kEventEndDocument)
    {
        if (msgpack::IsError(parser.GetError()))
        {
            NN_NPNS_WARN("Failed to parse JSON.\n");
            return; // drop
        }
        if (ev != JsonStreamParser::kEventKeyName)
        {
            continue;
        }

        const char* pKeyName = parser.GetToken().buf;
        if (std::strcmp(pKeyName, "receiver_id") == 0 && parser.Next() == JsonStreamParser::kEventString)
        {
            const JsonStreamParser::Token& token = parser.GetToken();
            if (!ParseReceiverId(&data.m_ReceiverId, token.buf))
            {
                NN_NPNS_WARN("Failed to parse receiver id: %s\n", token.buf);
                return; // drop
            }
            continue;
        }
        if (std::strcmp(pKeyName, "application_id") == 0)
        {
            parser.Next();
            const JsonStreamParser::Token& token = parser.GetToken();
            if (!ParseApplicationId(&data.m_ApplicationId, token.buf))
            {
                NN_NPNS_WARN("Failed to parse application id: %s\n", token.buf);
                return; // drop
            }
            continue;
        }
        if (std::strcmp(pKeyName, "nt") == 0 && parser.Next() == JsonStreamParser::kEventString)
        {
            int ret = util::Strlcpy(data.m_Token.data, parser.GetToken().buf, TokenLength);
            if (ret >= PayloadMaxLength)
            {
                NN_NPNS_WARN("nt field is too large.\n");
                return; // drop
            }
            data.m_Type = NotificationType_Normal;
            continue;
        }
        if (std::strcmp(pKeyName, "data") == 0 && parser.Next() == JsonStreamParser::kEventString)
        {
            int ret = util::Strlcpy(reinterpret_cast<char*>(data.m_Payload), parser.GetToken().buf, PayloadMaxLength);
            if (ret >= PayloadMaxLength)
            {
                NN_NPNS_WARN("Too large data.\n");
                return; // drop
            }
            data.m_sizePayload = static_cast<uint16_t>(ret);
            continue;
        }
    }

    NN_NPNS_INFO("Received notification. (ApplicationId: 0x%016llx, Data: '%s')\n", data.m_ApplicationId, data.m_Payload);

    NN_NPNS_RECORD_EVENT(ReceiveNotification);
    if (bgtc::IsInHalfAwake())
    {
        NN_NPNS_RECORD_EVENT(ReceiveNotificationHalfawake);
    }
    g_Daemon.GetRouter().DeliverNotification(m_ndSendBuffer);

}

void Client::OnReceiveDisconnectError(const Xmpp::Stanza& stanzaIncoming)
{
    TimeSpan backoff;
    if (ParseBackoffStanza(&backoff, stanzaIncoming))
    {
        m_backoffWaitFromServer = backoff;
    }
}

void Client::OnReceiveIdle()
{
}

void Client::OnConnected()
{
    AddHandler(IdleHandlerEntry, "urn:npns:idle", "idle", NULL);

    // XMPP仕様として最初に Presence を送る.
    SetStatus(m_InitialStatus);

    EnableStreamManagement();

    // 以前受け取ったバックオフ秒数は無効化
    m_backoffWaitFromServer = TimeSpan(0);

    // WSP 無効化(必要なら後に有効化される) / pong ハンドラの登録
    m_SessionMaintainer.Initialize();

    // 統計情報のため NetworkProfile を送る
    Result result = SendOutOfHomeNetworkProfileAsync();
    if (result.IsFailure())
    {
        NN_NPNS_ERROR("Failed to Client::SendOutOfHomeNetworkProfileAsync(). (0x%08x)\n", result.GetInnerValueForDebug());
    }

    // システムバージョン送信
    result = SendSystemVersionAsync();
    if (result.IsFailure())
    {
        NN_NPNS_ERROR("Failed to Client::SendSystemVersionAsync(). (0x%08x)\n", result.GetInnerValueForDebug());
    }

    ActivateKeepAlive();
}

void Client::OnBeforeDisconnect()
{
    Xmpp::OnBeforeDisconnect();

    if (IsConnected())
    {
        // 切断処理高速化のため TCP_NODELAY:1 とする.
        // 再接続時には socket 再生成されるため元に戻さなくて良い.
        Xmpp::SetTcpNoDelay(true);
    }
}

void Client::OnFlush()
{
}

int Client::IdleHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const, void * const userdata)
{
    static_cast<Client*>(userdata)->OnReceiveIdle();
    return HandlerResult_NotOneShot;
}

void Client::TriggerWhiteSpacePing()
{
    m_SessionMaintainer.SendWhiteSpacePingIfNeeded();
}

Result Client::GetDeviceAccountIdList(int* pOutActualLen, uint64_t* pOutList, int outListLen) const
{
    int actualUserListLen;
    nn::account::Uid userList[nn::account::UserCountMax]; // 128byte

    Result result = nn::account::ListAllUsers(&actualUserListLen, userList, nn::account::UserCountMax);

    if (result.IsFailure())
    {
        NN_NPNS_ERROR("Failed to nn::account::ListAllUsers(). (0x%08x)\n", result.GetInnerValueForDebug());
    }
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    int actualOutListLen = 0;

    if (actualUserListLen > 0)
    {
        for (int i = 0; i < actualUserListLen && i < outListLen ; ++i)
        {
            uint64_t outId;
            result = nn::account::GetBaasDeviceAccountId(&outId, userList[i]);
            if (result.IsSuccess())
            {
                pOutList[actualOutListLen] = outId;
                actualOutListLen++;
            }
            else
            {
                // ネットワークサービスアカウントを利用可能でないなどのエラーが正常系でありえる
                NN_NPNS_TRACE("Failed to nn::account::GetBaasDeviceAccountId(). (0x%08x)\n", result.GetInnerValueForDebug());
            }
        }
    }

    *pOutActualLen = actualOutListLen;
    return ResultSuccess();
}

bool Client::MakeUInt64Csv(char* pOut, size_t outSize, const char* format, uint64_t* values, int valueLen) const
{
    char* p = pOut;
    int remainOutSize = static_cast<int>(outSize);

    for (int i = 0; i < valueLen ; ++i)
    {
        int written = util::SNPrintf(p, remainOutSize, format, values[i]);
        if (written <= 0 || (remainOutSize - written) <= 0)
        {
            return false;
        }
        remainOutSize -= written;
        p += written;

        if (i < (valueLen - 1))
        {
            written = util::SNPrintf(p, remainOutSize, ",");
            if (written <= 0 || (remainOutSize - written) <= 0)
            {
                return false;
            }
            remainOutSize -= written;
            p += written;
        }
    }
    return true;
}

Result Client::SendDeviceAccountListAsync()
{
    int actualLen;
    uint64_t deviceAccountIdList[nn::account::UserCountMax];
    auto result = GetDeviceAccountIdList(&actualLen, deviceAccountIdList, nn::account::UserCountMax);
    if (result.IsFailure())
    {
        NN_NPNS_ERROR("Failed to Client::GetDeviceAccountIdList(). (0x%08x)\n", result.GetInnerValueForDebug());
        return result;
    }

    Xmpp::Stanza stanzaIq(Xmpp::GetContext(), "iq");
    stanzaIq.SetType("set");
    stanzaIq.SetAttribute("to", Xmpp::GetConnectedDomain());

    Xmpp::Stanza stanzaAccounts(Xmpp::GetContext(), "accounts", "urn:npns:device-accounts");

    if (actualLen > 0)
    {
        char csvBuffer[256]; // DA1つで 16 文字、最大 8 ユーザーなので 256byte で十分
        bool makeCsvResult = MakeUInt64Csv(csvBuffer, sizeof(csvBuffer), "%016llx", deviceAccountIdList, actualLen);
        if (!makeCsvResult)
        {
            NN_NPNS_ERROR("Failed to Client::MakeUInt64Csv()\n");
            return ResultUnexpected();
        }
        stanzaAccounts.AddChildText(csvBuffer);
    }

    stanzaIq.AddChild(stanzaAccounts);

    Xmpp::Send(stanzaIq);

    return ResultSuccess();
}

Result Client::SendOutOfHomeNetworkProfileAsync()
{
    nn::nifm::NetworkProfileData* pProfile = new nn::nifm::NetworkProfileData;
    NN_RESULT_THROW_UNLESS(pProfile != nullptr, ResultOutOfMemory());

    NN_UTIL_SCOPE_EXIT
    {
        delete pProfile;
    };

    NN_NPNS_DETAIL_RETURN_IF_FAILED(nn::nifm::GetCurrentNetworkProfile(pProfile));

    // SSID リストに含まれる場合だけ送る(SIGLO-42142)
    if (pProfile->networkProfileType == nn::nifm::NetworkProfileType_SsidList)
    {
        Xmpp::Stanza stanzaIq(Xmpp::GetContext(), "iq");
        stanzaIq.SetType("set");
        stanzaIq.SetAttribute("to", Xmpp::GetConnectedDomain());

        char uuidString[nn::util::Uuid::StringSize];
        pProfile->id.ToString(uuidString, sizeof(uuidString));

        Xmpp::Stanza stanzaUuid(Xmpp::GetContext(), "uuid", "urn:npns:ooh");
        stanzaUuid.AddChildText(uuidString);

        stanzaIq.AddChild(stanzaUuid);

        Xmpp::Send(stanzaIq);

        NN_NPNS_INFO("Send network profile UUID:%s\n", uuidString);
    }

    return ResultSuccess();
}

Result Client::SendSystemVersionAsync()
{
    Xmpp::Stanza stanzaIq(Xmpp::GetContext(), "iq");
    stanzaIq.SetType("set");
    stanzaIq.SetAttribute("to", Xmpp::GetConnectedDomain());

    Xmpp::Stanza stanzaSystemVersion(Xmpp::GetContext(), "device", "urn:npns:system-version");

    nn::settings::system::FirmwareVersion fwVer;
    nn::settings::system::GetFirmwareVersion(&fwVer);
    stanzaSystemVersion.AddChildText(fwVer.displayVersion);

    stanzaIq.AddChild(stanzaSystemVersion);

    Xmpp::Send(stanzaIq);

    return ResultSuccess();
}

void Client::SendWhiteSpacePingModeAsync(const char* mode)
{
    NN_SDK_ASSERT(IsUsingProxy());

    Xmpp::Stanza stanzaIq(Xmpp::GetContext(), "iq");
    stanzaIq.SetType("set");
    stanzaIq.SetAttribute("to", Xmpp::GetConnectedDomain());

    Xmpp::Stanza stanzaMode(Xmpp::GetContext(), "mode", "urn:npns:wsp");
    stanzaMode.AddChildText(mode);
    stanzaIq.AddChild(stanzaMode);

    Xmpp::Send(stanzaIq);
}

bool Client::ParseBackoffStanza(TimeSpan* pOut, const Stanza& stanzaIncoming) const
{
    Stanza stanzaBody;
    if (!stanzaIncoming.GetChildByName("server-kick", stanzaBody))
    {
        return false;
    }

    int64_t backoffSeconds;
    if (!stanzaBody.GetAttributeAsInteger(&backoffSeconds, "backoff", 10))
    {
        NN_NPNS_WARN("Failed to parse backoff stanza.\n");
        return false;
    }

    if (backoffSeconds <= 0)
    {
        NN_NPNS_WARN("Invlaid backoff seconds (%lld).\n", backoffSeconds);
        return false;
    }

    NN_NPNS_INFO("Received server-kick. backoff seconds:%lld\n", backoffSeconds);
    *pOut = TimeSpan::FromSeconds(backoffSeconds);
    return true;
}

bool Client::ParseWhiteSpacePingIntervalStanza(TimeSpan* pOutNormal, TimeSpan* pOutLazy, const Stanza& stanzaIncoming) const
{
    if (!stanzaIncoming.GetName() || std::strncmp(stanzaIncoming.GetName(), "intervals", sizeof("intervals")) != 0)
    {
        return false;
    }

    Stanza child;
    if (!stanzaIncoming.GetChildren(child))
    {
        return false;
    }

    nn::util::optional<int64_t> normal, lazy;

    for (int i = 0; i < 2; ++i)
    {
        int64_t outSeconds;
        if (child.GetAttribute("mode") && child.GetAttributeAsInteger(&outSeconds, "seconds", 10))
        {
            NN_NPNS_TRACE("White Space Ping %s : interval %lld\n", child.GetAttribute("mode"), outSeconds);
            if (std::strncmp(child.GetAttribute("mode"), "normal", sizeof("normal")) == 0)
            {
                normal = outSeconds;
            }
            else if (std::strncmp(child.GetAttribute("mode"), "lazy", sizeof("lazy")) == 0)
            {
                lazy = outSeconds;
            }
        }
        if (!child.GetNext())
        {
            break;
        }
    }

    if (!normal || !lazy || *normal < 0 || *lazy < 0)
    {
        return false;
    }

    *pOutNormal = TimeSpan::FromSeconds(*normal);
    *pOutLazy = TimeSpan::FromSeconds(*lazy);
    return true;
}

void Client::RunOnce(unsigned long timeout)
{
    if (IsUsingProxy())
    {
        TriggerWhiteSpacePing();
    }

    if (m_AccountChangeEvent.TryWait())
    {
        m_AccountChangeEvent.Clear();

        if (IsConnected())
        {
            // フレンドプレゼンス状態更新のため DA 一覧を送る
            auto result = SendDeviceAccountListAsync();
            NN_UNUSED(result);
        }
    }

    Xmpp::RunOnce(timeout);
}

}}
