﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/bgsu/bgsu_HeapTemplate.h>

//-----------------------------------------------------------------------------
// メモリ関連の初期化とオーバーライド
//
struct DefaultHeapTag;
nn::bgsu::StaticHeapTemplate<DefaultHeapTag, 4 * 1024, nn::bgsu::ExpHeapAllocator> s_DefaultHeap("default");

#ifndef NN_BUILD_CONFIG_OS_WIN
extern "C" void* malloc(size_t size)               { return s_DefaultHeap.Allocate(size); }
extern "C" void* realloc(void* p, size_t size)     { return s_DefaultHeap.Reallocate(p, size); }
extern "C" void  free(void* p)                     { s_DefaultHeap.Free(p); }
extern "C" void* calloc(size_t n, size_t size)     { return s_DefaultHeap.AllocateZeroInitializedArray(n, size); }
extern "C" void* aligned_alloc(size_t align, size_t size) { return s_DefaultHeap.AllocateAligned(size, align);  }
extern "C" size_t malloc_usable_size (void *p)
{
    NN_UNUSED(p);
    NN_ABORT("malloc_usable_size is not implemented");
}
#endif

void* operator new(std::size_t size)                                      { return malloc(size); }
void* operator new(std::size_t size, const std::nothrow_t&) NN_NOEXCEPT   { return malloc(size); }
void  operator delete(void* ptr) NN_NOEXCEPT                              { free(ptr); }

void* operator new[](std::size_t size)                                    { return malloc(size); }
void* operator new[](std::size_t size, const std::nothrow_t&) NN_NOEXCEPT { return malloc(size); }
void  operator delete[](void* ptr) NN_NOEXCEPT                            { free(ptr); }

//-----------------------------------------------------------------------------
// nn::lm::Initialize の宣言
//

namespace nn { namespace lm {

void Initialize() NN_NOEXCEPT;

}} // namespace nn::lm

//-----------------------------------------------------------------------------
// 各サービスモジュールの宣言
//
void EntryArp();
void EntryBgtc();

//-----------------------------------------------------------------------------
// nn::init の関数
//

extern "C" void nninitStartup()
{
}
extern "C" void nndiagStartup()
{
#ifndef NN_SDK_BUILD_RELEASE
    nn::lm::Initialize();
#endif
}
extern "C" void nnMain()
{
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(glue, Main));

    // サービスの初期化 -------------------------------------------------------------------------------
    EntryArp();
    EntryBgtc(); // シャットダウンまでブロック
}
